;       PDP8E.ASM
;
;       This program is to allow a PDP8/E to be emulated on the 6809 single
;       board computer.  It is based on the PDP8.ASM program, with the
;       addition of the KM8-E Memory Extension and Time Share module.
;
;
;       Emulator data:
;
;       (1) Emulated program space will be two bytes for each PDP8 memory
;           location.
;       (2) Non existent fields are read as 0.
;       (3) To increase speed the emulator code is in RAM.
;       (4) Interrupt inhibit (removed by a JMP or JMS) is indicated by
;           the first bit in the interrupt flags being set.  Instruction
;           delay (due to ION) is handled by a changing a pointer.  If the
;           pointer indicates that the interrupts are already enabled it is
;           not changed.  It is assumed that ION delay and JMP/JMS inhibit
;           are not counted in checking if the interrupt system is on.  See
;           the JMP code for further explanation.
;
;
;       Memory data format:
;
;               0000CCCIZAAAAAAA
;
;               0= Must be 0 (only 12 bits used).
;               C= Opcode
;               I= Indirect address if 1.
;               Z= Current page if 1.
;               A= Page address.
;
;
;       Memory Extension and Time Share:
;
;               There are three register sets: Saved, Working and Buffer.
;               Each set has both Field numbers and pointers.  The User
;               field is really a flag for User/Executive mode.  If the
;               bit is 0 it is in the Executive (standard) mode, else it
;               is in the User mode in which all LAS, HLT and IOT
;               instructions are bypassed and the Time Share interrupt
;               set.  The registers include the memory pointers and the
;               field number bytes, as follows:
;
;                       U= Time Share flag (0= Executive, 1= User).
;                     III= Instruction field number.
;                     DDD= Data field number.
;               
;                __________________________________________________________
;               | Function  || RAM Registers                               |
;               |           ||_____________________________________________|
;               |           || I field | D field | Field bytes and bit map |
;               |           || pointer | pointer |_________________________|
;               |           ||         |         | U and I    | D          |
;               |           ||         |         | (U0000III) | (00000DDD) |
;               |===========||=========|=========|============|============|
;               |   Working || WrkIptr | WrkDptr |     WorkUI |      WorkD |
;               |-----------||---------|---------|------------|------------|
;               | Interrupt || SavIptr | SavDptr |     SaveUI |      SaveD |
;               |-----------||---------|---------|------------|------------|
;               |   J Delay || BufIptr |         |     BuffUI |            |
;               |___________||_________|_________|____________|____________|
;
;
;       Modifications:
;
;       (1) 4-8-97 F. Wilson: Fixed small errors in the Extended Memory
;       and Time Share module.  Is not tested yet, and will need update
;       of the test compiler.
;       (2) 4-15-97 F. Wilson: Added checks for breakpoints and a
;       monitor.  Single stepping is the starting default.
;       (3) 5-2-97  F. Wilson: Put Instruction Field into GTF.
;       (4) 5-6-97  F. Wilson: Removed single step routine and
;       incorporated into a primitive monitor. Checked emulator
;       speed with the LED display on and it drops by over 40%.
;       (5) 5-12-97 F. Wilson: Removed indirects from field
;       pointer access in field change IOT's.
;       (6) 6-11-97 F. Wilson: Changed vector lists to end in 0.
;       (7) 9-3-97  F. Wilson: Put ^C exit into the binary loader.
;       Made location 200 in Field 0 the default program start.
;
;
;       To do:
;
;       (1) Add "paper tape" load and punch.  Perhaps can have a buffer
;       for main program download to decrease the binary loader delay
;       and then re-use the.  Use up and down loading from the PC,
;       with time delay to indicate end of data.
;       (2) Get panel switch setting from the keyboard. Default is the
;       program start address.
;       (3) Have HLT instruction set a flag and jump directly into
;       the monitor.
;

; Program equates:

Opmask  equ     %00001110       Upper byte opcode mask.
Wmask   equ     %00001111       Upper byte memory data mask.
Wsize   equ     @7777           Memory data mask.
Fsize   equ     @10000*2        4096 words/memory location.
Offset  equ     %01111111       Page offset bits.
LoPage  equ     %10000000       Current page (low byte).
Lmask   equ     %00010000       Isolate link.
ACmask  equ     %00001111       Drop link.
IOTmask equ     %01111110       Isolate IOT address.
LtoC    equ     %11110000       Add to link (moves to carry).
IOsetup equ     8               IO device setup ("impossible" IOT).
IOreset equ     9               Clear/CAF.
CR      equ     $0D             Carriage return.
LF      equ     $0A             Line feed.
CtrlC   equ     $03             Control C.
Space   equ     $20             Space.
BS      equ     $08             Backspace.
XON     equ     $11             Data link "on"  prompt (DC1 or ^Q)
XOFF    equ     $13             Data link "off" prompt (DC3 or ^S).
TRUE    equ     1               For logical comparison.
FALSE   equ     0                "     "        "
IRQoff  equ     %00010000       IRQ off condition code mask.
FIRQoff equ     %01000000       FIRQ off.
Imask   equ     FIRQoff+IRQoff  Turns off maskable interrupts.
Zcode   equ     %00000100       Zero condition.
BRAval  equ     $20             Branch always instruction.
BRNval  equ     $21               "    never       "
BEQval  equ     $27               "    if equal    "
Ubit    equ     %10000000       Field number bit masks.
Fbits   equ     %00000111
Dbits   equ     %00000111
Jdelay  equ     %10000000       JMP/JMS interrupt inhibit.
UserI   equ     %01000000       User Interrupt bit.
SetIO   equ     %0010000000000000       Start bit for interrupt flags.
; Monitor flags:
Monitor equ     %10000000       Call monitor (from keyboard ^S).
Break0  equ     %01000000       Break point #0.
Break1  equ     %00100000       Break point #1.
Break2  equ     %00010000       Break point #2.
Halt    equ     %00001000       Halt instruction.
I_path  equ     %00000100       Interrupt service.
SnglStp equ     %00000010       Single stepping.
; Data display strobes:
Strb0   equ     %00000001
Strb1   equ     %00000010
Strb2   equ     %00000100
Strb3   equ     %00001000

; Page 0 storage:

        org     $0

PC      rmb     2               Program counter.
AC      rmb     2               Accumulator.
MQ      rmb     2               Multiplier quotient.
Link    rmb     1
Instr   rmb     2               Present instruction.
SR      rmb     2               Front panel switch register.
Temp2   rmb     2
Temp1   rmb     1
IOflags rmb     2               Input output device interrupt requests.
Uflag   rmb     2               User interrupt bit.
Flagset rmb     2               Temporary for device flag setup.
Dcount  rmb     1               Temporary for device #.
IOctrl  rmb     1               Temporary device IOT.
Vsetup  rmb     2               Pointer to the ROM vector setup routine.
Idelay  rmb     1               ION delay count.
Ifield  rmb     1               Used by the binary loader.
; Following locations are used by the Memory Extension and Time Share.
; Keeping the User flag in the MSB allows a simple sign check.
; Have WorkD follow WorkUI and SaveD follow SaveUI so can transfer as
; a word.
WrkIptr rmb     2               Working I field (also use X for pointer).
WrkDptr rmb     2
WorkUI  rmb     1
WorkD   rmb     1
SavIptr rmb     2               Saved during interrupt processing.
SavDptr rmb     2
SaveUI  rmb     1
SaveD   rmb     1
BufIptr rmb     2               Loaded into working U and F by JMP or JMS.
BuffUI  rmb     1
; Breakpoint and Monitor:
Monflag rmb     1               Monitor call flag.
CCsave  rmb     1
; Machine state display:
PIAsave rmb     1               PIA side A output data value.

; Emulator

        org     $200

; Start up emulator.
Start:  stx     Vsetup          IO devices may need during setup.
        ldd     #Vlist          Should check the version (in D) first.
        jsr     ,X              Set up ROM access vectors.
        jsr     RAMclr          Clear (existing) memory.
        ldx     #Sysmsg         Send startup message.
        jsr     [>Msgptr]
        lda     #Monitor        Call Monitor on start.
        sta     Monflag
        jsr     PDPload         LOAD BINARY FILE (sets PC and Ifield).
        ldd     #@200           Set default start.
        std     PC
        clr     Ifield
        lda     #IOsetup        Set up IO devices (KL8-E turns on IRQ).
        bsr     _clrsys
        lda     Ifield          Set up the field registers.
        sta     WorkUI          Save.
        sta     WorkD
        sta     BuffUI          Same as Extended Address Load on panel.
        ldx     #Spc8msg        Start with correct printout offset.
        jsr     [>Msgptr]
        lsla                    Get field pointer.
        ldu     #GetFld
        ldx     A,U
        beq     Flderr          No memory allocated.
        stx     WrkIptr
        stx     WrkDptr
        stx     BufIptr
        clra                    Set the Save Field registers to field 0.
        sta     SaveUI          NOTE: Real PDP8 may have random value.
        sta     SaveD
        ldd     0,U             Leave X set to the Instruction field.
        std     SavIptr
        std     SavDptr
        ; Set up display pointers:
        ldu     PIAptr
        lda     [>PIAdata]
        sta     PIAsave
        ; Set up display strobes:
        lda     #Strb0|Strb1|Strb2|Strb3
        jsr     [>PIAset]       Set PIA strobe pins to output.
        lda     #$34            Set side B.
        sta     3,U
        lda     #$FF
        sta     2,U
        lda     #$30
        sta     3,U
; End of side B setup.

        ; NOTE: Switch register usually at start address, as
        ;       standard startup sequence is Address Load,
        ;       Clear and Continue.
        ldd     PC              Set switch register to start address.
        std     SR
        jmp     Emulate         Start.
; Memory not allocated for the specified
; field.
Flderr  jsr     Errpnt
        fdb     NoFLD
; Panel "start" or CAF. Clears the AC, MQ
; and link.  Does not affect the data and
; instruction field settings.  Sets the
; Time Share to executive mode (U= 0).
; Peripheral flags cleared, but interrupt
; requests (usualy) enabled. Interrupt
; system is off and ION and JMP/JMS
; inhibits cleared.
; On exit D= 0.
Clear:  lda     #IOreset        Reset flags. Enable interrupt requests.
_clrsys clr     IOflags
        clr     IOflags+1
        jsr     IOset           Enter with A= IO setup flag.
        bne     _clrerr
        ldd     #Doinstr
        std     Emulate+1
        ldd     #0
        std     AC
        std     MQ
        sta     Link
        rts
_clrerr jsr     Errpnt
        fdb     Clrmsg

; ISZ increments the data at the address
; and skips the next instruction if it
; becomes 0. AC and Link not affected.
; Moved here so can fall through into
; the PC skip.
ISZsub  jsr     GetAdr
        beq     NextI
        lslb
        rola
        leau    D,U
        ldd     ,U
        addd    #1
        anda    #Wmask
        std     ,U
        cmpd    #0
        bne     NextI
;
; Skip next instruction.
PCskip: ldd     PC
_skip   addd    #2
        bra     _savpc
; Next instruction routine.
NextI:  ldd     PC              Get the PC.

; TEMPORARY CHECK ROUTINE:
; Check data at last step to see if have any
; extra bits set.
;NextI:  lda     PC
;        ora     AC
;        ora     MQ
;        ora     Instr
;        bita    #%11110000      All high bits must = 0.
;        bne     Bit_err
;        lda     Link
;        bita    #Lmask^$FF
;        beq     GetI
;Bit_err jsr     Errpnt
;        fdb     Bitmsg
;Bitmsg  fcc     "Bits set in locations that should be 0."
;        fcb     0
;GetI:   ldd     PC              Get the PC.
; END CHECK:

_nxtni  addd    #1              Update to next.
_savpc  anda    #Wmask          In case of overflow.
        std     PC
; Following is a jump to Doinstr, Chkdly or Chkint
; depending on the interrupt status.
Emulate jmp     Doinstr         Jump object can change.
; Process ION delay count. When count is 0 set Emulate to
; go to interrupt check and then fall through into check.
Chkdly: dec     Idelay          ION delay finished?
        bne     Doinstr
        ldd     #Chkint         Change Emulate object.
        std     Emulate+1
; Process interrupts. The interrupts have to
; be enabled and the ION delay (1 istruction)
; and JMP/JMS delay (due to I field load)
; finished:
Chkint: ldd     IOflags         Get device flags.
        ble     Doinstr         Check for request (exit if none or J delay).
        ldd     #Doinstr        Turn off interrupts.
        std     Emulate+1
        ldd     WorkUI          Save U, I and D.
        std     SaveUI
        ldd     WrkDptr
        std     SavDptr
        stx     SavIptr
        ldx     GetFld          Set F and D to field 0 & U to exec.
        stx     WrkIptr
        stx     WrkDptr
        clra
        clrb
        std     WorkUI
        ldd     PC              Save return address at location 0.
        std     0,X
        ldd     #1              Do next instruction.
        std     PC
        lslb
Icheck  bra     _getI           BRN if flagging interrupt service.
        lda     Monflag
        ora     #I_path
        sta     Monflag
        clra
        bra     _getI
; Go to next instruction.
Doinstr ldd     PC              Restore PC data.
        lslb                    Multiply by 2 for relative location.
        rola
_getI   ldd     D,X             ; Get instruction.
        std     Instr
; Next instruction ready for decode.
; Use this point for single step and
; monitoring.  BRA/BRN at Stpchk for
; state check or full speed.
; X = instruction field location.
; D = instruction.
Stpchk  bra     Display         ; BRA/BRN point for state checks and display.
Stpxit  anda    #Opmask
        ldu     #Ops
        jmp     [A,U]

;======================================;
; State checks. Use BRA/BRN for each   ;
; check or bypass. Only the A and X    ;
; registers must be preserved.         ;
;======================================;
; Output CPU state using PIA-B for
; data and PIA-A for control.
Display bra     Monchk0         ; BRA/BRN point.
        ldu     PIAptr          ; Get PIA location.
        lda     WorkUI
        lsla
        lsla
        lsla
        lsla
        ora     PC
        ldb     PC+1
        sta     2,U             ; Send high byte.
        lda     PIAsave         ; Get side A output.
        ora     #Strb0          ; Trigger first latch.
        sta     0,U
        stb     2,U             ; Send low byte.
        anda    #Strb0^$FF      ; Trigger second latch.
        ora     #Strb1
        sta     0,U
        ldd     AC
        sta     2,U             ; Send high byte.
        lda     PIAsave         ; Get side A output.
        ora     #Strb2          ; Trigger third latch.
        sta     0,U
        stb     2,U             ; Send low byte.
        anda    #Strb2^$FF      ; Trigger fourth latch.
        ora     #Strb3
        sta     0,U
        lda     PIAsave
        sta     0,U             ; Restore side A output.
; Fall through:
Monchk0 bra     Monchk1         ; BRA/BRN point.
; Break check.  Sets flag bit Break0.
Monchk1 bra     Monchk2         ; BRA/BRN point.
; Break check.  Sets flag bit Break1.
Monchk2 bra     Chkexit         ; BRA/BRN point.
; Break check.  Sets flag bit Break2.
Chkexit lda     Monflag         ; Monitor call if != 0.
        bne     PDPmon
        lda     Instr           ; Get instruction.
        jmp     Stpxit          ; Process it.

; Enter monitor if:
;       Breakpoint(s) became true,
;       HLT, keyboard break (^S)
;       or single stepping.
;
; NOTES:
; (1)   If HLT: Was a direct jump and
;       special processing is needed.
; (2)   On entry, may want to drop
;       single stepping.
;
; Process any breakpoints and then
; give a menue.  Include all the
; usual stuff, such as:
;       Print memory
;       Single step
;       Set various breaks:
;               Memory location
;               Memory data
;               Instruction cycles
;       Load switch register (for OSR)
;       Load address
;       Load extended address
;       Clear
;       Continue
;
; Start Monitor:
;       Presently only monitor startup (from a keyboard
;       ^S call) and single step flags are serviced.
;       In order to allow TTY output during stepping
;       the interrupts are off in this mode.
PDPmon: 
        lda     Monflag         ; Check for keyboard call.
        bita    #Monitor
        beq     Single          ; Branch if not (default).
        anda    #Monitor^$FF    ; Remove keyboard call.
        ora     #SnglStp        ; Assume want single stepping.
        sta     Monflag
        tfr     CC,A            ; Save condition code.
        sta     CCsave
        orcc    #Imask          ; Turn off interrupts.
        ldx     #SSmsg          ; Print options.
        jsr     [>Msgptr]       ; Fall through to "Single".
; Prints system data and then
; waits for Space to continue,
; K to enter keyboard character,
; D to enable display, ? for an
; option list or CR to exit
; stepping.
Single: 
        jsr     Pntreg          ; Show system state.
_sschk  jsr     [>Serinptr]
        cmpa    #CR
        bne     Chkspc
        lda     Monflag         ; Drop single stepping.
        anda    #SnglStp^$FF
        sta     Monflag
        jsr     [>CRptr]        ; Clear up screen a bit.
        lda     CCsave          ; Restore interrupts.
        tfr     A,CC
        bra     _ssxit
Chkspc  cmpa    #Space
        beq     _ssxit          ; Next step.
        cmpa    #'?
        bne     _sskbd
        ldx     #SSmsg
        jsr     [>Msgptr]
        bra     _sschk
_sskbd  anda    #%01011111      ; Drop lower case bit.
        cmpa    #'K             ; Keyboard input?
        bne     _ssdsp
        ldx     #Kbdmsg
        jsr     [>Msgptr]       ; Ask for keyboard input.
        jsr     [>Serinptr]     ; Get it.
        jsr     [>Seroutptr]    ; Echo.
        jsr     _ttisub         ; Install it.
        jsr     [>CRptr]
        bra     _stprtn         ; Continue.
_ssdsp  cmpa    #'E             ; Enable output of CPU state.
        bne     _ssnd
        ldx     #Enabmsg
        jsr     [>Msgptr]
        lda     #BRNval         ; Remove bypass.
_ssbr   sta     Display
        bra     _stprtn
_ssnd   cmpa    #'D             ; Disable output of CPU state.
        bne     _sschk
        ldx     #Dismsg
        jsr     [>Msgptr]
        lda     #BRAval
        bra     _ssbr
_stprtn ldx     #Spc8msg
        jsr     [>Msgptr]       ; Make up for Instruction print.
_ssxit:

; Exit the monitor:
;
        lda     Monflag         ; Remove remaining breaks.
        anda    #Break0|Break1|Break2|I_path^$FF
        sta     Monflag
        bsr     SetStep         ; Fix up loop status.
        lda     Instr           ; Restore CPU registers.
        ldx     WrkIptr
        jmp     Stpxit          ; Continue emulation.

; Sets correct BRA/BRN instruction
; in Stpchk. If not going to use
; the Monitor, display or check
; any breakpoints it sets it to
; BRN, else to BRA.
SetStep lda     Monflag         ; Monitor called if any bits set.
        bne     SetBRA
        lda     #BRNval
        cmpa    Display
        beq     SetBRA
        cmpa    Monchk0
        beq     SetBRA
        cmpa    Monchk1
        beq     SetBRA
        cmpa    Monchk2
        beq     SetBRA
        cmpa    Icheck
        beq     SetBRA
SetBRN  sta     Stpchk
        rts
SetBRA  lda     #BRAval
        bra     SetBRN

; Illegal instruction trap.
Illegal jsr     Errpnt
        fdb     Insterr

; Clear existing RAM fields.
; NOTE: Temporarily set to
;       fill with HLT (7402).
RAMclr: lda     #@10
        pshs    A
        ldu     #GetFld
_nextF  ldx     ,U++
        beq     _noFld
        ldd     #Fsize
_clrF   pshs    D               ; Temporary
        ldd     #@7402          ;     "
        std     ,X++            ;     "
;_clrF    clr     ,X+
;        addd    #-1
        puls    D               ; Temporary
        addd    #-2             ;     "
        bne     _clrF
_noFld  dec     0,S
        bne     _nextF
        puls    A,PC

; Access all IO devices. Enter
; with the IOP code in
; register A. Only codes
; >= 8 used.
; If return with = status all
; existing devices were able to
; comply.
; On an error the routine
; takes an immediate exit and
; returns the exit code in A.
IOset:  sta     IOctrl
        ldu     #Device
        ldd     #SetIO          Set up initial flag.
        std     Flagset
        lda     #64             Device count.
        sta     Dcount
_ioloop ldd     ,U++            Check for device exists.
        cmpd    #Nodev
        beq     _iolxit
        lda     IOctrl          Get IOT.
        pshs    U               Device changes U.
        jsr     [-2,U]          Do setup.
        puls    U
        bne     _seterr
_iolxit dec     Dcount          Done?
        bne     _ioloop
_seterr rts                     Return == 0 status if no error.

; Get device flag bit in D.
; Used by IO devices during
; initial setup. If return 0
; are out of flags and device
; must return != status.
; NOTE: Flagset must be initialised
;       with the SetIO value.
;       Are 15 flags available.
;       The MSB is reserved for
;       interrupt processing
;       inhibit till JMP/JMS.
;       Next bit is for the Time
;       Share interrupt.
Getflag ldd     Flagset         Get flag.
        pshs    D
        lsra                    Update flag.
        rorb
        std     Flagset
        puls    D
        rts

; AND the AC with the data at the address.
; Link not affected.
; Indirect will use the data field.
; Non existent data field read as 0.
ANDsub: jsr     GetAdr
        beq     _dcaxit         Check for indirect and no field.
_doand  lslb
        rola
        ldd     D,U
        anda    AC
        andb    AC+1
        std     AC
        jmp     NextI

; TAD similar to AND, except that the
; Link is complemented by any overflow.
TADsub: jsr     GetAdr
        lbeq    NextI
        lslb
        rola
        ldd     D,U
        ora     Link
        addd    AC
        pshs    A
        anda    #ACmask
        std     AC
        puls    A
        anda    #Lmask
        sta     Link
        jmp     NextI

; DCA similar to AND. AC stored at the
; address and then cleared. Link is not
; affected.
DCAsub  jsr     GetAdr
        beq     _dcaxit
        lslb
        rola
        leau    D,U
        ldd     AC
        std     ,U
_dcaxit clra
        clrb
        std     AC
        jmp     NextI

; Go to an address.
; Checks for the interrupt inhibit, and if
; set removes it.  The I and U fields must
; be current with the Buffer after this
; instruction.  An instruction field
; change always implies in interrupt
; inhibit.  The Time Share instruction
; CUF may not set the inhibit, as it
; stops the interrupt.
JMPsub  lda     IOflags         Check for interrupt inhibit.  
        bpl     _dojmp
        anda    #Jdelay^$FF     Remove inhibit.
        sta     IOflags
        ldx     BufIptr         Update working field pointers.
        stx     WrkIptr
_dojmp  lda     BuffUI          Update U (also does F).
        sta     WorkUI
        jsr     GetAdr
        jmp     _savpc

; Jump to a subroutine. Stores the return address at the
; subroutine entry point and continues at the next
; instruction.  Interrupt inhibit and U, F and D handling
; as in the above JMP instruction.
JMSsub  lda     IOflags
        bpl     _dojms
        anda    #Jdelay^$FF
        sta     IOflags
        ldx     BufIptr
        stx     WrkIptr
_dojms  lda     BuffUI
        sta     WorkUI
        jsr     GetAdr
        std     Temp2
        lslb
        rola
        leau    D,X
        ldd     PC
        addd    #1
        anda    #Wmask
        std     ,U
        ldd     Temp2
        jmp     _nxtni

; Will use the middle 6 bits of the instruction
; as the device address and the last 3 bits to
; indicate the action to be taken by the device.
; Access the device handler with the IOT code
; in A. Codes above 7 are "impossible" and used
; for special access. In this case the handler
; will exit with a status code in A.
IOTsub  tst     WorkUI          User Mode?
        lbmi    U_Mode
        ldu     #Device
        ldd     Instr
        lsra
        rorb
        rorb
        andb    #IOTmask
        lda     Instr+1
        anda    #%111           IOT code in A.
        jmp     [B,U]

; OPR routine. One of the most complex
; due to the microcoded execution.
OPRsub: ldd     Instr
        rora                    Test for group 1.
        bcs     _chkpg2
        bitb    #%10000000
        beq     _cll
        clra                    CLA
        sta     AC
        sta     AC+1
_cll    bitb    #%01000000
        beq     _cma
        clr     Link            CLL
_cma    bitb    #%00100000
        beq     _cml
        ldd     AC              CMA
        coma
        anda    #ACmask
        comb
        std     AC
_cml    ldb     Instr+1
        bitb    #%00010000
        beq     _iac
        lda     Link            CML
        coma
        anda    #Lmask
        sta     Link
_iac    bitb    #%00000001
        beq     _rotate
        ldd     AC              IAC
        addd    #1
        anda    #ACmask
        std     AC
_rotate ldb     Instr+1
        andb    #%00001110      Isolate rotate and swap bits.
        ldu     #Group1
        jmp     [B,U]

_chkpg2 bitb    #%00000001
        bne     _chkpg3
        bitb    #%01000000
        beq     _skp1
        lda     AC              SMA
        bita    #%00001000
        bne     SkpTest
_skp1   bitb    #%00100000
        beq     _skp2
        ldd     AC              SZA
        beq     SkpTest
_skp2   ldb     Instr+1
        bitb    #%00010000
        beq     _skp3
        lda     Link            SNL
        bne     SkpTest
_skp3   bitb    #%00001000
        beq    _skp4
Skip    ldd     PC
        addd    #1
        anda    #ACmask
        std     PC
        bra     _skp4
SkpTest ldb     Instr+1
        bitb    #%00001000
        beq     Skip
_skp4   ldb     Instr+1
        bpl     _skp5
        clra                    CLA
        clrb
        std     AC
; Test for OSR and HLT. Seldom used
; so will start with a fast check:
_skp5   ldb     Instr+1         Fast exit check.
        bitb    #%00000110      OSR|HLT?
        lbeq    NextI
        tst     WorkUI          Check for OSR/HLT block.
        lbmi    U_Mode
_skp6   bitb    #%00000100
        beq     _skp7
        ldd     AC              OSR
        ora     SR
        orb     SR+1
        std     AC
        ldb     Instr+1
_skp7   bitb    #%00000010
        lbeq    NextI
        ; NOTE: In the future will set the halt
        ;       bit and jump to the monitor.
        jmp     Hlterr

_chkpg3 bitb    #%10000000
        beq     _chkmqa
        clra
        sta     AC              CLA
        sta     AC+1
_chkmqa andb    #%01010000      Isolate MQA and MQL bits.
        lbeq    NextI
        cmpb    #%01000000
        bne     _chkmql
        ldd     AC              MQA
        ora     MQ
        orb     MQ+1
        std     AC
        jmp     NextI
_chkmql cmpb    #%00010000
        bne     _swp
        ldd     AC              MQL
        std     MQ
        clra
        clrb
        std     AC
        jmp     NextI
_swp    ldd     AC              SWP
        std     Temp2
        ldd     MQ
        std     AC
        ldd     Temp2
        std     MQ
        jmp     NextI

; PROGRAM HALTED
Hlterr: jsr     Errpnt          HLT
        fdb     Hltmsg

BSW:    ldd     AC
        rolb                    Upper and lower bytes in A and B.
        rola
        rolb
        rola
        exg     A,B             Swap bytes.
        lsra                    Align A.
        lsra
        rolb                    Align B.
        rolb
        lsra                    Align AC.
        rorb
        lsra
        rorb
        std     AC
        jmp     NextI

ROL     lda     Link
        adda    #LtoC           Move Link to carry.
        ldd     AC              Does not change carry.
_left   rolb                    Rotate AC and Link (carry) left.
        rola
        sta     Link
        anda    #ACmask         Save AC.
        std     AC
        lda     Link            Save Link.
        anda    #Lmask
        sta     Link
        jmp     NextI

RTL     lda     Link
        adda    #LtoC
        ldd     AC
        rolb
        rola
        adda    #LtoC           Just affects the Link bit.
        bra     _left

ROR     ldd     AC
        ora     Link
_right  lsra
        rorb
        std     AC
        rora                    Carry to Link "the easy way".
        lsra
        lsra
        lsra
        sta     Link
        jmp     NextI

RTR     ldd     AC
        ora     Link
        lsra
        rorb
        bcc     _right
        ora     #Lmask
        bra     _right

ROR_ROL:
        ldd     AC              Start with ROR.
        ora     Link
        lsra
        rorb
        std     Temp2           First shift to MD.
        lda     Link            Now ROL.
        adda    #LtoC           Move Link to carry.
        ldd     AC              Does not change carry.
        rolb                    Rotate AC and Link (carry) left.
        rola
        sta     Link
        anda    #ACmask         MD and AC to AC.
        anda    Temp2
        andb    Temp2+1
        std     AC
        lda     Link            Save Link.
        anda    #Lmask
        sta     Link
        jmp     NextI

ROL_ROR:
        lda     Link            Start with ROL.
        adda    #LtoC           Move Link to carry.
        ldd     AC              Does not change carry.
        rolb                    Rotate AC and Link (carry) left.
        rola
        anda    #ACmask         Save AC.
        std     Temp2
        ldd     AC              Now do ROR.
        ora     Link
        lsra
        rorb
        anda    Temp2           MD and AC to AC.
        andb    Temp2+1
        std     AC
        rora                    Carry to Link "the easy way".
        lsra
        lsra
        lsra
        sta     Link
        jmp     NextI

; Default "no device" IOT.
; Evidently if the device
; does not exist there is
; no action.
Nodev   equ     NextI
; NOTE: The following is the
;       previous code:
;Nodev   jsr     Errpnt
;        fdb     NoIOT
;NoIOT   fcc     "Tried to access a non-existent IO device"
;        fcb     $00

; Get the address referenced by the instruction.
; If AND, TAD, ISZ or DCA an indirect will be from
; the data field.  Set U for use by the above
; instructions.
; If the I and Z bits indicate indirect page 0 and
; the location is @10 through @17 the location is
; a pre increment autoindex access.
GetAdr: ldd     Instr
        tstb                    Test for page 0.
        bmi     CurPg
        lsra                    Test for indirect.
        bcs     Pg0Ind
        clra                    Set PC for page 0.
        ldu     WrkIptr
        rts
Pg0Ind: clra
        cmpb    #@10            Check for autoindex.
        blo     Get0I
        cmpb    #@17
        bhi     Get0I
        lslb                    Pre increment.
        leau    D,X
        ldd     ,U
        addd    #1
        anda    #Wmask
        std     ,U
        ldu     WrkDptr
        rts
Get0I:  lslb
        ldd     D,X
        ldu     WrkDptr
        rts
CurPg   andb    #Offset
        stb     Temp1           Save current page offset.
        lsra                    Check for indirect.
        bcs     CurInd
        ldd     PC              Get current page.
        andb    #LoPage
        orb     Temp1           Insert offset.
        ldu     WrkIptr
        rts
CurInd: ldd     PC
        andb    #LoPage
        orb     Temp1
        lslb
        rola
        ldd     D,X             Get indirect address.
        ldu     WrkDptr
        rts

; Field location array. If location= 0
; the field does not exist. Field 0
; must always be there.
GetFld:
        fdb     Field0          ; Must exist.
        fdb     Field1          ; Minimum for OS8.
        fdb     Field2
        fdb     Field3
        fdb     0               Field 4 (non existent).
        fdb     0               Field 5 (non existent).
        fdb     0               Field 6 (non existent).
        fdb     0               Field 7 (non existent).

; Opcode routine address array. Use opcode*2 as an index.
Ops:    fdb     ANDsub
        fdb     TADsub
        fdb     ISZsub
        fdb     DCAsub
        fdb     JMSsub
        fdb     JMPsub
        fdb     IOTsub
        fdb     OPRsub

; Rotate and swap routines for group 1 OPR.
; NOTE: The "undefined" instructions may have
;       rotated the AC and Link into the MD
;       register and then in the opposite
;       direction into itself.  Next the AC
;       and MD registers may have been anded
;       and put into the AC.  State of the
;       link would depend on which direction
;       was done first.
Group1: fdb     NextI           No operation.
        fdb     BSW             Byte swap AC.
        fdb     ROL             Rotate AC and Link left.
        fdb     RTL
        fdb     ROR             Rotate AC and Link right.
        fdb     RTR
        fdb     ROL_ROR         ROR-ROL (undefined on E).
        fdb     Illegal         RTR-RTL (undefined on E.

; Device routine address array. 64 Possible devices.
; NOTE: The KM8-E is done but not tested.
Device: fdb     IOT00           Device 00 Interrupts (basic system).
        fdb     Nodev           Device 01.
        fdb     Nodev           Device 02.
        fdb     IOT03           Device 03 KL8-E Teletype keyboard/reader.
        fdb     IOT04           Device 04.  "      "     teleprinter/punch.
        fdb     Nodev           Device 05.
        fdb     Nodev           Device 06.
        fdb     Nodev           Device 07.
        fdb     Nodev           Device 10 KP8-E Power Fail Detect.
        fdb     Nodev           Device 11.
        fdb     Nodev           Device 12.
        fdb     Nodev           Device 13 DK8-EA/EC/EP Real Time Clocks.
        fdb     Nodev           Device 14.
        fdb     Nodev           Device 15.
        fdb     Nodev           Device 16.
        fdb     Nodev           Device 17.
        fdb     IOT20           Device 20 KM8-E Memory ext. & Time Share.
        fdb     IOT21           Device 21   "      "    "   "   "    "
        fdb     IOT22           Device 22   "      "    "   "   "    "
        fdb     IOT23           Device 23   "      "    "   "   "    "
        fdb     IOT24           Device 24   "      "    "   "   "    "
        fdb     IOT25           Device 25   "      "    "   "   "    "
        fdb     IOT26           Device 26   "      "    "   "   "    "
        fdb     IOT27           Device 27   "      "    "   "   "    "
        fdb     Nodev           Device 30.
        fdb     Nodev           Device 31.
        fdb     Nodev           Device 32.
        fdb     Nodev           Device 33.
        fdb     Nodev           Device 34.
        fdb     Nodev           Device 35.
        fdb     Nodev           Device 36.
        fdb     Nodev           Device 37.
        fdb     Nodev           Device 40.
        fdb     Nodev           Device 41.
        fdb     Nodev           Device 42.
        fdb     Nodev           Device 43.
        fdb     Nodev           Device 44.
        fdb     Nodev           Device 45.
        fdb     Nodev           Device 46.
        fdb     Nodev           Device 47.
        fdb     Nodev           Device 50.
        fdb     Nodev           Device 51.
        fdb     Nodev           Device 52.
        fdb     Nodev           Device 53.
        fdb     Nodev           Device 54.
        fdb     Nodev           Device 55.
        fdb     Nodev           Device 56.
        fdb     Nodev           Device 57.
        fdb     Nodev           Device 60.
        fdb     Nodev           Device 61.
        fdb     Nodev           Device 62.
        fdb     Nodev           Device 63.
        fdb     Nodev           Device 64.
        fdb     Nodev           Device 65.
        fdb     Nodev           Device 66.
        fdb     Nodev           Device 67.
        fdb     Nodev           Device 70.
        fdb     Nodev           Device 71.
        fdb     Nodev           Device 72.
        fdb     Nodev           Device 73.
        fdb     Nodev           Device 74.
        fdb     Nodev           Device 75.
        fdb     Nodev           Device 76.
        fdb     Nodev           Device 77.

; Interrupt IOT module.  Comes with the basic system.
;
; Standard IOPs' are 0 through 7.  Higher ones are
; "impossible", and used by the emulator for system
; initiation (8) and reset (9).
IOT00:  ldu     #IOP00          Use an opcode array.
        lsla                    Get the IOP code.
        jmp     [A,U]

IOP00   fdb     SKON            ; 0
        fdb     ION             ; 1
        fdb     IOF             ; 2
        fdb     SRQ             ; 3
        fdb     GTF             ; 4
        fdb     RTF             ; 5
        fdb     IOT00_6         ; 6
        fdb     CAF             ; 7
        fdb     No_op           ; 8 Initialise.
        fdb     No_op           ; 9 Reset.

; Skip if interrupt on and turn off.
; Assume "on" not influenced by 
; ION and JMP/JMS inhibits.
SKON:   ldd     #Doinstr
        cmpd    Emulate+1
        lbeq    NextI
        std     Emulate+1
        jmp     PCskip

; Turn interrupt on. Actual response delayed
; till after the next instruction (unless
; are already on).
; NOTE: This is independent of JMP/JMS inhibit.
ION:    ldd     Emulate+1
        cmpd    #Doinstr
        lbne    NextI
        lda     #2
        sta     Idelay
        ldd     #Chkdly
_setjmp std     Emulate+1
        jmp     NextI

; Turn interrupts off (unconditional).
; NOTE: Does not change JMP/JMS inhibit.
IOF:    ldd     #Doinstr
        bra     _setjmp

; Skip if interrupt request.
SRQ:    lda     IOflags
        anda    #Jdelay^$FF     Drop JMP/JMS inhibit.
        ora     IOflags+1
        lbne    PCskip
        jmp     NextI

; Get flags.
GTF:    ldb     WorkUI          Get I field and user state.
        bpl     _getInt         Test for User mode.
        orb     #%00001000      Set U bit.
_getInt anda    #%00001111      Keep U and F bits.
        ldu     Emulate+1       Check interrupt status.
        cmpu    #Doinstr
        beq     _getD
        orb     #%00010000      Set Ion bit.
_getD   lslb                    Move.
        lslb
        lslb
        orb     WorkD           Add in D field.
        lda     IOflags         Check for interrupt req.
        anda    #Jdelay^$FF     Drop JMP/JMS delay.
        ora     IOflags+1
        bne     _setI
        clra
        bra     _getBlk
_setI   lda     #%00000010      Put in IRQ.
_getBlk tst     IOflags         Check for interrupt block.
        bpl     _getLnk
        ora     #%00000001      Put in JMP/JMS delay.
_getLnk tst     Link
        beq     _savFlg
        ora     #%10000000      Put in Link.
_savFlg std     AC
        jmp     NextI

; Restore flags.
RTF:    ldd     AC              Restore Link.
        rola
        anda    #Lmask
        sta     Link
        andb    #Dbits          Restore D.
        stb     WorkD
        lslb
        ldu     #GetFld
        ldd     B,U
        std     WrkDptr
        lda     AC+1            Restore U and I (to Buffer).
        lsra
        lsra
        lsra
        anda    #%00001111
        adda    #%01111000      Shift U to MSB.
        anda    #Ubit|Fbits
        sta     BuffUI
        lsla                    Drops U.
        ldd     A,U
        std     BufIptr
        ldd     #Chkint         Interrupts on. The JMP/JMS delay
        std     Emulate+1        takes care of the 1 cycle delay.
_setJ   lda     #Jdelay         Set JMP/JMS delay.
        ora     IOflags
        sta     IOflags
        jmp     NextI

; Clear all Flags.
CAF:    jsr     Clear
        jmp     NextI

IOT00_6 jsr     Errpnt
        fdb     NoIerr

; Use for bypassing IOT device setup
; or initialisation.
No_op:  clra                    No error.
        rts                     Do not do setup or initialise.

; KM8-E Memory Extension and Time-Share Option.
;
; Memory extension allows up to 8 fields (4096 words/field).
; Time share has Executive and User modes, controlled by the
; U bit.  If it is 1 the computer is in time share mode and
; all OSR, HLT and IOT instructions are ignored and a time
; share interrupt is generated.  The interrupt sets the
; Instruction and Data fields and the time share bit to 0.
IOT20:  ldb     #0*2            ; Field number times 2.
        bra     _km8iot
IOT21:  ldb     #1*2
        bra     _km8iot
IOT22:  ldb     #2*2
        bra     _km8iot
IOT23:  ldb     #3*2
        bra     _km8iot
IOT24:  ldb     #4*2
        bra     _km8iot
IOT25:  ldb     #5*2
        bra     _km8iot
IOT26:  ldb     #6*2
        bra     _km8iot
IOT27:  ldb     #7*2
_km8iot lsla                    ; IOT times 2.
        ldu     #mm8_op0
        jmp     [A,U]

mm8_op0 fdb     IOT2N           ; 0
        fdb     CDF             ; 1
        fdb     CIF             ; 2
        fdb     CDF_CIF         ; 3
        fdb     mm8_N4          ; 4
        fdb     IOT2N           ; 5
        fdb     IOT2N           ; 6
        fdb     IOT2N           ; 7
        fdb     No_op           ; 8
        fdb     No_op           ; 9

mm8_N4  ldu     #mm8_op1
        jmp     [B,U]

mm8_op1 fdb     CINT
        fdb     RDF
        fdb     RIF
        fdb     RIB
        fdb     RMF
        fdb     SINT
        fdb     CUF
        fdb     SUF

CDF:    ldu     #GetFld
        ldu     B,U
        stu     WrkDptr
        lsrb
        stb     WorkD
        jmp     NextI

CIF:    ldu     #GetFld         ; Must have I field.
        ldu     B,U
        lbeq    Flderr
        lsrb
_cif0   stu     BufIptr
        tst     BuffUI          ; Keep Time Share Mode.
        beq     _cif1
        orb     #Ubit
_cif1   stb     BuffUI
        jmp     _setJ

CDF_CIF ldu     #GetFld
        ldu     B,U
        lbeq    Flderr          ; Must have I field.
        stu     WrkDptr
        lsrb
        stb     WorkD
        bra     _cif0

RDF:    ldb     WorkD
_rdf0   lsla
        lsla
        lsla
_rdf1   orb     AC+1
        stb     AC+1
        jmp     NextI

RIF:    ldb     WorkUI
        bra     _rdf0

RIB:    ldb     SaveUI
        lslb
        lslb
        lslb
        orb     SaveD
        tst     SaveUI
        beq     _rdf1
        orb     #%01000000      ; Set U mode.
        bra     _rdf1

RMF:    ldd     SaveUI
        sta     BuffUI
        stb     WorkD
        ldd     SavDptr
        std     WrkDptr
        ldd     SavIptr
        std     BufIptr
        jmp     _setJ

CINT:   lda     IOflags
        anda    #UserI^$FF
        sta     IOflags
        jmp     NextI

SINT:   lda     IOflags
        anda    #UserI
        lbne    PCskip
        jmp     NextI

; Set Time Share mode to Executive.
; NOTE: Not clear if this sets the
;       JMP/JMS delay, as it stops
;       Time Share interrupts.
CUF:    lda     BuffUI
        anda    #Ubit^$FF
        sta     BuffUI
        jmp     NextI

SUF:    lda     BuffUI
        ora     #Ubit
_suf0   sta     BuffUI
        jmp     _setJ

; Time share interrupt. User
; mode and LAS, HLT or IOT
; instruction.  Instruction
; treated as a no-op and
; interrupt generated.
U_Mode:
        lda     IOflags         ; User interrupt in first byte.
        ora     #UserI
        sta     IOflags
        jmp     NextI

; KM8-E decode error:

IOT2N:
        jsr     Errpnt
        fdb     KM8Eerr


; KL8-E Console Teleprinter Control Module.
;
; Includes both IOT 03 and IOT 04. Interrupt request enable is
; common to both, and set on at reset. Also controlled by AC11
; on a KIE instruction. If the bit is 0, interrupt requests are
; disabled.
IOT03:  ldu     #IOP03          Use an opcode array.
        bra     _iotdec

IOT04:  ldu     #IOP04
_iotdec lsla                    Get the IOP code.
        jmp     [A,U]

IOP03   fdb     KCF             0
        fdb     KSF             1
        fdb     KCC             2
        fdb     IOT03_3         3
        fdb     KRS             4
        fdb     KIE             5
        fdb     KRB             6
        fdb     IOT03_7         7
        fdb     No_op           8 Initialise.
        fdb     No_op           9 Reset.

IOP04   fdb     TFL             0
        fdb     TSF             1
        fdb     TCF             2
        fdb     IOT04_3         3
        fdb     TPC             4
        fdb     TSK             5
        fdb     TLS             6
        fdb     IOT04_7         7
        fdb     TTinit          8 Initialise.
        fdb     TTrst           9 Reset.

; Variables used by the module:
;
;       Keep following two together:
Imod    rmb     1               Serial input corrected initial delay.
Itime   rmb     1                  "     "   original     "      "
;
Ibit    rmb     2               Interrupt flag bit.
Icom    rmb     2               Complement of flag bit.
Istat   rmb     1               Interrupt enable (FALSE == not enabled).
;       Keep following two together:
TTIflag rmb     1               Character received flag (TRUE == received).
TTOflag rmb     1               Output character done flag (TRUE == done).
;
TTIdata rmb     1               Input character storage.

; System ROM vector setup table:
IOT00vct:
Sertime fdb     Si_Time_wO      Serial input Itime and Btime.
SetIRQ  fdb     IRQ_wO          IRQ vector location.
RS_ion  fdb     RS_on_sO        Turn on RS232 input interrupt.
RS_ioff fdb     RS_off_sO       Turn off RS232 input interrupt.
        fdb     0

; Initialise the TTY handler.
TTinit: ldd     #IOT00vct       Set up the ROM vectors.
        jsr     [>Vsetup]
        lda     [>Sertime]      Get serial input Itime values.
        sta     Itime
        suba    #16             Approximately 80~ correction.
        sta     Imod
        jsr     Getflag         Get the interrupt bit.
        std     Ibit
        coma                    Interrupt bit complement.
        comb
        std     Icom
        ldd     #TTIchar        Set up the input interrupt.
        std     [>SetIRQ]
        jsr     [>RS_ion]       Turn the interrupt on.
TTrst:  clra                    Clear the device flags.
        clrb
        std     TTIflag
        inca                    Set up the interrupt status.
        sta     Istat
        ldd     IOflags
        anda    Icom
        andb    Icom+1
        std     IOflags
        andcc   #IRQoff^$FF     Turn IRQ on.
        orcc    #Zcode          Successful setup.
        rts                     And return.

; Character input handler. Driven
; by IRQ, and depends on a baud
; rate of 9600 or less, which takes
; over 200 CPU cycles/baud. As the
; longest instruction is less than
; 20 cycles this is less than 5%
; jitter. Special check for ^S to
; call the monitor.
TTIchar:
        ldd     Imod            Get both Itime values.
        sta     [>Sertime]      Correct for interrupt overhead.
        jsr     [>Serinptr]     Get serial input data.
        stb     [>Sertime]      Restore serial input timing.
        cmpa    #XOFF           Turn on single step (^S)?
        beq     _ttstep
        bsr     _ttisub         Put character in system.
        rti

_ttstep lda     Monflag         Call monitor.
        ora     #Monitor
        sta     Monflag
        jsr     SetStep
        rti

_ttisub ora     #$80            Add in for TTY compatibility.
        sta     TTIdata
        lda     #1              Set keyboard\reader flag.
        sta     TTIflag
        tst     Istat           Update interrupt request.
        beq     _charxt
        ldd     IOflags         Interrupt was enabled.
        ora     Ibit
        orb     Ibit+1
        std     IOflags         Request interrupt.
_charxt rts

; Update the interrupt request.
; If interrupt enabled (and a flag
; set) request an interrupt.
; If interrupt disabled clear the
; request.
; 
Isetup: tst     Istat           Interrupt enabled?
        beq     Iclr
        ldd     TTIflag         Any requests?
        beq     Iclr
Iset:   ldd     IOflags         Request interrupt.
        ora     Ibit
        orb     Ibit+1
        bra     _isetxt
Iclr:   ldd     IOflags         Clear interrupt request.
        anda    Icom
        andb    Icom+1
_isetxt std     IOflags
        jmp     NextI

; TTI will use the IRQ interrupt to get a character from the keyboard.
; Since the worst case instruction takes less that 20 cycles to finish
; and 9600 baud has > 200 cycles/baud the uncertainty is < 5%. The IRQ
; interrupt will be enabled to trap character input to a handler that
; will set the Initial baud time to correct for servicing delay.
; On exit the timing will be reset.
; TTO will not turn the interrupt off while printing characters. This
; could chop an output character up, making it look like a link exit.
; In the future the NMI could be used to output a character, followed
; by a returned character. The characters could either be NULL or data.
; Using this at 57.6 Kbaud and 1 Khz (or less) NMI enables full dialog
; with low overhead.

; TTI section (Teletype keyboard or paper tape reader).

; Clear keyboard flag.
; Does not enable the paper tape
; reader to get another character.
KCF:    bra     _kccxit

; Skip on keyboard flag.
KSF:    tst     TTIflag
_ksfskp lbne    PCskip
        jmp     NextI

; Clear the keyboard flag.
; Enables the paper tape reader to
; get another character.
; NOTE: DEC modifies the standard
;       ASR 33 Teletype to input
;       a single character from the
;       reader on each request.
;       It is assumed that a keyboard
;       character always sets the
;       flag.
KCC:    clra
        clrb
_kccset std     AC
;       jsr     RDRrun          NOTE: Routine does not exist yet.
_kccxit clr     TTIflag
        jmp     Isetup

; Read keyboard buffer static.
KRS:    ldd     AC
        orb     TTIdata
        std     AC
        jmp     NextI

; Set/clear interrupt enable.
KIE:    ldd     AC
        andb    #1
        stb     Istat
        bra     Isetup

; Read keyboard buffer dynamic.
KRB:    clra
        ldb     TTIdata
        bra     _kccset

; TTI error message(s).
IOT03_3:
IOT03_7:
        jsr     Errpnt
        fdb     TTI_3_7

; TTO section (Teletype printer or paper tape punch).

; Set the teleprinter flag.
TFL:    lda     #1
        sta     TTOflag
        jmp     Isetup

; Skip on teleprinter flag.
TSF:    tst     TTOflag
        bra     _ksfskp

; Clear the teleprinter flag.
TCF:    clr     TTOflag
        jmp     Isetup

; Load teleprinter and print.
; Does not clear the
; Teleprinter flag.
TPC:    lda     AC+1
        anda    #$7F            ; Drop high bit.
        jsr     [>Conptr]
        lda     #1
        sta     TTOflag
        jmp     Isetup

; Skip on printer of keyboard flag
; if the interrupt is enabled.
TSK:    lda     TTIflag
        ora     TTOflag
        anda    Istat
        lbeq    NextI
        jmp     PCskip

; Load teleprinter sequence.
; Same as TPC, except that the
; Teleprinter flag is cleared
; first. Since the character
; output is not interrupt
; driven the effect is the
; same as TPC.
TLS:    bra     TPC             No difference.

; TTO error message(s).
IOT04_3:
IOT04_7:
        jsr     Errpnt
        fdb     TTO_3_4

; Binary program loader. Field and
; origin will be set up to field 0
; and PC= 200. Data must start and
; end with leader trailer code.
; Ending code can be followed by
; nulls. First character after the
; ending code is passed back to the
; caller.
; An error is flagged by <mumble>.
; Program PC and Field will have the
; exit values from the loader, which
; serves as a default starting address.
;
; Entry to loader. Sets up starting
; defaults and bypasses L/T code.
;
PDPbin  ldx     GetFld          ; Default field is 0.
        clra
        clrb
        sta     Ifield
        std     _binsum         ; Clear checksum.
        ldd     #@200           ; Default origin.
        std     PC
        lda     #CtrlC
        jsr     [>Fastinptr]
        bra     _bin2
_bin0   jsr     [>Serinptr]     ; Bypass L/T code.
_bin2   cmpa    #@200
        beq     _bin0
        bra     _bin1           ; Go to loop decode.
; Binary loader. Exits on L/T bit.
_binchk jsr     [>Serinptr]
        cmpa    #@200           ; Exit (L/T code)?
        bne     _bin1
        ldd     _binsum         ; Undo last add.
        subd    _binsv0
        subd    _binsv1
        subd    [_binptr]       ; Update checksum.
        anda    #%00001111
        std     _binsum         ; NOTE: Should be 0!
        ldd     _bindat         ; Restore last data.
        std     [_binptr]
        ldd     PC              ; Restore PC.
        subd    #1
        anda    #Wmask
        std     PC
        bra     _binxit
_bin1   sta     _bin3           ; Will need a copy.
        ldd     _binsv1         ; Move saved data.
        std     _binsv0
        ldb     _bin3
        std     _binsv1         ; Save present data.
        addd    _binsum         ; Update checksum.
        std     _binsum
        lda     _bin3
        bita    #@300           ; Origin or field setting?
        bne     _binloc
        bsr     _binwrd
        pshs    D
        ldd     PC
        lslb                    ; Make into word offset.
        rola
        leau    D,X
        stu     _binptr         ; Save previous data.
        ldd     ,U
        std     _bindat
        puls    D               ; Load new data.
        std     ,U
        ldd     PC              ; Update "PC".
        addd    #1
        anda    #%00001111      ; Mask out overflow.
        std     PC
        bra     _binchk
_binptr rmb     2               ; Absolute load location.
_bindat rmb     2               ; Previous memory data.
_binsum rmb     2               ; Checksum.
_binsv0 fcb     0
        rmb     1               ; Last input data.
_binsv1 fcb     0
        rmb     1               ; Present input data.
; Origin or field setting.
_binloc anda    #%11000000      ; Test for field.
        cmpa    #@300
        beq     _fset
        bsr     _binwrd
        std     PC
        bra     _binchk
_fset   lda     _bin3           ; Field format.
        anda    #%00111111
        cmpa    #@7             ; Maximum field number.
        bhi     _binerr
        sta     Ifield          ; Set field number.
        ldu     #GetFld         ; Make field pointer.
        lsla
        ldx     A,U
        beq     _binerr         ; Field exists?
        lbra    _binchk

; Exit from loader. Bypass exit code
; exit on first non exit. Exit code
; is L/T, possibly followed by nulls.
;
_binxit jsr     [>Serinptr]
        cmpa    #@200
        beq     _binxit
        bra     _binx2
_binx1  jsr     [>Serinptr]
_binx2  tsta
        beq     _binx1
        rts

; Enter with high 6 bits of 12 bit word
; stored in _bin3.  Will get next data
; byte and use these to make a 12 bit
; word in D.
;
_binwrd jsr     [>Serinptr]     ; Origin. Get low 6 bits.
        bita    #%11000000      ; Check for error.
        bne     _binerr
        sta     _bin4
        ldd     _binsv1
        std     _binsv0
        ldb     _bin4
        std     _binsv1
        addd    _binsum
        std     _binsum
        lda     _bin3
        anda    #%00111111      ; Drop high bits.
        clrb
        lsra
        rorb
        lsra
        rorb
        orb     _bin4
        rts
_bin3   rmb     1
_bin4   rmb     1

; Error on binary load.
;
_binerr jsr     Errpnt
        fdb     Lderr

; Print the PDP8 registers.

Pntreg: lda     Link            ; Link bit in wrong place.
        lsra                    ; Move it.
        lsra
        lsra
        lsra
        sta     _stp2
        lda     WorkUI          ; U in MSB.
        lsla                    ; U to carry and 0 to LSB.
        rola                    ; U to LSB and next 4 bits= 0.
        sta     _stp3
        ldu     #_stp4          ; Flag set routine.
        ldd     Emulate+1       ; Check for ION.
        cmpd    #Chkint
        bsr     ClrFlag
        lda     IOflags         ; Check for NoI.
        anda    #Jdelay
        bsr     SetFlag
        lda     IOflags         ; Check for Ibuss.
        anda    #Jdelay^$FF
        ora     IOflags+1
        bsr     SetFlag
        ldx     #Stpmsg
_stp0   jsr     [>Msgptr]
        ldu     ,X++
        lda     ,X+
        jsr     Pntoctl
        lda     ,X
        bne     _stp1
        rts
_stp1   cmpa    #CR             ; CR/LF flag.
        bne     _stp0
        jsr     [>CRptr]
        leax    1,X
        bra     _stp0

_stp2   rmb     1               ; Formatted copy of Link.
_stp3   rmb     1               ; Formatted User mode.
_stp4   rmb     1               ; True if interrupt system is on.
_stp5   rmb     1               ; True if interrupt requests blocked.
_stp6   rmb     1               ; True if have interrupt requests.

; Set location pointed to by U to 1 if enter with =,
; 0 otherwise. Exit with U incremented.
ClrFlag:
        beq     _setf
        bra     _clrf
; Set location pointed to by U to 1 if enter with !=,
; 0 otherwise. Exit with U incremented.
SetFlag:
        beq     _clrf
_setf   lda     #1
_setfx  sta     ,U+
        rts
_clrf   clra
        bra     _setfx

; Get octal number from the terminal.
; A= number of characters (1 to 4).
; U= storage location (2 bytes).
; Exits after getting the required
; digits or a CR. If
; had CR exits not =.
; A and B changed.
Getoctl pshs    A
        clra
        clrb
        std     ,U              ; Clear data.
_octg0  jsr     [>Serinptr]     ; Get digit or exit.
        cmpa    #CR
        beq     _octxit
        cmpa    #'0
        blo     _octg0
        cmpa    #'7
        bhi     _octg0
        jsr     [>Seroutptr]
        suba    #'0
        pshs    A
        ldd     ,U              ; Shift data.
        lslb                    
        rola
        lslb
        rola
        lslb
        rola
        orb     ,S+             ; Add in.
        std     ,U 
        dec     0,S             ; Done?
        bne     _octg0
_octxit lda     ,S+             ; If not 0 are digits left to do.
        rts

; Print an octal number.
; A= number of digits to (1 to 4).
; U= data location (2 bytes).
; A and B changed.
Pntoctl sta     _octp2          ; Save digit count.
        sta     _octp3          ; Also need character count.
        ldd     ,U
        std     _octp4
_octp0  lda     _octp5          ; Get data.
        anda    #7              ; Make into a character.
        adda    #'0
        pshs    A               ; Save on stack.
        ldd     _octp4          ; Move next digit into place.
        lsra
        rorb
        lsra
        rorb
        lsra
        rorb
        std     _octp4
        dec     _octp2
        bne     _octp0
_octp1  puls    A
        jsr     [>Conptr]
        dec     _octp3
        bne     _octp1
        rts
_octp2  rmb     1               ; Digit counter.
_octp3  rmb     1               ; Character counter.
_octp4  rmb     1               ; Data to print.
_octp5  rmb     1

; Print a binary number.
; Number in A, B changed.
Pntbin  sta     _binp2
        ldb     #8
_binp0  lda     _binp2
        lsla
        sta     _binp2
        rola
        anda    #%1
        adda    #'0
        pshs    A
        decb
        bne     _binp0
        ldb     #8
_binp1  puls    A
        jsr     [>Conptr]
        decb
        bne     _binp1
        rts
_binp2  rmb     1

; Monitor program load:
; Waits for a binary program to be loaded and
; then waits for XON. This means the terminal
; data link has been established. A startup
; message is printed and the routine exits.
PDPload jsr     PDPbin          ; Load the binary source.
        cmpa    #XON            ; Check for data link startup.
        bne     _lderr
        rts
_lderr  jsr     Errpnt
        fdb     NoXON

Sysmsg: fcb     CR,LF,CR,LF
        fcc     "                  PDP8-E EMULATOR"
        fcb     CR,LF,CR,LF
        fcc     " -> Waiting for binary file download at 9600 baud <-"
        fcb     CR,LF,$00

; Error message print routine.
; At error have a jsr to this
; routine followed by the error
; message.
Errpnt  orcc    #Imask          ; Turn off interrupts.
        jsr     [>CRptr]
        ldx     [0,S]           ; Get message pointer.
        jsr     [>Msgptr]
        jsr     [>CRptr]
        jsr     Pntreg
        bra     *               ; PROGRAM HALTED.

; Vector offset array. Loaded with access
; locations for the ROM routines.
Vlist:
Serinptr        fdb  Serin_sO
Seroutptr       fdb  Serout_sO
Clrlnptr        fdb  ClrLn_sO
Setfptr         fdb  Fclr_sO
CRexitptr       fdb  CRpause_sO
Baudptr         fdb  BDset_sO
CRptr           fdb  CRLF_sO
Conptr          fdb  ConOut_sO
Msgptr          fdb  Msg_sO
PIAptr          fdb  PIA_wO     ; PIA side A location.
PIAdata         fdb  PIA_dat_bO ; PIA side A output data.
PIAset          fdb  Aset_sO    ; PIA side A output pin set.
Fastinptr       fdb  Fastchk_sO
                fdb  0

; Message strings used by Pntreg
Stpmsg:
        fcc     " Df="
        fcb     $00
        fdb     WorkD
        fcb     $1
        fcc     " AC="
        fcb     $00
        fdb     AC
        fcb     $4
        fcc     " L="
        fcb     $00
        fdb     _stp2-1
        fcb     $1
        fcc     " MQ="
        fcb     $00
        fdb     MQ
        fcb     $4
        fcc     " U="
        fcb     $00
        fdb     _stp3-1
        fcb     $1
        fcc     " ION="
        fcb     $00
        fdb     _stp4-1
        fcb     $1
        fcc     " NoI="
        fcb     $00
        fdb     _stp5-1
        fcb     $1
        fcc     " Ireq="
        fcb     $00

;        fdb     _stp6-1
;        fcb     $1
        fdb     IOflags
        fcb     $5              ; All but JMP/JMS delay.

        fcb     CR              ; Show with next state.
        fcc     " If= "
        fcb     $00
        fdb     WorkUI
        fcb     $1
        fcc     " PC="
        fcb     $00
        fdb     PC
        fcb     $4
        fcc     " In="
        fcb     $00
        fdb     Instr
        fcb     $4
        fcb     $00             ; End of message list.

; Emulator messages.
Clrmsg: fcc     "Error in Clear subroutine (from IOset call)"
        fcb     $00
TTO_3_4 fcc     "IOP #3 or #7 attempted on device 04 (TTO)."
        fcb     $00
Insterr fcc     "Illegal instruction"
        fcb     $00
Hltmsg  fcc     "Program halt (HLT microcode in OPR instruction)"
        fcb     $00
NoIerr  fcc     "Unimplemented interrupt IOP attempted (IOT-00 #6)"
        fcb     $00
KM8Eerr fcc     "Unimplemented KM8-E instruction"
        fcb     $00
TTI_3_7 fcc     "IOP #3 or #7 attempted on device 03 (TTI)."
        fcb     $00
Lderr   fcc     "Error loading binary file"
        fcb     $00
NoXON   fcc     "Did not get XON as a start character after load"
        fcb     $00
NoFLD   fcc     "No memory allocated for this field"
        fcb     $00
Kbdmsg  fcb     CR
        fcb     LF
        fcc     " Keyboard input: "
        fcb     $00
Enabmsg fcb     CR, LF
        fcc     " Display on"
        fcb     CR, LF, $00
Dismsg  fcb     CR, LF
        fcc     " Display off"
        fcb     CR, LF, $00
SSmsg   fcb     CR
        fcb     LF
        fcc     " ?=     Print prompt list"
        fcb     CR
        fcb     LF
        fcc     " K=     Keyboard input"
        fcb     CR
        fcb     LF
        fcc     " E=     Enable display"
        fcb     CR
        fcb     LF
        FCC     " D=     Disable display"
        fcb     CR, LF
        fcc     " Space= Next step"
        fcb     CR
        fcb     LF
        fcc     " C-R=   Exit single stepping"
        fcb     CR
        fcb     LF
Spc8msg fcc     "                      "
        fcb     $00

; Simulated PDP8 memory. Maximum useage
; would be 64 Kbytes, which is the full
; 6809 addressing space. Originate at
; easily handled boundary.

        org     $2000

Field0: rmb     Fsize           ; $1000-$2FFF
Field1: rmb     Fsize           ; $3000-$4FFF
Field2: rmb     Fsize           ; $5000-$6FFF
Field3: rmb     Fsize           ; $7000-$8FFF
;Field4: rmb     Fsize
;Field5: rmb     Fsize
;Field6: rmb     Fsize
;Field7: rmb     Fsize

Memend  equ     *               Memory useage check.

        end     Start

