/* Fast S1-S9 record loader using
** COM at 57.6 Kbaud (1 start, 8 data and 2 stop
** bits). Give S record file name on the command
** line without an extension (".S19" will be
** appended). Changes file data to the binary
** equivalent with "S" changed to "B" and CR-LF
** removed. Starts communication link when finished.
** Cannot have more than one filename on the command
** line.
**
** CHANGES:
**   (1)  Feb 19,1999:
**        Direct port access to do fast binary
**        output.  The BIOSCOM functions do not
**        work with the newer fast processors,
**        and both the output and shift registers
**        must be clear before loading the output
**        register.
**
** TO DO:
**   (1)  Allow ".S19" extension on command line.
**   (2)  Look into multiple files and those without
**        a transfer address (i.e.: overlays).
**   (3)  Try direct access of port status, as the
**        bioscom function seems to be unable to
**        handle "break" correctly. Must disconnect
**        and then reconnect the line before break
**        is detected. May detect as if break was
**        a character, which would be in the chip
**        design.
*/

/* Start up the loader program on the HC11 board. */

#include  <stdio.h>
#include  <bios.h>
#include  <dos.h>
#include  <string.h>

#define   MAXLEN    78
#define   BUFSIZE   80
#define   DSR       0x0020
#define   CTS       0x0010
#define   BRK       0x1000
#define   RDY       0x0100
#define   ESCKEY    0x011B    /* Keyboard escape (ESC key) */
#define   CTLC      0x03      /* Control C (program exit) */
#define   XON       0x11      /* Also called DC1 */
#define   COM1      0
#define   COM2      1
#define   PORT      COM1

char s_exten[]=          ".S19";   /* File extension   */

main(argc, argv)

int  argc;
char *argv[];

{
FILE *f;
int       i, c;
unsigned int status;
char      filename[MAXLEN];

	/* Check that COM is active. If not,
	   the link may need a startup character
	   sent, so go into comlink. If working
	   into the ROM loader an error message
	   will be sent and the link exited.
	 */
	status= bioscom(3, NULL, PORT);
/*     printf("\t\nStatus= %x\n", status);*/
	if ((status & BRK) || (!(status & (DSR | CTS)))) comlink();

	/* Remove any characters from input buffer */
	while((bioscom(3, NULL, PORT) & 0X100) !=0) bioscom(2, NULL, PORT);

	/* If no filename start up com link */
	if (argc == 1) comlink();

	/* Open command line specified file (extension ".S19" assumed). */
	else if (argc == 2)
	{
		if (strlen(argv[1]) <= (MAXLEN-3))
		{
			/* Append extension to file name */
			strcpy(filename, argv[1]);
			strcat(filename, s_exten);
			if ((f = fopen(filename, "rb")) != NULL)
			{
				if (PORT == COM1) {
				/* Set COM2 to 57600 baud, no parity, 8 bits, 2 stop */
					outportb (0X3FB, 0X87);
					outportb (0X3F8, 0X02);
					outportb (0X3F9, 0X00);
					outportb (0X3FB, 0X07);
				}
				else if (PORT == COM2) {
					outportb (0X2FB, 0X87);
					outportb (0X2F8, 0X02);
					outportb (0X2F9, 0X00);
					outportb (0X2FB, 0X07);
				}
				else {
					printf ("unknown port\n");
					exit(1);
				}

				/* Send file */
				xfer(f);
				fclose(f);

				/* Start communication link */
				comlink();
			}
			else
			{
				printf("Could not open file %s\n", filename);
				return(1);
			}
		}
		else
		{
			printf("File name too long\n");
			return(1);
		}
	}
	else
	{
		printf("More than one file specified\n");
		return(1);
	}
}

/* Send file characters out COM: */

xfer(f)

FILE *f;

{
	char s_record[BUFSIZE], s_data;
	unsigned int line= 0, checksum;
	int  reclen, index, s_byte, s_word, recsize, endrec= 0;

	if (PORT == COM1) {
		/* wait for output buffer empty */
		while(((inp(0x3FD) & 0x60) != 0x60));
	}
	else {
		/* wait for output buffer empty */
		while(((inp(0x2FD) & 0x60) != 0x60));
	}

	while (fgets(s_record, BUFSIZE, f) != NULL)
	{
		reclen=   strlen(s_record);
		line++;
		if (reclen < 12)
			{
			printf ("record too short\n");
			exit(1);
			}
		if (reclen%2)
			{
			printf("improper record length\n");
			exit(1);
			}
		else reclen -= 2;   /* drop CR/LF */
		if(s_record[0]=='S')
			s_record[0]= 'B';
		else
			{
			printf ("not an S record\n");
			exit(1);
			}
		if (check(s_record, reclen) != 0xFF)
			{
			printf("bad checksum\n");
			exit(1);
			}
		if (s_record[1] == '1')
			{
			if (loadpoint(s_record) == 0x0000) endrec= 1;
			sendrec(s_record, reclen);
			}
		else if (s_record[1] == '9')
			{
			if ((loadpoint(s_record) == 0x0000)&&(!endrec))
				{
				printf("bad start (in loader code)\n");
				exit(1);
				}
			else
				{
				sendrec(s_record, reclen);
				break;
				}
			}
		else
			{
			printf("unacceptable record type (not S1 or S9)\n");
			exit(1);
			}
	}
	if ((fgets(s_record, BUFSIZE, f) != NULL))
		{
		printf("Warning: data after transfer record");
		exit(1);
		}

}

hexcon (s_data)

int  s_data;

{
	s_data &= 0xFF;

	if ((s_data >= '0') && (s_data <= '9' ))
		s_data -= '0';
	else if ((s_data >= 'A') && (s_data <= 'F'))
		s_data -= ('A' -10);
	else
		{
		printf("improper data (not hexadecimal)\n");
		exit(1);
		}
	return(s_data);
}

check(s_record, reclen)

char s_record[];
int  reclen;

{
int  index, s_byte;
unsigned int checksum;

	for (index= 0; index < reclen; index += 2)
		{
		s_byte= loadbyte(s_record, index);
		if (index == 2)
			{
			checksum= s_byte;
			if (((reclen/2) -s_byte) != 2)
				{
				printf("wrong record length\n");
				exit(1);
				}
			}
		else checksum += s_byte;
		}
	return (checksum&0xFF);
}

loadbyte(s_record, index)

char s_record[];
int  index;

{
	return(((hexcon(s_record[index])<<4)|hexcon(s_record[index+1]))&0xFF);
}

loadpoint(s_record)

char s_record[];

{
int  index, s_word;

	for (s_word= 0, index= 4; index < 8; index++)
		{
		s_word = (s_word << 4)|hexcon(s_record[index]);
		}
	return(s_word&0xFFFF);
}

sendrec(s_record, reclen)

char s_record[];
int  reclen;

{
int  index, s_byte;

	for (index= 0; index < reclen; index += 2) {
		s_byte= loadbyte(s_record, index);

		if (PORT == COM1) {
			outp(0x3F8,s_byte);
			/* wait till both registers empty */
			while(((inp(0x3FD) & 0x60) != 0x60));
		}
		else {
			outp(0x2F8,s_byte);
			/* wait till both registers empty */
			while(((inp(0x2FD) & 0x60) != 0x60));
		}
	}
}

/* Communicate via COM at 9600 baud.
 * Sends "XON" when the program is first
 * entered. Exits to DOS if receive "ESC"
 * from the keyboard or "^C" from the link.
 */

comlink() {
int key;
unsigned int status;
char c;

/* Set COM to 9600 baud, no parity, 8 bits, 2 stop */
bioscom(0, 0XE0|0X04|0X03, PORT);
/* Send startup character */
bioscom(1, XON, PORT);

/* Check that COM is active */
status= bioscom(3, NULL, PORT);
/*printf("\t\nStatus= %x\n", status);*/
if ((status & BRK) || (!(status & (DSR | CTS))))
{
	printf("\n\tSerial port not active\n");
	exit(1);
}

ioloop:

/* Check for keyboard character. If present send out COM2 */
if (bioskey(1) != 0) {
	key= bioskey(0);
	if (key == ESCKEY)
		exit(0);
	else 
		bioscom(1, key, PORT);
}
	   
/* Check for COM input. If present, send to screen */
if ((bioscom(3, NULL, PORT) & 0X100) != 0) {
	key= bioscom(2, NULL, PORT);
	if (key == CTLC)
		exit(0);
	else
		putch(key);
}

/* Do again */
goto ioloop;

}


