#include <stdlib.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <string.h>
#include <poll.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

int main(int argc, char*argv[]) {
  char* filename = "/tmp/test_file.txt";
  if (argc >= 2)
    filename = argv[1];

  int s = socket(AF_INET, SOCK_STREAM, 0);
  if (s < 0) {
    perror("socket");
    return -1;
  }

  int val = 1;
  if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val)) != 0) {
    perror("setsockopt");
    return -1;
  }

  in_addr_t ia = inet_addr("127.0.0.1");
  in_port_t ip = htons(1965);

  struct sockaddr_in in;
  in.sin_family = AF_INET;
  in.sin_port = ip;
  in.sin_addr.s_addr = ia;

  if (bind(s, (struct sockaddr*)&in, sizeof(in)) < 0) {
    perror("bind");
    return -1;
  }

  if (listen(s, 5) < 0) {
    perror("listen");
  }

  SSL_CTX* ctx = SSL_CTX_new(TLS_server_method());
  if (!ctx) {
    fprintf(stderr, "No SSL_CTX\n");
    return -1;
  }

  // cert.pm is the default certificate
  char const* cert_file = "cert.pem"; // TODO make configurable
  int r = SSL_CTX_use_certificate_chain_file(ctx, cert_file);
  if (r != 1) {
    unsigned long e = ERR_get_error();
    fprintf(stderr, "could not use cert chain file %s: %s (%lu)", cert_file, ERR_reason_error_string(e), e);
  }

  // load the private key from file, default "key.pem"
  char const* key_file = "key.pem"; // TODO make configurable

  ERR_clear_error();

  r = SSL_CTX_use_PrivateKey_file(ctx, key_file, SSL_FILETYPE_PEM);
  if (r != 1) {
    fprintf(stderr, "could not load private key\n");
    return -1;
  }

  // make sure the private key matches the certificate
  r = SSL_CTX_check_private_key(ctx);
  if (r != 1)
    fprintf(stderr, "private key does not match certificate chain: %s", ERR_reason_error_string(ERR_get_error()));

  // avoid SSLv2 and SSLv3
  SSL_CTX_set_options(ctx, SSL_OP_NO_SSLv2|SSL_OP_NO_SSLv3);


  const char* const PREFERRED_CIPHERS = "HIGH:!aNULL:!kRSA:!SRP:!PSK:!CAMELLIA:!RC4:!MD5:!DSS";
  if (SSL_CTX_set_cipher_list(ctx, PREFERRED_CIPHERS) == 0) {
    fprintf(stderr, "couldn't set cipher list\n");
    return -1;
  }

  SSL_CTX_set_verify(ctx, SSL_VERIFY_NONE, NULL);

  while (1) {
    int f = accept(s, NULL, NULL);
    if (f < 0) {
      perror("accept");
      return -1;
    }
    fprintf(stderr, "accepted a connection\n");

    SSL* ssl = SSL_new(ctx);
    if (!ssl) {
      fprintf(stderr, "No SSL\n");
    }

    SSL_set_fd(ssl, f);

    if (SSL_accept(ssl) <= 0) {
      fprintf(stderr, "error on accept: %s\n", ERR_reason_error_string(ERR_get_error()));
      return -1;
    }

    char* hdr = "application/octet-stream\r\n";

    FILE* fh = fopen(filename, "r");
    if (!fh) {
      perror("fopen");
      return -1;
    }
    fprintf(stderr, "opened %s\n", filename);

    int r = SSL_write(ssl, hdr, strlen(hdr));
    if (r != strlen(hdr)) {
      fprintf(stderr, "SSL error on header send: %d\n", r);
      return -1;
    }

    size_t total_sent = 0;
    while (!feof(fh) && !ferror(fh)) {
      size_t bufsz = 10240;
      char* buf = malloc(bufsz);

      size_t bufl = fread(buf, 1, bufsz, fh);
      if (bufl > 0) {
        int r = SSL_write(ssl, buf, bufl);
        if (r != bufl) {
          fprintf(stderr, "SSL error: %d\n", r);
          return -1;
        }
        total_sent += r;
        fprintf(stdout, "wrote %d bytes (%lu total)\n", r, total_sent);
      }
      free(buf);
    }

    SSL_shutdown(ssl);
    SSL_free(ssl);
    fprintf(stderr, "closed connection\n");
  }

  SSL_CTX_free(ctx);
}
