#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_path.h>
#include <libpxd/px_log.h>
#include <errno.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stddef.h>

#define PARANOID
#define COMPONENT_INCREMENT 8

#define ROUNDUP(x, r) ((((x) + (r) - 1)/(r))*(r))

_Bool px_path_is_valid(struct px_path const* path) {
  if (!path)
    return 0;

  // basically:
  return !!path->components_sz == (path->components != NULL);
}

static int count_path_components(char const* str) {
  if (!str || str[0] == '\0')
    return 0;

  int n_components = 1;
  for (char const* itr = str; *itr; ++itr) {
    if (itr[0] == '/') {
      while (itr[1] == '/')
        ++itr;
      if (itr[1] != '\0')
        ++n_components;
    } else {
      while (itr[1] != '/' && itr[1] != '\0')
        ++itr;
    }
  }
  return n_components;
}

struct px_path px_path_from_str(char const* str) {
  size_t n_components = 0;
  _Bool is_abs = false;
  _Bool is_dir = false;

  // null or empty string gets nothing
  if (!str || *str == '\0')
    return (struct px_path) { 0 };

  if (*str == '/') {
    is_abs = true;
    is_dir = true;
    while (*str == '/')
      ++str;
  }

  // determine the number of components we need.  each new component is
  // terminated by a slash or null.  other slashes are skipped
  // loop invariant: *comp_start != '/' is guaranteed, *comp_start == '\0' may be true
  char const* comp_start = str;
  while (*comp_start != '\0') {
    char const* comp_end = comp_start;
    while (*comp_end != '/' && *comp_end != '\0')
      ++comp_end;

    px_log_assert(comp_end != comp_start, "logic error?  comp_end should never match comp_start");
    ++n_components;

    while (*comp_end == '/')
      ++comp_end;
    comp_start = comp_end;
  }

  if (n_components == 0)
    return (struct px_path) { .is_abs = is_abs, .is_dir = is_dir, .components = NULL, .components_sz = 0 };

  px_log_assert(n_components > 0, "logic error, n_components is not expected to ever be 0 here");

  // sanity check
  if (n_components > PX_PATH_MAX_COMPONENTS) {
    px_log_warn("exceeded max number of components");
    return (struct px_path) { 0 };
  }

  // allocate space for the components
  // from here down on failure we need to goto FAIL to free components
  char** components = (char**)calloc(n_components, sizeof(char*));
  if (!components) {
    px_log_error("no memory");
    return (struct px_path) { 0 };
  }

  // copy the components
  size_t cur_idx = 0;
  comp_start = str;
  while (*comp_start != '\0') {
    char const* comp_end = comp_start;
    // find the first non-delimiting character
    while (*comp_end != '/' && *comp_end != '\0')
      ++comp_end;

    // get the component length and make a copy of it
    size_t comp_len = comp_end - comp_start;
    char* new_component = strndup(comp_start, comp_len);

    // determine whether the component is a directory
    // either the component:
    // - ends with a /
    // - is . and is either an absolute path or there was an earlier component (i.e. there was a /)
    // - is .. and is either an absolute path or there was an earlier component (i.e. there was a /)
    if (*comp_end == '/'
        || ((cur_idx > 0 || is_abs) && strcmp(new_component, ".") == 0)
        || ((cur_idx > 0 || is_abs) && strcmp(new_component, "..") == 0))
    {
      is_dir = true;
    } else {
      is_dir = false;
    }

    if (!new_component || *new_component == '\0') { // second one shouldn't happen
      free(new_component);
      goto FAIL;
    }

    if (comp_len == 1 && strcmp(new_component, ".") == 0) {
      free(new_component); // skip 'current directory'
    } else if (comp_len == 2 && strcmp(new_component, "..") == 0) {
      // double-dot .. removes the previous entry (if it's not the leading absolute component)
      free(new_component);
      if (cur_idx > 0) {
        --cur_idx;
        free(components[cur_idx]);
        components[cur_idx] = NULL;
      }
    } else {
      components[cur_idx] = new_component;
      ++cur_idx;
    }

    while (*comp_end == '/')
      ++comp_end;
    comp_start = comp_end;
  }

  px_log_assert(cur_idx <= n_components, "memory overrun");
  if (cur_idx != n_components) { // shorten the allocation if needed
    if (cur_idx > 0) {
      char** new_components = (char**)realloc(components, sizeof(char*) * cur_idx);
      if (new_components) // make sure the realloc succeeded
        components = new_components;
    } else { // no components
      free(components);
      components = NULL;
    }
  }

  return (struct px_path) { .is_abs = is_abs,
                            .is_dir = is_dir,
                            .components = components,
                            .components_sz = cur_idx };

FAIL:
  if (components) {
    for (size_t i = 0; i < n_components && components[i] != NULL; ++i)
      free(components[i]);
    free(components);
  }
  return (struct px_path) { 0 };
}

struct px_path px_path_from_str_clean(char const* str) {
  struct px_path from_str = px_path_from_str(str);
  struct px_path cleaned = px_path_clean(&from_str);
  px_path_reset(&from_str);
  return cleaned;
}

static int path_needs_split(struct px_path const* path) {
  if (!path || path->components_sz == 0)
    return 0;

  for (size_t i = 0, n = path->components_sz; i < n; ++i) {
    if (!path->components[i] || path->components[i][0] == '\0')
      continue;

    // if a nonempty path component does not have one subcomponent then it
    // means it's got an internal slash and could be split into two additional
    // components
    if (count_path_components(path->components[i]) != 1)
      return 1;
  }
  return 0;
}

// compresses a path into a minimal set of acceptable components
// note this doesn't check individual components for internal slashes - those
// shouldn't be there anyway
struct px_path px_path_clean(struct px_path const* path) {
  struct px_path cleaned = { 0 };

  if (!path || path->components_sz == 0)
    return cleaned;

  // we don't handle malformed components with internal slashes
  if (path_needs_split(path))
    return cleaned;

  size_t n_components = path->components_sz;

  size_t out_i = 0;
  char** new_components = (char**)calloc(n_components, sizeof(char*));

  for (size_t in_i = 0; in_i < n_components; ++in_i) {

    // null or empty component, skip
    if (!path->components[in_i] || path->components[in_i][0] == '\0')
      continue;

    // single dot, skip
    if (strncmp(path->components[in_i], "./", 2) == 0 || strcmp(path->components[in_i], ".") == 0)
      continue;

    // double dot == previous dir, remove the last entry if it's not the first / (i.e. absolute dir marker)
    if (strncmp(path->components[in_i], "../", 3) == 0 || strcmp(path->components[in_i], "..") == 0) {
      if (out_i > 0)
        --out_i;

      // keep a leading '/' (from an abs path)
      if (out_i == 0 && new_components[out_i] && new_components[out_i][0] == '/') {
        ++out_i;
        continue;
      }

      free(new_components[out_i]);
      new_components[out_i] = NULL;
      continue;
    }

    if (path->components[in_i][0] == '/') {
      // if it's the first index for both sets then copy the / to make an
      // absolute path, otherwise skip it
      if (in_i == 0 && out_i == 0) {
        new_components[out_i] = strdup("/");
        if (!new_components[out_i])
          goto FAIL;
        ++out_i;
      }
      continue;
    }

    // no more special cases

    // find the next / or the end of string
    char* itr = path->components[in_i];
    while (*itr && *itr != '/')
      ++itr;

    // if we landed on a (series of) slash, include it/the first one
    if (*itr == '/')
      ++itr;

    char* new_comp = NULL;
    if (in_i < n_components - 1) {
      // anything that isn't the last element is done in this block

      if (itr[-1] != '/') {
        // if the last character in the string is not a slash then add one
        size_t comp_len = strlen(path->components[in_i]);
        new_comp = calloc(1, comp_len + 2);
        if (!new_comp)
          goto FAIL;
        strcpy(new_comp, path->components[in_i]);
        new_comp[comp_len] = '/';
        new_comp[comp_len + 1] = '\0';
      } else {
        // slash is already present in the intermediate component, copy it directly
        new_comp = strndup(path->components[in_i], itr - path->components[in_i]);
      }
    } else {
      // if it's the last component, then copy it without any other checking
      new_comp = strndup(path->components[in_i], itr - path->components[in_i]);
    }

    if (!new_comp)
      goto FAIL;
    new_components[out_i] = new_comp;
    ++out_i;
  }

  cleaned.components_sz = out_i;
  cleaned.components = new_components;
  return cleaned;

FAIL:
  if (new_components) {
    for (size_t i = 0; i < out_i; ++i) {
      free(new_components[i]);
    }
    free(new_components);
  }
  cleaned = (struct px_path) { 0 };
  return cleaned;
}



void px_path_init(struct px_path* path) {
  memset(path, 0, sizeof(*path));
}

void px_path_reset(struct px_path* path) {
  if (!path)
    return;
  if (path->components) {
    for (size_t i = 0, n = path->components_sz; i < n; ++i) {
      free(path->components[i]);
      path->components[i] = NULL;
    }
    free(path->components);
  }
  px_path_init(path);
}

struct px_path px_path_copy(struct px_path const* src) {
  struct px_path dst = { 0 };
  if (!src)
    return dst;

  if (!src->components || src->components_sz == 0) {
    return dst;
  }

  char** new_components = (char**)calloc(src->components_sz, sizeof(char*));
  if (!new_components) // couldn't copy
    return dst;

  for (size_t i = 0, n = src->components_sz; i < n; ++i) {
    if (!src->components[i])
      continue;
    char* str_i = strdup(src->components[i]);
    if (!str_i) { // couldn't strdup, free everything else and bail
      for (size_t j = 0; j < i; ++j)
        free(new_components[j]);
      free(new_components);
      return dst;
    }
    new_components[i] = str_i;
  }

  dst.components = new_components;
  dst.components_sz = src->components_sz;
  dst.is_abs = src->is_abs;
  dst.is_dir = src->is_dir;
  return dst;
}

void px_path_move(struct px_path* dst, struct px_path* src) {
  *dst = *src;
  *src = (struct px_path) { 0 };
}

struct px_path px_path_strip_leading(struct px_path const* src, unsigned n) {
  if (!src || src->components_sz == 0)
    return (struct px_path) { .components = NULL, .components_sz = 0 };

  if ((size_t)n > src->components_sz)
    n = src->components_sz;

  if (n == 0)
    return px_path_copy(src);

  // we're abusing the px_path structure a bit by constructing this on the
  // stack, but it avoids a temporary copy since we're passing it directly to
  // px_path_copy which does not care where the components are allocated
  struct px_path dummy = {  .is_abs = src->is_abs,
                            .is_dir = src->is_dir,
                            .components_sz = src->components_sz - n,
                            .components = src->components + n };

  struct px_path ret = px_path_copy(&dummy);

  return ret;
}

struct px_path px_path_concat(struct px_path const* lpath, struct px_path const* rpath) {
  if (!lpath)
    return px_path_copy(rpath);

  if (!rpath)
    return px_path_copy(lpath);

  _Bool result_is_abs = (lpath && lpath->is_abs) || (lpath == NULL && rpath->is_abs);
  _Bool result_is_dir = (rpath && rpath->is_dir) || (rpath == NULL && lpath->is_dir);

  // we skip the first componet, which is the '/' component, in rpath (if it's absolute)
  size_t n_total_components = lpath->components_sz + rpath->components_sz;
  if (n_total_components == 0)
    return (struct px_path) { 0 };

  char** new_components = (char**)calloc(n_total_components, sizeof(char*));
  if (!new_components) {
    px_log_error("no memory");
    return (struct px_path) { 0 };
  }

  size_t comp_idx = 0;
  char* const* cur_component = lpath->components;
  char* const* end_l_component = lpath->components + lpath->components_sz;
  while (cur_component != end_l_component) {
    new_components[comp_idx] = strdup(*cur_component);
    if (!new_components[comp_idx]) {
      px_log_error("no memory");
      goto FAIL;
    }
    ++cur_component;
    ++comp_idx;
  }

  cur_component = rpath->components;
  char* const* end_r_component = rpath->components + rpath->components_sz;
  while (cur_component != end_r_component) {
    new_components[comp_idx] = strdup(*cur_component);
    if (!new_components[comp_idx]) {
      px_log_error("no memory");
      goto FAIL;
    }
    ++cur_component;
    ++comp_idx;
  }

  px_log_assert(comp_idx == n_total_components, "%lu != %lu", comp_idx, n_total_components);

  return (struct px_path) { .is_abs = result_is_abs,
                            .is_dir = result_is_dir,
                            .components = new_components,
                            .components_sz = n_total_components };
FAIL:
  for (size_t i = 0; i < n_total_components && new_components[i] != NULL; ++i) {
    free(new_components[i]);
  }
  free(new_components);
  return (struct px_path) { 0 };
}

_Bool px_path_str_is_abs(char const* str) {
  return str && str[0] == '/';
}

_Bool px_path_str_is_dir(char const* str) {
  size_t slen = str ? strlen(str) : 0;
  // we're a directory if:
  // - end with a /
  // - end with /.
  // - end with /..
  return slen > 0
         && (str[slen - 1] == '/'
             || (slen > 2 && strcmp(&str[slen - 2], "/.") == 0)
             || (slen > 3 && strcmp(&str[slen - 3], "/..") == 0));
}

struct px_path px_path_concat_consume(struct px_path* pre, struct px_path* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(pre);
  px_path_reset(suf);
  return concat;
}

struct px_path px_path_concat_consume0(struct px_path* pre, struct px_path const* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(pre);
  return concat;
}

struct px_path px_path_concat_consume1(struct px_path const* pre, struct px_path* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(suf);
  return concat;
}

struct px_path px_path_concat_clean(struct px_path const* pre, struct px_path const* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  struct px_path cleaned = px_path_clean(&concat);
  px_path_reset(&concat);
  return cleaned;
}

char* px_path_to_str(struct px_path const* path) {
  if (!path)
    return NULL;

  if (path->components_sz == 0)
    return path->is_abs ? strdup("/") : strdup("");

  _Bool     is_dir = path->is_dir;
  unsigned  is_abs = path->is_abs;

  char** components = path->components; // skip the absolute component
  size_t n_components = path->components_sz;

  // don't need a dir terminator if it's only '/'
  _Bool need_dir_terminator = (is_dir && (!is_abs || n_components > 0));
  size_t nchars = is_abs + (unsigned)need_dir_terminator + 1; // absolute / and null byte
  if (n_components > 1)         // only need separators if there are more than one component
    nchars += n_components - 1; // then add one char for each separator

  for (unsigned i = 0; i < n_components; ++i) {
    size_t new_nchars = nchars + strlen(components[i]);
    if (new_nchars < nchars) { // overflow
      px_log_error("overflow");
      return NULL;
    }
    nchars = new_nchars;
  }

  char* path_str = (char*)malloc(nchars);
  if (!path_str) {
    px_log_error("no memory");
    return NULL;
  }

  char* str_out = path_str;
  if (is_abs) {
    *str_out = '/';
    ++str_out;
  }

  for (size_t comp_idx = 0; comp_idx < n_components; ++comp_idx) {
    char const* comp_itr = components[comp_idx];
    while (*comp_itr != '\0') {
      *str_out = *comp_itr;
      ++str_out;
      ++comp_itr;
    }

    // add a separator if needed
    if (comp_idx < n_components - 1) {
      *str_out = '/';
      ++str_out;
    }
  }

  // append a terminating / for directories
  if (need_dir_terminator) {
    *str_out = '/';
    ++str_out;
  }

  *str_out = '\0';
  px_log_assert(str_out == path_str + nchars - 1, "logic error");
  return path_str;
}

char const* px_path_get_component(struct px_path const* path, int comp) {
  if (!path)
    return NULL;

  int idx = comp >= 0 ? comp : (int)path->components_sz + comp;
  if (idx >= 0 && idx < (int)path->components_sz)
    return path->components[idx];
  return NULL;
}

_Bool px_path_is_abs(struct px_path const* path) {
  return path && path->is_abs;
}

_Bool px_path_is_dir(struct px_path const* path) {
  return path && path->is_dir;
}

_Bool px_path_make_abs(struct px_path* path) {
  if (!path)
    return false;
  path->is_abs = true;
  return true;
}

_Bool px_path_make_rel(struct px_path* path) {
  if (!path)
    return false;

  path->is_abs = false;
  return true;
}

_Bool px_path_make_dir(struct px_path* path) {
  if (!path)
    return false;
  path->is_dir = true;
  return true;
}

_Bool px_path_make_file(struct px_path* path) {
  if (!path)
    return false;
  path->is_dir = false;
  return true;
}
