#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_listen.h>
#include <libpxd/px_log.h>
#include <arpa/inet.h>
#include <errno.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>

int px_bind_inet4(in_addr_t addr, in_port_t port) {
  struct sockaddr_in in = { .sin_family = AF_INET,
                            .sin_addr = { .s_addr = addr },
                            .sin_port = port };

  struct addrinfo ai = { .ai_family = AF_INET,
                         .ai_socktype = SOCK_STREAM,
                         .ai_protocol = IPPROTO_TCP,
                         .ai_addr = (struct sockaddr*)&in,
                         .ai_addrlen = sizeof(in) };

  return px_bind_addrinfo(&ai);
}

int px_bind_unix(char const* path) {
  if (!path || *path == '\0')
    return -1;
  size_t path_len = strlen(path);
  const size_t max_path_len = sizeof(((struct sockaddr_un*)0)->sun_path);
  if (path_len > max_path_len) {
    px_log_error("path %s is too long (max %lu)\n", path, max_path_len);
    return -1;
  }

  struct sockaddr_un un = { .sun_family = AF_UNIX };
  memcpy(un.sun_path, path, path_len);
  if (path_len < max_path_len)
    px_log_assert(un.sun_path[path_len] == '\0', "path is not null terminated");

  struct addrinfo ai = { .ai_family = AF_UNIX,
                         .ai_socktype = SOCK_STREAM,
                         .ai_protocol = 0,
                         .ai_addr = (struct sockaddr*)&un,
                         .ai_addrlen = sizeof(un) };

  return px_bind_addrinfo(&ai);
}

//_Bool struct px_listener* listener,
int px_bind_addrinfo(struct addrinfo const* ai) {
  int s = socket(ai->ai_family, ai->ai_socktype | SOCK_CLOEXEC | SOCK_NONBLOCK, ai->ai_protocol);
  if (s < 0) {
    int e = errno;
    px_log_error("could not create socket: %d, %s\n", e, strerror(e));
    errno = e;
    return -1;
  }

  int val = 1;
  if (ai->ai_family == AF_INET || ai->ai_family == AF_INET6) {
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val)) != 0) {
      int e = errno;
      px_log_error("could not SO_REUSEADDR: %s", strerror(e));
      errno = e;
    }
  } else {
    // some extra checks to make sure we don't unlink something we shouldn't
    px_log_assert(ai->ai_addrlen >= sizeof(struct sockaddr_un), "invalid struct size");
    struct sockaddr_un* un = (struct sockaddr_un*)ai->ai_addr;
    // TODO check for nul in sun_path
    unlink(un->sun_path);
  }

  int r = bind(s, ai->ai_addr, ai->ai_addrlen);
  if (r < 0) {
    int e = errno;
    px_log_error("could not bind socket: %s", strerror(e));
    close(s);
    errno = e;
    return false;
  }

  r = listen(s, 5);
  if (r < 0) {
    int e = errno;
    px_log_error("could not listen on socket: %s", strerror(e));
    close(s);
    errno = e;
    return false;
  }

  return s;
}

//struct px_listener* listener,
int px_bind(char const* listen_addr, char const* host, char const* port)
{
  int fd = -1;
  char const unix_prefix[] = "unix:";

  // if our listen address is prefixed with unix: then the rest of the string
  // represents a unix socket path to listen on
  if (listen_addr && strncmp(listen_addr, unix_prefix, sizeof(unix_prefix) - 1) == 0) {
    //px_log_assert(sizeof(listen_buf) >= sizeof(((struct sockaddr_un*)0)->sun_path),
    //              "buffer sizes need adjustment");

    char const* laddr = listen_addr + (sizeof(unix_prefix) - 1);

    // make sure port is either missing of empty, it makes no sense in this context
    if (port && *port != '\0')
      px_log_warn("listening socket %s was also specified with port %s.  "
                  "ports are meaningless for unix sockets", laddr, port);
    port = NULL;
    fd = px_bind_unix(laddr);
  } else if (listen_addr || host || port) { // not a unix socket, figure out where to listen on tcp

    char const* laddr = listen_addr ? listen_addr : host;
    if (listen_addr && strcmp(listen_addr, "any") == 0)
      laddr = NULL;

    struct addrinfo* ai = NULL;
    struct addrinfo* itr = NULL;
    struct addrinfo hints = { .ai_family = AF_UNSPEC, .ai_protocol = IPPROTO_TCP, .ai_flags = AI_PASSIVE };

    int e = getaddrinfo(laddr, port, &hints, &ai);

    if (e != 0) {
      px_log_error("could not getaddrinfo(%s:%s): %s",
                   laddr ? laddr : "(null)",
                   port ? port : "(null)",
                   gai_strerror(e));
      return -1;
    }

    for (itr = ai; itr; itr = itr->ai_next) {
      if ((itr->ai_family != AF_INET && itr->ai_family != AF_INET6)
          || itr->ai_protocol != IPPROTO_TCP)
        continue;

      fd = px_bind_addrinfo(ai);
      if (fd >= 0)
        break;
    }

    freeaddrinfo(ai);
  } else {
    struct sockaddr_in in = { .sin_family = AF_INET,
      .sin_addr = { .s_addr = INADDR_ANY },
      .sin_port = htons(GEMINI_PORT) };

    struct addrinfo ai = {  .ai_family = AF_INET,
                            .ai_socktype = SOCK_STREAM,
                            .ai_protocol = IPPROTO_TCP,
                            .ai_addr = (struct sockaddr*)&in,
                            .ai_addrlen = sizeof(in) };

    fd = px_bind_addrinfo(&ai);
  }

  return fd;
}

void px_listen_socket_update_addrstr(struct px_listen_socket* lsn) {
  memset(lsn->host_str, 0, sizeof(lsn->host_str));

  if (lsn->fd < 0)
    return;

  struct sockaddr_storage ss;
  memset(&ss, 0, sizeof(ss));

  socklen_t sa_len = sizeof(ss);
  if (getsockname(lsn->fd, (struct sockaddr*)&ss, &sa_len) < 0) {
    px_log_error("could not getsockname for fd %d: %s", lsn->fd, strerror(errno));
    return;
  }
  px_log_assert(sa_len <= sizeof(ss), "programming error, buffers need to be resized");

  if (!px_net_addr_to_str(lsn->fd, &lsn->host_str[0], sizeof(lsn->host_str), NULL, 0, NULL, 0, false))
    memset(lsn->host_str, 0, sizeof(lsn->host_str));
}

