#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_gemini_ctx.h>
#include <libpxd/px_log.h>
#include <ctype.h>
#include <errno.h>
#include <poll.h>
#include <unistd.h>

// TLS has a max message length of 16k, use that as our default buffer size so
// that we can be as efficient as possible
#define PX_GEMINI_QUEUE_SZ 16384

static int get_ssl_gemctx_index(void) {
  static int ssl_data_idx = -1;
  if (ssl_data_idx < 0)
    ssl_data_idx = SSL_get_ex_new_index(0, NULL, NULL, NULL, NULL);
  return ssl_data_idx;
}

typedef px_op_status (*gemini_stage_function) (struct px_gemini_context*);

// functions for the the different stages of the Gemini exchange
static px_op_status on_start_exchange       (struct px_gemini_context* gemctx);
static px_op_status on_handshake            (struct px_gemini_context* gemctx);
static px_op_status on_handshake_complete   (struct px_gemini_context* gemctx);
static px_op_status on_request_receive      (struct px_gemini_context* gemctx);
static px_op_status on_response_send        (struct px_gemini_context* gemctx);
static px_op_status on_response_flush       (struct px_gemini_context* gemctx);
static px_op_status on_shutdown             (struct px_gemini_context* gemctx);

static SSL_CTX* default_sslctx = NULL;

void px_gemini_context_set_default_sslctx(SSL_CTX* def) {
  default_sslctx = def;
}

void px_gemini_context_init(struct px_gemini_context* gemctx) {
  *gemctx = (struct px_gemini_context) { 0 };
  px_connection_init(&gemctx->conn);
}

void px_gemini_context_reset(struct px_gemini_context* gemctx) {
  if (!gemctx)
    return;

  //px_gemini_reset(&gemctx->gemreq);
  free(gemctx->sni_hostname);
  if (gemctx->client_cert.cert)
    X509_free(gemctx->client_cert.cert);
  free(gemctx->client_cert.subject_name);
  free(gemctx->client_cert.digests.cert_digest);
  free(gemctx->client_cert.digests.pubkey_digest);

  px_url_reset(&gemctx->request.url);

  // reset the connection and gemini stuff
  px_connection_reset(&gemctx->conn);

  px_gemini_context_init(gemctx);
}

_Bool px_gemini_context_setup_exchange(
    struct px_gemini_context* gemctx,
    int fd,
    SSL_CTX* sslctx,
    struct px_gemini_callbacks* callbacks,
    struct px_gemini_settings* settings)
{
  if (!sslctx)
    sslctx = default_sslctx;

  if (!px_connection_start(&gemctx->conn, fd, sslctx)) {
    px_log_error("could not start connection");
    return false;
  }

  if (callbacks)
    gemctx->callbacks = *callbacks;

  if (settings)
    gemctx->settings = *settings;

  return true;
}

_Bool px_gemini_context_data_transmitted(struct px_gemini_context const* gemctx) {
  return gemctx->stage > PX_GEMINI_RESPONSE_FLUSH;
}

_Bool px_gemini_context_is_finished(struct px_gemini_context const* gemctx) {
  return gemctx->stage == PX_GEMINI_DONE || gemctx->stage == PX_GEMINI_ERROR;
}

_Bool px_gemini_context_in_error(struct px_gemini_context const* gemctx) {
  return gemctx->stage == PX_GEMINI_ERROR;
}

_Bool px_gemini_context_did_partial_shutdown(struct px_gemini_context const* gemctx) {
  const int ssl_shutdown_done = (SSL_SENT_SHUTDOWN | SSL_RECEIVED_SHUTDOWN);
  int ssl_shutdown = gemctx->conn.ssl ? SSL_get_shutdown(gemctx->conn.ssl) : 0;
  return (ssl_shutdown & ssl_shutdown_done) != 0;
}

_Bool px_gemini_context_did_server_shutdown(struct px_gemini_context const* gemctx) {
  int ssl_shutdown = gemctx->conn.ssl ? SSL_get_shutdown(gemctx->conn.ssl) : 0;
  return (ssl_shutdown & SSL_SENT_SHUTDOWN) != 0;
}

_Bool px_gemini_context_did_client_shutdown(struct px_gemini_context const* gemctx) {
  int ssl_shutdown = gemctx->conn.ssl ? SSL_get_shutdown(gemctx->conn.ssl) : 0;
  return (ssl_shutdown & SSL_RECEIVED_SHUTDOWN) != 0;
}

_Bool px_gemini_context_did_full_shutdown(struct px_gemini_context const* gemctx) {
  const int ssl_shutdown_done = (SSL_SENT_SHUTDOWN | SSL_RECEIVED_SHUTDOWN);
  int ssl_shutdown = gemctx->conn.ssl ? SSL_get_shutdown(gemctx->conn.ssl) : 0;
  return ((ssl_shutdown & ssl_shutdown_done) == ssl_shutdown_done);
}

_Bool px_gemini_context_waiting_on_network(struct px_gemini_context const* gemctx) {
  return px_connection_waiting_on_network(&gemctx->conn);
}

// this gets called after the SSL connection has received an SNI name.  we need
// to determine if the hostname is acceptable or not.  this is a proxy function
// for calling gemctx->callbacks.on_sni
static int gemini_context_tlsext_servername_cb(SSL *s, int *al, void *arg) {
  (void)al;
  (void)arg;

  int idx = get_ssl_gemctx_index();
  if (idx == -1) // unable to figure out an index?  accept the connection
    return SSL_TLSEXT_ERR_OK;

  // no gemini context?
  // TODO this should probably be an error
  struct px_gemini_context* gemctx = SSL_get_ex_data(s, idx);
  if (gemctx == NULL)
    return SSL_TLSEXT_ERR_OK;

  int servername_type = SSL_get_servername_type(s);
  char const* servername = (servername_type == TLSEXT_NAMETYPE_host_name)
                           ? SSL_get_servername(s, servername_type)
                           : NULL;
  // if there is an SNI callback then run it.
  // TODO I'm not sure this is right, maybe need to check if we're in a
  // handshake?
  if (gemctx->settings.sni_required) {
    if (servername == NULL) {
      px_log_info("%s: SNI required, but no SNI server name was found", gemctx->conn.peer_addr_str);
      return SSL_TLSEXT_ERR_NOACK;
    }
  }

  px_log_assert(gemctx->sni_hostname == NULL, "sni hostname should not be set");
  if (servername) {
    gemctx->sni_hostname = strdup(servername);
    if (gemctx->sni_hostname == NULL) {
      px_log_error("no memory");
      return SSL_TLSEXT_ERR_NOACK;
    }
  }

  if (gemctx->callbacks.on_sni) {
    // run the callback, translate the return into what OpenSSL expects
    _Bool res = gemctx->callbacks.on_sni(gemctx, servername);

    // if res == true then continue with the TLS connection, otherwise abort
    return res ? SSL_TLSEXT_ERR_OK : SSL_TLSEXT_ERR_NOACK;
  }

  // no callback, accept any hostname
  return SSL_TLSEXT_ERR_OK;
}

px_op_status on_start_exchange(struct px_gemini_context* gemctx) {
  px_log_assert(gemctx, "gemctx should not be NULL");
  px_log_assert(gemctx->conn.ssl, "gemctx ssl should have been set up already");

  int idx = get_ssl_gemctx_index();
  if (idx == -1) { // unable to figure out an index?  accept the connection
    px_log_warn("could not get ssl gemctx data index");
    return PX_OP_ERROR;
  }

  // hook our gemctx into the SSL data structure so we can get to it from OpenSSL
  // callbacks
  if (SSL_set_ex_data(gemctx->conn.ssl, idx, gemctx) != 1) {
    px_log_warn("could not set ssl gemctx data at index %d", idx);
    return PX_OP_ERROR;
  }

  // set up the SNI callback; it is where we call gemctx->callbacks.on_sni
  SSL_CTX* sslctx = SSL_get_SSL_CTX(gemctx->conn.ssl);
  px_log_assert(sslctx, "could not get SSL_CTX from SSL");

  if (SSL_CTX_set_tlsext_servername_callback(sslctx, gemini_context_tlsext_servername_cb) != 1)
    return PX_OP_ERROR;

  // if we require a client cert on the SSL level then set the verify callback
  if (gemctx->settings.client_cert_requirement == PX_GEMINI_CLIENT_CERT_REQUIRE_SSL) {
    int (*cb)(int, X509_STORE_CTX *) = SSL_get_verify_callback(gemctx->conn.ssl);
    SSL_set_verify(gemctx->conn.ssl, SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT, cb);
  }

  // if the user wanted any notification when the connection was set up then call that callback now
  if (gemctx->callbacks.on_start)
    return gemctx->callbacks.on_start(gemctx);

  return PX_OP_DONE;
}

px_op_status on_handshake(struct px_gemini_context* gemctx) {
  px_op_status r = px_connection_do_handshake(&gemctx->conn);
  if (r != PX_OP_DONE)
    return r;
  return on_handshake_complete(gemctx);
}

static struct px_x509_digests hash_x509(X509* cert) {
  struct px_x509_digests digests = { 0 };
  unsigned char digest_buf[EVP_MAX_MD_SIZE];
  EVP_MD const* hashfunc = EVP_sha256();

  if (!cert || !hashfunc)
    return digests;

  // we do this to reduce boilerplate code
  // this is just pairs of digest functions to perform on the cert and
  // destinations for the resulting hex-encoded digest strings
  struct target_digest {
    int (*dgstfunc)(const X509 *, const EVP_MD *, unsigned char *, unsigned int *);
    char** out;
  };

  // define what we want to do
  struct target_digest target_ops[] = {
    { .dgstfunc = X509_digest, .out = &digests.cert_digest },
    { .dgstfunc = X509_pubkey_digest, .out = &digests.pubkey_digest }
  };

  char const* hexchars = "0123456789abcdef";

  for (unsigned opidx = 0, nops = px_n_elements(target_ops); opidx < nops; ++opidx) {
    // clear out our buffer
    unsigned digest_buf_sz = sizeof(digest_buf);
    memset(digest_buf, 0, digest_buf_sz);

    if (target_ops[opidx].dgstfunc(cert, hashfunc, digest_buf, &digest_buf_sz) != 1) {
      break;
    }

    unsigned hex_str_len = 2 * digest_buf_sz + 1;
    char* hexencoded = (char*)calloc(1, hex_str_len);
    if (!hexencoded) {
      px_log_error("no memory for hashing certificate fields");
      break;
    }

    char* outitr = hexencoded;
    for (unsigned i = 0; i < digest_buf_sz; ++i, outitr += 2) {
      unsigned b = digest_buf[i];
      outitr[0] = hexchars[(b & 0xF0) >> 4];
      outitr[1] = hexchars[(b & 0x0F)];
    }
    *outitr = '\0';

    *target_ops[opidx].out = hexencoded;
  }

  // check for errors
  if (!digests.cert_digest || !digests.pubkey_digest) {
    free(digests.cert_digest);
    free(digests.pubkey_digest);
    memset(&digests, 0, sizeof(digests));
  }

  return digests;
}

static char* get_cn_from_x509(X509* cert) {
  // Get the subject name
  X509_NAME* subject_name = X509_get_subject_name(cert);
  if (!subject_name)
    return NULL;

  // Find the CN entry (NID_commonName)
  int cn_index = X509_NAME_get_index_by_NID(subject_name, NID_commonName, -1);
  if (cn_index < 0)
    return NULL;

  X509_NAME_ENTRY *cn_entry = X509_NAME_get_entry(subject_name, cn_index);
  if (!cn_entry)
    return NULL;

  // Get the ASN1_STRING
  ASN1_STRING *cn_asn1 = X509_NAME_ENTRY_get_data(cn_entry);
  if (!cn_asn1)
    return NULL;

  // Convert to UTF-8 string
  char* cn_str = NULL;
  int len = ASN1_STRING_to_UTF8((unsigned char**)&cn_str, cn_asn1);
  if (len < 0)
    return NULL;

  // cn_str now contains the CN as a null-terminated string that must be
  // freed via OPENSSL_free.  we need to ensure this is freeable via free()
  char* ret = cn_str ? strdup(cn_str) : NULL;
  OPENSSL_free(cn_str);

  return ret;
}

px_op_status on_handshake_complete(struct px_gemini_context* gemctx) {

  // TODO we need to do more checking to make sure that the verification process for this certificate
  // succeeded.  if the validity of the cert isn't verified in the verification callback then this may be a
  // completely bogus cert and we'll still use it as a client cert
  X509* x509 = SSL_get_peer_certificate(gemctx->conn.ssl);
  if (x509) {
    gemctx->client_cert.cert = x509;

    gemctx->client_cert.subject_name = get_cn_from_x509(x509);

    gemctx->client_cert.digests = hash_x509(x509);
    if (!gemctx->client_cert.digests.cert_digest)
      px_log_warn("%s: could not hash client certificate", gemctx->conn.peer_addr_str);
    else if (!gemctx->client_cert.digests.pubkey_digest)
      px_log_warn("%s: could not hash client pubkey", gemctx->conn.peer_addr_str);
  }

  if (gemctx->callbacks.on_handshake_complete)
    return gemctx->callbacks.on_handshake_complete(gemctx);
  return PX_OP_DONE;
}

static void queue_server_error_msg(struct px_gemini_context* gemctx) {
  static char const errmsg[] = "41 Internal server error\r\n";
  if (px_connection_queue_write(&gemctx->conn, errmsg, sizeof(errmsg) - 1) != sizeof(errmsg) - 1)
    px_log_warn("%s: could not queue internal server error header", gemctx->conn.peer_addr_str);
}

static void queue_bad_request_msg(struct px_gemini_context* gemctx) {
  static char const errmsg[] = "59 Bad request\r\n";
  if (px_connection_queue_write(&gemctx->conn, errmsg, sizeof(errmsg) - 1) != sizeof(errmsg) - 1)
    px_log_warn("%s: could not queue bad request header", gemctx->conn.peer_addr_str);
}

// @brief default action after receiving request data.
// @detail checks up to 1024 bytes for a valid URL
px_op_status px_gemini_context_try_parse_request(struct px_gemini_context* gemctx) {

  size_t scanlen = gemctx->request.reqbuf_sz < PX_GEMINI_REQUEST_SZ
                   ? gemctx->request.reqbuf_sz
                   : PX_GEMINI_REQUEST_SZ;

  uint8_t* crnl = memmem(gemctx->request.reqbuf, scanlen, "\r\n", 2);

  if (!crnl) {
    if (gemctx->request.reqbuf_sz >= PX_GEMINI_REQUEST_SZ) {
      px_log_info("%s: request is too long", gemctx->conn.peer_addr_str);
      return PX_OP_ERROR;
    }
    return PX_OP_RETRY;
  }


  size_t reqlen = crnl - gemctx->request.reqbuf;

  // first, make sure there aren't any null bytes.  those are never allowed and
  // the URL should be rejected outright if one is present
  uint8_t const* nullp = memchr(gemctx->request.reqbuf, '\0', reqlen);
  if (nullp != NULL) {
    px_log_info("%s: bad request, null byte in the request", gemctx->conn.peer_addr_str);
    px_url_reset(&gemctx->request.url);
    queue_bad_request_msg(gemctx);
    gemctx->stage = PX_GEMINI_RESPONSE_FLUSH;
    return PX_OP_DONE;
  }

  if (!px_url_from_buffer(&gemctx->request.url, gemctx->request.reqbuf, reqlen)) {
    px_log_info("%s: bad request, could not parse URL", gemctx->conn.peer_addr_str);
    px_url_reset(&gemctx->request.url);
    queue_bad_request_msg(gemctx);
    gemctx->stage = PX_GEMINI_RESPONSE_FLUSH;
    return PX_OP_DONE;
  }

  // normalize the request URL
  // since relative paths don't make any sense (we have no relative URL to
  // compare) then assume that the client provided a path relative to / i.e.
  // empty and null paths get translated to /, and relative paths get turned
  // absolute.  in either case this strips dots from the url string
  // TODO should we make it possible to have a null path explicitly?  there are
  // several assumptions in downstream code that will require some work if we
  // want to do that.
  struct px_path cleaned_path = px_path_from_str(gemctx->request.url.path);
  px_path_make_abs(&cleaned_path);
  char* cleaned_path_str = px_path_to_str(&cleaned_path);

  free(gemctx->request.url.path);
  gemctx->request.url.path = cleaned_path_str;

  // code needs to be able to rely on this being non-empty and/or an absolute path
  if (!gemctx->request.url.path) {
    px_log_error("%s: no memory for url cleaning", gemctx->conn.peer_addr_str);
    px_path_reset(&cleaned_path);
    return PX_OP_ERROR;
  }

  px_path_reset(&cleaned_path);

  return PX_OP_DONE;
}

static _Bool check_request_basic(struct px_gemini_context* gemctx) {
  if (gemctx->settings.client_cert_requirement != PX_GEMINI_CLIENT_CERT_OPTIONAL
      && !gemctx->client_cert.cert)
  {
    px_log_error("%s: client cert required but not provided", gemctx->conn.peer_addr_str);
    return false;
  }

  if (!px_gemini_is_valid_url(&gemctx->request.url, false /* TODO hook this*/)) {
    px_log_error("%s: invalid gemini request", gemctx->conn.peer_addr_str);
    return false;
  }

  return true;
}

// @brief action to take for receiving a (portion of a) gemini request
// @detail this reads in some amount of data from the socket, and buffers it in gemctx.request.reqbuf.  the
// user callback is then called (or, the default is used if not defined)
px_op_status on_request_receive(struct px_gemini_context* gemctx) {
  // sanity check
  px_log_assert(gemctx->request.reqbuf_sz <= sizeof(gemctx->request.reqbuf),
                "overflow calculating buffer size");

  size_t    buffer_used_sz  = gemctx->request.reqbuf_sz;
  uint8_t*  in_buffer       = gemctx->request.reqbuf + buffer_used_sz;
  size_t    in_buffer_sz    = sizeof(gemctx->request.reqbuf) - buffer_used_sz;

  px_op_status r = px_connection_recv(&gemctx->conn, in_buffer, &in_buffer_sz);
  if (r == PX_OP_ERROR) {
    px_log_info("%s: connection error while reading gemini request", gemctx->conn.peer_addr_str);
    return r;
  }

  gemctx->request.reqbuf_sz += in_buffer_sz;
  px_log_assert(gemctx->request.reqbuf_sz <= sizeof(gemctx->request.reqbuf), "buffer overflow");

  // user callback for this stage overrides other checks
  if (gemctx->callbacks.on_request_receive) {
    r = gemctx->callbacks.on_request_receive(gemctx, gemctx->request.reqbuf, gemctx->request.reqbuf_sz);
  } else {
    r = px_gemini_context_try_parse_request(gemctx);
    if (r == PX_OP_DONE) {
      if (!check_request_basic(gemctx)) {
        // the request is invalid?  queue up an error message and drop to flush mode
        px_connection_clear_outqueue(&gemctx->conn);
        px_log_error("%s: internal server error", gemctx->conn.peer_addr_str);
        queue_server_error_msg(gemctx);
        gemctx->stage = PX_GEMINI_RESPONSE_FLUSH;
        return PX_OP_DONE;
      }

      gemctx->request.valid = true;
      if (gemctx->request.valid && gemctx->callbacks.on_request_complete)
        r = gemctx->callbacks.on_request_complete(gemctx) ? PX_OP_DONE : PX_OP_ERROR;
    }
  }
  return r;
}

// @brief check if we have a valid header and update the gemctx header fields
// if so.  this is only a basic check ensuring that length and \r\n delimiters
// are present
static _Bool check_response_header_basic(struct px_gemini_context* gemctx) {
  struct px_queue* q = px_queue_front(&gemctx->conn.outq_head);
  px_log_assert(q != &gemctx->conn.outq_head, "this should never be called with an empty response queue");

  struct px_conn_buffer* buf = px_conn_buffer_from_queue(q);

  if (buf->data_sz < 4) // need at least <digit><digit>\r\n
    return false;

  // find the carriage return/newline
  uint8_t* crnl = (uint8_t*)memmem(buf->data, buf->data_sz, "\r\n", 2);
  if (crnl == NULL)
    return false;

  size_t respbuf_sz = crnl - buf->data;

  if (respbuf_sz > 1024) // 1024 byte max request size
    return false;

  // check that the status code is numeric
  if (!isdigit((unsigned char)buf->data[0]) || !isdigit((unsigned char)buf->data[1]))
    return false;

  // the header must have a space between the code and any metadata.
  if (crnl != &buf->data[2] && buf->data[2] != ' ')
    return false;

  return true;
}

static px_op_status on_response_transmit(struct px_gemini_context* gemctx) {

  // the first bytes through should be a gemini header.  ensure that a header has been sent
  // 1) must be in response-send mode
  // 2) must have the header check enabled
  // 3) total queued bytes should be 0 (i.e. nothing sent)
  // 4) the output queue must not be empty (i.e. there is something to check)
  if (gemctx->stage == PX_GEMINI_RESPONSE_SEND
      && !gemctx->settings.disable_header_check
      && gemctx->conn.stats.total_sslwrite == 0
      && !px_connection_outqueue_is_empty(&gemctx->conn))
  {
    gemctx->response.header_sent = check_response_header_basic(gemctx);
    if (!gemctx->response.header_sent) {
      px_connection_clear_outqueue(&gemctx->conn);
      px_log_error("%s: internal server error", gemctx->conn.peer_addr_str);
      queue_server_error_msg(gemctx);
      gemctx->stage = PX_GEMINI_RESPONSE_FLUSH;
    }
  }

  (void)px_connection_discard_readable(&gemctx->conn); // TODO warn?
  return px_connection_send(&gemctx->conn);
}

px_op_status on_response_send(struct px_gemini_context* gemctx) {
  // NOTE: data needs to be added externally via px_gemini_context_queue_data()
  px_op_status r = on_response_transmit(gemctx);
  return (gemctx->stage == PX_GEMINI_RESPONSE_SEND && r == PX_OP_DONE) ? PX_OP_RETRY : r;
}

px_op_status on_response_flush(struct px_gemini_context* gemctx) {
  if (gemctx->conn.stats.total_queued == 0) {
    px_log_error("%s: internal server error", gemctx->conn.peer_addr_str);
    queue_server_error_msg(gemctx);
  }

  // transmit whatever we have in the buffer until it's gone
  return on_response_transmit(gemctx);
}

px_op_status on_shutdown(struct px_gemini_context* gemctx) {
  // we may want to flush our buffers and then indicate that some error
  // occurred.  this is the place to do it
  if (gemctx->shutdown_ungracefully) {
    px_gemini_context_terminate_exchange(gemctx);
    return PX_OP_ERROR;
  }

  px_op_status r = px_connection_shutdown(&gemctx->conn);
  if (r != PX_OP_RETRY) {
    if (r == PX_OP_ERROR)
      // if we send the shutdown message then sometimes the client will just
      // close the connection (rudely!) and not send us a notification back.
      // if we did send the close-notify then assume that's what happened (or
      // more correctly, we check if the SSL layer thinks we sent the
      // close-notify - it may still be stuck in the outgoing buffer; TODO
      // check that case?)
      if (!px_connection_sent_shutdown(&gemctx->conn))
        px_log_info("%s: error on connection shutdown", gemctx->conn.peer_addr_str);
    px_gemini_context_terminate_exchange(gemctx);
  }
  return r;
}

void px_gemini_context_terminate_exchange(struct px_gemini_context* gemctx) {
  if (gemctx->conn.fd >= 0) {
    if (close(gemctx->conn.fd) != 0) {
      int e = errno;
      px_log_warn("%s: failed to close fd %d: %s", gemctx->conn.peer_addr_str, gemctx->conn.fd, strerror(e));
    }
    gemctx->conn.fd = -1;
  }
}

// map our stage to a function
static gemini_stage_function const stage_functions[] = {
  [PX_GEMINI_START]               = on_start_exchange,
  [PX_GEMINI_HANDSHAKE]           = on_handshake,
  [PX_GEMINI_REQUEST_RECEIVE]     = on_request_receive,
  [PX_GEMINI_RESPONSE_SEND]       = on_response_send,
  [PX_GEMINI_RESPONSE_FLUSH]      = on_response_flush,
  [PX_GEMINI_SHUTDOWN]            = on_shutdown,
  [PX_GEMINI_DONE]                = NULL,
  [PX_GEMINI_ERROR]               = NULL
};

px_op_status px_gemini_context_iterate_until(struct px_gemini_context* gemctx, px_gemini_stage stage) {
  if (gemctx->stage == PX_GEMINI_ERROR)
    return PX_OP_ERROR;

  size_t const n_ops = px_n_elements(stage_functions);
  px_log_assert(n_ops == (int)PX_GEMINI_NUM_STAGES, "programming error, bad function array");

  px_op_status r = PX_OP_DONE;
  while (gemctx->stage < stage && r == PX_OP_DONE) {
    int istage = (int)gemctx->stage;

    if (!stage_functions[istage])
      break;

    r = stage_functions[istage](gemctx);

    // retry or error breaks out of the loop
    if (r != PX_OP_DONE)
      break;

    // if the functioin returned DONE and did not update the gemini stage
    // itself then go to the next stage in the sequence
    if (istage == (int)gemctx->stage) {
      istage += 1;
      gemctx->stage = (px_gemini_stage)istage;
    }
  }

  if (r == PX_OP_ERROR) {
    //px_log_error("%s: error during gemini exchange", gemctx->conn.peer_addr_str);
    gemctx->stage = PX_GEMINI_ERROR;
    px_gemini_context_terminate_exchange(gemctx);
  }
  return r;
}

px_op_status px_gemini_context_iterate/*px_gemini_context_iterate*/(struct px_gemini_context* gemctx) {
  return px_gemini_context_iterate_until(gemctx, PX_GEMINI_DONE);
}

_Bool px_gemini_context_received_request(struct px_gemini_context const* gemctx) {
  return gemctx->request.valid;
}

_Bool px_gemini_context_header_sent(struct px_gemini_context const* gemctx) {
  return gemctx->response.header_sent;
}

_Bool px_gemini_context_needs_data(struct px_gemini_context const* gemctx) {
  return (gemctx->stage == PX_GEMINI_RESPONSE_SEND);
}

_Bool px_gemini_context_queue_data(struct px_gemini_context* gemctx, struct px_conn_buffer* buf) {
  if (!buf)
    return false;

  if (gemctx->stage != PX_GEMINI_RESPONSE_SEND) {
    px_conn_buffer_free(buf);
    return false;
  }

  if (!px_connection_queue_buffer(&gemctx->conn, buf)) {
    px_conn_buffer_free(buf);
    return false;
  }
  return true;
}

_Bool px_gemini_context_end_data(struct px_gemini_context* gemctx) {
  // calling this before the response-send phase is an error
  if (gemctx->stage == PX_GEMINI_ERROR || gemctx->stage < PX_GEMINI_RESPONSE_SEND) {
    gemctx->stage = PX_GEMINI_ERROR;
    return false;
  }

  // if we're in the response-send mode (or flush) then ending data
  // transmission is a normal operation, return true
  if (gemctx->stage == PX_GEMINI_RESPONSE_SEND) {
    gemctx->stage = PX_GEMINI_RESPONSE_FLUSH;
  } else if (gemctx->stage == PX_GEMINI_RESPONSE_FLUSH) {
    if (!px_connection_needs_send(&gemctx->conn)) {
      gemctx->stage = PX_GEMINI_SHUTDOWN;
      px_connection_shutdown(&gemctx->conn);
    }
  }

  return true;
}

#ifndef PX_NO_POLL

int px_gemini_context_get_net_poll_events(struct px_gemini_context const* gemctx) {
  // always response to readable data
  int events = px_connection_get_poll_events(&gemctx->conn);

  // if we're in shutdown and haven't sent the close_notify yet then we need to
  // be able to write
  if (gemctx->stage == PX_GEMINI_SHUTDOWN && !px_connection_sent_shutdown(&gemctx->conn))
    events |= POLLOUT;

  // TODO I don't think this is the best way to do this.  it seems like it'd be
  // easy to get stuck in a loop
  // if the context is sending data we may have sent all of the buffered data
  // over the wire but need to fetch more from the response callback
  if (gemctx->stage == PX_GEMINI_RESPONSE_SEND)
    events |= POLLOUT;

  return events;
}

struct pollfd px_gemini_context_get_network_pollfd(struct px_gemini_context const* gemctx) {
  struct pollfd ret = { .fd = -1 };
  if (!gemctx || gemctx->conn.fd < 0)
    return ret;
  ret.fd = gemctx->conn.fd;
  ret.events = px_gemini_context_get_net_poll_events(gemctx);
  return ret;
}

px_op_status  px_gemini_context_wait_for_network(struct px_gemini_context const* gemctx, int timeout_ms) {
  if (gemctx->conn.fd < 0)
    return PX_OP_ERROR;

  int events = px_gemini_context_get_net_poll_events(gemctx);
  if (events == 0)
    return PX_OP_RETRY;

  // if there's something in the SSL buffer then return immediately
  if (gemctx->conn.ssl && (SSL_pending(gemctx->conn.ssl) > 0 || BIO_pending(gemctx->conn.ssl_in) > 0))
    return PX_OP_DONE;

  struct pollfd pfd = px_gemini_context_get_network_pollfd(gemctx);

  int r = poll(&pfd, 1, timeout_ms);
  if (r < 0)
    return PX_OP_ERROR;
  if (r == 0) // timeout
    return PX_OP_RETRY;

  return ((pfd.revents & ~(POLLIN | POLLOUT)) == 0) ? PX_OP_DONE : PX_OP_RETRY;
}

#endif // PX_NO_POLL

_Bool px_gemini_context_pre_handshake(struct px_gemini_context* gemctx) {
  return gemctx->stage < PX_GEMINI_REQUEST_RECEIVE;
}

_Bool px_gemini_context_pre_request(struct px_gemini_context const* gemctx) {
  return gemctx->stage < PX_GEMINI_RESPONSE_SEND;
}

