#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <polluxd.h>
#include <polluxd_actions.h>
#include <polluxd_cgi.h>
#include <libpxd/px_log.h>
#include <libpxd/px_path.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

static char const* px_get_mimetype_by_extension(char const* filename) {
  if (!filename)
    return NULL;
  char const* dot = strrchr(filename, '.');
  if (dot != NULL) {
    struct mimetype_map_entry { char const* ext; char const* mimetype; };

    // these were taken from the Mozilla page about mimetypes
    // https://developer.mozilla.org/en-US/docs/Web/HTTP/MIME_types/Common_types
    static const struct mimetype_map_entry map[] = {
      { .ext = ".gmi", .mimetype = "text/gemini" },
      { .ext = ".aac", .mimetype = "audio/aac" },
      { .ext = ".abw", .mimetype = "application/x-abiword" },
      { .ext = ".apng", .mimetype = "image/apng" },
      { .ext = ".arc", .mimetype = "application/x-freearc" },
      { .ext = ".avif", .mimetype = "image/avif" },
      { .ext = ".avi", .mimetype = "video/x-msvideo" },
      { .ext = ".azw", .mimetype = "application/vnd.amazon.ebook" },
      { .ext = ".bin", .mimetype = "application/octet-stream" },
      { .ext = ".bmp", .mimetype = "image/bmp" },
      { .ext = ".bz", .mimetype = "application/x-bzip" },
      { .ext = ".bz2", .mimetype = "application/x-bzip2" },
      { .ext = ".cda", .mimetype = "application/x-cdf" },
      { .ext = ".csh", .mimetype = "application/x-csh" },
      { .ext = ".css", .mimetype = "text/css" },
      { .ext = ".csv", .mimetype = "text/csv" },
      { .ext = ".doc", .mimetype = "application/msword" },
      { .ext = ".docx", .mimetype = "application/vnd.openxmlformats-officedocument.wordprocessingml.document" },
      { .ext = ".eot", .mimetype = "application/vnd.ms-fontobject" },
      { .ext = ".epub", .mimetype = "application/epub+zip" },
      { .ext = ".gz", .mimetype = "application/x-gzip" },
      { .ext = ".gif", .mimetype = "image/gif" },
      { .ext = ".html", .mimetype = "text/html" },
      { .ext = ".ico", .mimetype = "image/vnd.microsoft.icon" },
      { .ext = ".ics", .mimetype = "text/calendar" },
      { .ext = ".jar", .mimetype = "application/java-archive" },
      { .ext = ".jpeg", .mimetype = "image/jpeg" },
      { .ext = ".jpg", .mimetype = "image/jpeg" },
      { .ext = ".js", .mimetype = "text/javascript" },
      { .ext = ".json", .mimetype = "application/json" },
      { .ext = ".jsonld", .mimetype = "application/ld+json" },
      { .ext = ".midi", .mimetype = "audio/x-midi" },
      { .ext = ".mjs", .mimetype = "text/javascript" },
      { .ext = ".mp3", .mimetype = "audio/mpeg" },
      { .ext = ".mp4", .mimetype = "video/mp4" },
      { .ext = ".mpeg", .mimetype = "video/mpeg" },
      { .ext = ".mpkg", .mimetype = "application/vnd.apple.installer+xml" },
      { .ext = ".odp", .mimetype = "application/vnd.oasis.opendocument.presentation" },
      { .ext = ".ods", .mimetype = "application/vnd.oasis.opendocument.spreadsheet" },
      { .ext = ".odt", .mimetype = "application/vnd.oasis.opendocument.text" },
      { .ext = ".oga", .mimetype = "audio/ogg" },
      { .ext = ".ogv", .mimetype = "video/ogg" },
      { .ext = ".ogx", .mimetype = "application/ogg" },
      { .ext = ".opus", .mimetype = "audio/ogg" },
      { .ext = ".otf", .mimetype = "font/otf" },
      { .ext = ".png", .mimetype = "image/png" },
      { .ext = ".pdf", .mimetype = "application/pdf" },
      { .ext = ".php", .mimetype = "application/x-httpd-php" },
      { .ext = ".ppt", .mimetype = "application/vnd.ms-powerpoint" },
      { .ext = ".pptx", .mimetype = "application/vnd.openxmlformats-officedocument.presentationml.presentation" },
      { .ext = ".rar", .mimetype = "application/vnd.rar" },
      { .ext = ".rtf", .mimetype = "application/rtf" },
      { .ext = ".sh", .mimetype = "application/x-sh" },
      { .ext = ".svg", .mimetype = "image/svg+xml" },
      { .ext = ".tar", .mimetype = "application/x-tar" },
      { .ext = ".tiff", .mimetype = "image/tiff" },
      { .ext = ".ts", .mimetype = "video/mp2t" },
      { .ext = ".ttf", .mimetype = "font/ttf" },
      { .ext = ".txt", .mimetype = "text/plain" },
      { .ext = ".vsd", .mimetype = "application/vnd.visio" },
      { .ext = ".wav", .mimetype = "audio/wav" },
      { .ext = ".weba", .mimetype = "audio/webm" },
      { .ext = ".webm", .mimetype = "video/webm" },
      { .ext = ".webp", .mimetype = "image/webp" },
      { .ext = ".woff", .mimetype = "font/woff" },
      { .ext = ".woff2", .mimetype = "font/woff2" },
      { .ext = ".xhtml", .mimetype = "application/xhtml+xml" },
      { .ext = ".xls", .mimetype = "application/vnd.ms-excel" },
      { .ext = ".xlsx", .mimetype = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet" },
      { .ext = ".xml", .mimetype = "application/xml" },
      { .ext = ".xul", .mimetype = "application/vnd.mozilla.xul+xml" },
      { .ext = ".zip", .mimetype = "application/x-zip-compressed" },
      { .ext = ".3gp", .mimetype = "audio/3gpp" },
      { .ext = ".3g2", .mimetype = "audio/3gpp2" },
      { .ext = ".7z", .mimetype = "application/x-7z-compressed" },
    };

    for (unsigned i = 0; i < px_n_elements(map); ++i) {
      if (strcmp(dot, map[i].ext) == 0)
        return map[i].mimetype;
    }
  }
  return "application/octet-stream";
}

char const* px_get_mimetype(char const* filename) {
  char const* mimetype = NULL;
  mimetype = px_get_mimetype_by_extension(filename);
  if (!mimetype)
    mimetype = "application/octet-stream";
  return mimetype;
}

int escape_nongraph(unsigned char c) {
  return !isgraph(c);
}

_Bool polluxd_deny_cb(struct polluxd_client* client) {
  char* encoded_path = client->routed_path_str
    ? px_url_encode_str(client->routed_path_str, escape_nongraph)
    : NULL;
  char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);

  px_log_info("%s: denying access to %s -> %s",
      client->gemctx.conn.peer_addr_str,
      encoded_url_path ? encoded_url_path : "(no url)",
      encoded_path ? encoded_path : "(no path)");
  free(encoded_url_path);
  free(encoded_path);

  struct px_conn_buffer *cb = px_conn_buffer_get();
  if (!cb)
    return false;

  static char const header[] = "50 Permission denied\r\n";
  const size_t n_cp = sizeof(header) - 1;
  memcpy(cb->data, header, n_cp);
  cb->data_sz = n_cp;

  (void)px_gemini_context_queue_data(&client->gemctx, cb);

  return false;
}

static _Bool dirfd_in_docroot(int dirfd, char const* docroot) {
  struct stat root_s;
  struct stat docroot_s;
  if (stat("/", &root_s) < 0 || stat(docroot, &docroot_s) < 0)
    return false;

  char dots_buf[1024] = ".";
  unsigned dots_len = 1;

  while (dots_len < sizeof(dots_buf) - 1) {
    struct stat s;
    if (fstatat(dirfd, dots_buf, &s, 0) < 0)
      return false;

    // is the ancestor the docroot?  if so then we can open the file!
    if (s.st_ino == docroot_s.st_ino
        && s.st_dev == docroot_s.st_dev)
    {
      return true;
    }

    // check if the ancestor is the filesystem root.  if so then the file is not
    // in the docroot and we are done
    if (s.st_ino == root_s.st_ino
        && s.st_dev == root_s.st_dev)
    {
      return false;
    }

    if (dots_len + sizeof("/..") > sizeof(dots_buf)) // out of space?
      return false;

    memcpy(dots_buf + dots_len, "/..", sizeof("/.."));
    dots_len += sizeof("/..") - 1; // exclude the null byte
  }
  return false;
}

static _Bool serve_dir(char const* dirname,
                       char const* index_file,
                       _Bool autoindex,
                       char const* docroot,
                       struct polluxd_client* client,
                       struct px_conn_buffer* cb,
                       const size_t cb_max_sz)
{
  int dirfd = open(dirname, O_RDONLY | O_DIRECTORY | O_NOFOLLOW | O_NONBLOCK | O_CLOEXEC);

  if (dirfd < 0) {
    if (errno != ENOTDIR) { // failed for some reason other than not being a directory? error response
      px_log_info("%s: could not open directory %s: %s",
          client->gemctx.conn.peer_addr_str, dirname, strerror(errno));
      goto ERR;
    }

    char const* url_path = client->gemctx.request.url.path;

    // find the last non-slash character and the back up until we find a
    // non-slash char
    char const* end_slash = strrchr(url_path, '/'); // get end of string
    while (end_slash != url_path && *(end_slash - 1) == '/') {
      --end_slash;
    }

    unsigned l = end_slash - url_path;
    px_log_assert(l <= strlen(url_path), "programmer made a math error");
    char* newpath = strndup(url_path, l);
    if (!newpath) {
      px_log_error("no memory");
      goto ERR;
    }

    // build a new px_url with the slash-stripped path.  we're abusing the API a
    // bit by doing this since the urls usually own their pointers, but
    // px_url_to_str doesn't modify the px_url so it doesn't care who owns the
    // internal pointers
    struct px_url redir = client->gemctx.request.url;
    redir.path = newpath;
    char* url_str = px_url_to_str(&redir);
    free(newpath);

    // url string built ok, send the redirect response
    if (url_str) {
      int r = snprintf((char*)cb->data, cb_max_sz, "30 %s\r\n", url_str);
      if (r > 0)
        cb->data_sz = (size_t)r;
      free(url_str);
    } // else send nothing so that we send an internal server error response
    return false;
  }

  if (!dirfd_in_docroot(dirfd, docroot)) {
    px_log_info("%s: resolved path of %s is not in the docroot %s",
        client->gemctx.conn.peer_addr_str, client->routed_path_str, docroot);
    goto ERR;
  }

  // we opened the directory.  now, see if we have an autoindex file
  if (index_file) {
    px_log_assert(strchr(index_file, '/') == NULL, "index file name should not have a slash");

    // open the index file (if possible) and don't allow the index file to be a
    // symlink, just for the sake of simplicity, otherwise we'd have to do a
    // whole additional check to make sure the index file is in the docroot
    // TODO maybe that extra check is worth it?
    int index_fd = openat(dirfd, index_file, O_RDONLY | O_NOFOLLOW | O_NONBLOCK | O_CLOEXEC);
    FILE* fstr = NULL;
    if (index_fd >= 0 && (fstr = fdopen(index_fd, "r")) != NULL) {
      close(dirfd);

      char* pathname = NULL;
      if (asprintf(&pathname, "%s/%s", dirname, index_file) < 0)
        pathname = NULL; // keep compiler warnings quiet
      client->fileinfo.pathname = pathname;
      client->fileinfo.file = fstr;

      char const* mimetype = px_get_mimetype(index_file);
      int r = snprintf((char*)cb->data, cb_max_sz, "20 %s\r\n", mimetype);
      if (r > 0)
        cb->data_sz = (size_t)r;
      return true;
    } else {
      if (index_fd >= 0)
        close(index_fd);
    }
  }

  if (autoindex) {
    DIR* d = fdopendir(dirfd);
    if (!d) {
      px_log_error("no memory");
      goto ERR;
    }

    client->fileinfo.dir = d;
    client->fileinfo.pathname = strdup(dirname);
    int r = snprintf((char*)cb->data, cb_max_sz, "20 text/gemini\r\n# Directory listing\n");
    if (r > 0)
      cb->data_sz = (size_t)r;
    return true;
  }

ERR:
  if (dirfd >= 0)
    close(dirfd);
  int r = snprintf((char*)cb->data, cb_max_sz, "51 Not found\r\n");
  if (r > 0)
    cb->data_sz = (size_t)r;
  return false;
}

static _Bool serve_file(char const* dirname,
                        char const* basename,
                        char const* docroot,
                        struct polluxd_client* client,
                        struct px_conn_buffer* cb,
                        const size_t cb_max_sz)
{

  // TODO if the directory is not readable then this will fail.  it may be nice
  // to be able to set directories to o-r o+x so that there are (kind of) guarantees the
  // contents can't be listed but individual files can still be accessible
  int dirfd = open(dirname, O_RDONLY | O_DIRECTORY | O_NOFOLLOW | O_NONBLOCK | O_CLOEXEC);
  char* pathname = NULL;

  if (dirfd < 0) {
    px_log_info("%s: could not open parent directory %s: %s",
        client->gemctx.conn.peer_addr_str, dirname, strerror(errno));
    goto ERR;
  }

  if (!dirfd_in_docroot(dirfd, docroot)) {
    px_log_info("%s: parent dir %s is not in the docroot %s",
        client->gemctx.conn.peer_addr_str, dirname, docroot);
    goto ERR;
  }

  struct stat s;
  if (fstatat(dirfd, basename, &s, 0) < 0) {
    px_log_info("%s: could not stat %s/%s: %s",
        client->gemctx.conn.peer_addr_str, dirname, basename, strerror(errno));
    goto ERR;
  }

  if ((s.st_mode & S_IFMT) == S_IFDIR) { // if the target is a directory then send a redirect
    close(dirfd);
    char* url_str = px_url_to_str(&client->gemctx.request.url);
    int r = snprintf((char*)cb->data, cb_max_sz, "30 %s/\r\n", url_str ? url_str : "");
    free(url_str);

    if (r > 0)
      cb->data_sz = (size_t)r;
    return false;
  }

  // we need this for socket/connect so go ahead and make it now
  if (asprintf(&pathname, "%s/%s", dirname, basename) < 0 || !pathname) {
    px_log_info("%s: asprintf(pathname) failed: %s", client->gemctx.conn.peer_addr_str, strerror(errno));
    goto ERR;
  }

  int ffd = -1;
  if ((s.st_mode & S_IFMT) == S_IFSOCK) { // if it's a socket we have to open it via socket/connect

    // try to open up a socket
    struct sockaddr_un un = { 0 };
    size_t pnlen = strlen(pathname);
    if (pnlen > sizeof(un.sun_path)) {
      px_log_info("%s: socket pathname too long", client->gemctx.conn.peer_addr_str);
      goto ERR;
    }

    un.sun_family = AF_UNIX;
    memcpy(un.sun_path, pathname, pnlen);

    ffd = socket(AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC | SOCK_NONBLOCK, 0);
    if (ffd < 0) {
      px_log_info("%s: could not create a socket for %s: %s",
                  client->gemctx.conn.peer_addr_str,
                  pathname,
                  strerror(errno));
      goto ERR;
    }

    // block waiting for an accept
    // this will succeed whether or not a server accepts the connection
    if (connect(ffd, (struct sockaddr const*)&un, sizeof(un)) < 0) {
      close(ffd);
      px_log_info("%s: could not connect to %s: %s",
                  client->gemctx.conn.peer_addr_str, pathname, strerror(errno));
      goto ERR;
    }
  } else {
    ffd = openat(dirfd, basename, O_RDONLY | O_NOFOLLOW | O_NONBLOCK | O_CLOEXEC);
    if (ffd < 0) {
      px_log_info("%s: could not open file %s/%s: %s",
                  client->gemctx.conn.peer_addr_str, dirname, basename, strerror(errno));
      goto ERR;
    }
  }

  close(dirfd);
  dirfd = -1;

  FILE* fstr = fdopen(ffd, "r");
  if (!fstr) {
    px_log_info("%s: couldn't open fstream: %s", client->gemctx.conn.peer_addr_str, strerror(errno));
    close(ffd);
    goto ERR;
  }

  client->fileinfo.pathname = pathname;
  client->fileinfo.file = fstr;

  char const* mimetype = px_get_mimetype(basename);
  int r = snprintf((char*)cb->data, cb_max_sz, "20 %s\r\n", mimetype);
  if (r > 0)
    cb->data_sz = (size_t)r;
  return true;

ERR:
  free(pathname);
  if (dirfd >= 0)
    close(dirfd);

  {
    int r = snprintf((char*)cb->data, cb_max_sz, "51 Not found\r\n");
    if (r > 0)
      cb->data_sz = (size_t)r;
  }
  return false;
}

static _Bool list_dir(struct polluxd_client* client, struct px_conn_buffer* cb) {
  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;
  size_t added_sz = 0;
  while (true) {
    // save the current position in case we run out of space
    long dpos = telldir(client->fileinfo.dir);
    if (dpos < 0) // error? assume we're done
      return false;

    errno = 0;
    struct dirent* de = readdir(client->fileinfo.dir);
    if (!de) {
      int e = errno;
      if (e == 0) // end of directory, done
        return false;

      if (e == EAGAIN || e == EWOULDBLOCK) { // blocking on dir info.  probably won't happen?
        client->file_ev.fd = dirfd(client->fileinfo.dir);
        client->file_ev.events = PX_EVENT_READ;
        client->file_ev.has_timeout = false;
        client->file_ev.priv = client;
        px_queue_extract(&client->file_ev.eventq);
        px_queue_append(&client->pxd->workq.events_head, &client->file_ev.eventq);
        return true;
      }
      return false;
    }

    _Bool is_dir = (de->d_type == DT_DIR);

    char sz_label_buf[32] = { 0 }; //< buffer for any file size label

    // if it's a file then get its size, stick i
    px_log_assert(client->route, "assumption violation");
    if (!client->route->autoindex_skip_file_sizes && !is_dir) {
      struct stat fs;
      if (fstatat(dirfd(client->fileinfo.dir), de->d_name, &fs, 0) >= 0) {
        unsigned long sz_val = fs.st_size >= 0 ? fs.st_size : 0;
        char sz_unit = 'B';
        if (sz_val >= 1024) {
          double sz_val_d = sz_val;
          if (sz_val >= 1024 * 1024) {
            if (sz_val >= 1024 * 1024 * 1024) {
              sz_val_d = sz_val / (1024*1024*1024);
              sz_unit = 'G';
            } else {
              sz_val_d = sz_val / (1024*1024);
              sz_unit = 'M';
            }
          } else {
            sz_val_d = sz_val / (1024);
            sz_unit = 'K';
          }
          (void)snprintf(&sz_label_buf[0], sizeof(sz_label_buf), " (%.02f %c)", sz_val_d, sz_unit);
        } else {
          (void)snprintf(&sz_label_buf[0], sizeof(sz_label_buf), " (%lu %c)", sz_val, sz_unit);
        }
      }
      sz_label_buf[sizeof(sz_label_buf) - 1] = '\0';
    }

    char* d_name_encoded = px_url_encode_str(de->d_name, escape_nongraph);
    int r = snprintf((char*)cb->data + added_sz, cb_max_sz - added_sz,
        "=> %s%s %s%s\n",
        d_name_encoded ? d_name_encoded : "?",
        is_dir ? "/" : "",
        de->d_name,
        sz_label_buf);
    free(d_name_encoded);

    if (r < 0) {
      return false;
    } else if ((size_t)r > (cb_max_sz - added_sz)) {
      // if we didn't even add one directory then we got a buffer that's too
      // small.  assume that's always going to happen and break out of adding
      // directories
      if (added_sz == 0)
        return false;

      // reset the dir position to before the current entry so it can be read
      // on the next iteration
      seekdir(client->fileinfo.dir, dpos);
      return true;
    }

    added_sz += r;
    cb->data_sz = added_sz;
  }
  return false;
}

static _Bool send_file(struct polluxd_client* client, struct px_conn_buffer* cb) {
  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;

  errno = 0;
  size_t r = fread(cb->data, 1, cb_max_sz, client->fileinfo.file);
  if (r == 0) {
    int e = errno;

    // end of file?  we're done
    if (feof(client->fileinfo.file)) {
      px_log_info("%s: done transmitting file", client->gemctx.conn.peer_addr_str);
      return false;
    }

    // error?  if we just don't have data available then it's ok
    if (e == EAGAIN || e == EWOULDBLOCK) {
      clearerr(client->fileinfo.file);
      client->file_ev.fd = fileno(client->fileinfo.file);
      client->file_ev.events = PX_EVENT_READ;
      client->file_ev.has_timeout = false;
      client->file_ev.priv = client;
      px_queue_extract(&client->file_ev.eventq);
      px_queue_append(&client->pxd->workq.events_head, &client->file_ev.eventq);
      return true;
    }
    return false;
  }

  // successful read
  cb->data_sz = (size_t)r < cb_max_sz ? r : cb_max_sz;
  return true;
}

_Bool polluxd_open_file_cb(struct polluxd_client* client) {

  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;

  struct px_conn_buffer* cb = px_conn_buffer_get();
  if (!cb) // no memory
    return false;

  px_log_assert(!client->fileinfo.file && !client->fileinfo.dir, "assumption violation");
  if (client->fileinfo.file || client->fileinfo.dir) { // this should never be true
    px_log_error("%s: internal server error", client->gemctx.conn.peer_addr_str);
    static char const errmsg[] = "41 Internal server error\r\n";
    memcpy(cb->data, errmsg, sizeof(errmsg) - 1);
    cb->data_sz = sizeof(errmsg) - 1;
    (void)px_gemini_context_queue_data(&client->gemctx, cb);
    return false;
  }

  px_log_assert(client->route, "assumption violation");
  if (!client->route) {
    px_log_error("%s: no route", client->gemctx.conn.peer_addr_str);
    return false;
  }

  _Bool ret = false;
  _Bool request_is_dir = px_path_str_is_dir(client->gemctx.request.url.path);

  // resolve all links in the routed path to make sure we know exactly what
  // filesystem entry we're dealing with
  char* resolved_path = realpath(client->routed_path_str, NULL);

  if (resolved_path) {
    px_log_info("%s: resolved path %s -> %s",
                client->gemctx.conn.peer_addr_str,
                client->routed_path_str,
                resolved_path);

    char const* docroot = client->route->docroot ? client->route->docroot : client->pxd->conf.docroot;
    if (!docroot || *docroot == '\0')
      docroot = "/";

    if (request_is_dir) {
      ret = serve_dir(resolved_path,
                      client->route->index_file,
                      client->route->autoindex,
                      docroot,
                      client,
                      cb,
                      cb_max_sz);
    } else {
      char const* dirname = ".";
      char const* basename = resolved_path;
      char* last_slash = strrchr(resolved_path, '/');
      if (last_slash) { // separate the dirname and basename
        *last_slash = '\0';
        basename = last_slash + 1;

        // handle the case that resolved_path is a file in the root
        dirname = (last_slash != resolved_path) ? resolved_path : "/";
      }
      ret = serve_file(dirname, basename, docroot, client, cb, cb_max_sz);
    }
  } else { // path resolution failed
    px_log_info("%s: could not resolve path for %s: %s",
                client->gemctx.conn.peer_addr_str, client->routed_path_str, strerror(errno));
    int r = snprintf((char*)cb->data, cb_max_sz, "51 Not found\r\n");
    if (r > 0)
      cb->data_sz = (size_t)r;
  }

  free(resolved_path);

  px_log_assert(((!client->fileinfo.file && !client->fileinfo.dir)
                || !client->fileinfo.file != !client->fileinfo.dir),
                "both a file and directory were opened!");

  {
    if (client->fileinfo.file || client->fileinfo.dir)
      px_log_info("%s: %s %s",
                  client->gemctx.conn.peer_addr_str,
                  client->fileinfo.file ? "serving file" : "listing directory",
                  client->fileinfo.pathname ? client->fileinfo.pathname : "(no path info)");
    else
      px_log_info("%s: nothing to serve", client->gemctx.conn.peer_addr_str);
  }

  if (ret)
    client->data_cb = polluxd_send_fileinfo_cb;

  if (cb->data_sz > 0) {
    if (!px_gemini_context_queue_data(&client->gemctx, cb))
      ret = false;
  } else {
    px_conn_buffer_free(cb);
  }

  return ret;
}

_Bool polluxd_send_fileinfo_cb(struct polluxd_client* client) {

  if (!client->fileinfo.file && !client->fileinfo.dir)
    // if no file or dir is open in the fileinfo then no data can be sent
    return false;

  struct px_conn_buffer* cb = px_conn_buffer_get();
  if (!cb) // no memory
    return false;

  _Bool ret = false;
  if (client->fileinfo.file) {
    ret = send_file(client, cb);
  } else if (client->fileinfo.dir) {
    ret = list_dir(client, cb);
  }

  if (cb->data_sz > 0) {
    if (!px_gemini_context_queue_data(&client->gemctx, cb))
      ret = false;
  } else {
    px_conn_buffer_free(cb);
  }

  return ret;
}

#define CGI_BUFFER_SZ 4096
// @brief pack the CGI environment into a buffer and send it to the cgi process
static _Bool send_cgi_environment(struct polluxd_client* client, int cgi_sock) {

  // simple protocol:
  // make a 4k buffer, write key=value\0 strings into it (which will be used as
  // environment vars), and send it over cgi_sock as one big block
  char* buf = (char*)calloc(1, CGI_BUFFER_SZ);

  if (!buf)
    return false;

  char* out = buf;
  size_t out_sz = CGI_BUFFER_SZ;
  size_t buf_sz = 0;

  struct kv { char const* key; char const* val; };

  char* url_str = px_url_to_str(&client->gemctx.request.url);
  const SSL_CIPHER *cipher = SSL_get_current_cipher(client->gemctx.conn.ssl);
  struct kv env[] = {
    { "SERVER_PROTOCOL", "gemini" },
    { "SERVER_SOFTWARE", "libpxd/polluxd" },
    { "GEMINI_URL", url_str },
    { "PATH_TRANSLATED", client->routed_path_str },
    { "SCRIPT_NAME", client->gemctx.request.url.path },
    { "QUERY_STRING", client->gemctx.request.url.query },
    { "HOSTNAME", client->gemctx.request.url.host },
    { "SERVER_NAME", client->gemctx.sni_hostname },
    { "SERVER_ADDR", client->gemctx.conn.host_addr_str },
    { "REMOTE_ADDR", client->gemctx.conn.peer_addr_str },
    { "TLS_CIPHER", SSL_CIPHER_get_name(cipher) },
    { "REMOTE_USER", client->gemctx.client_cert.subject_name },
    { "TLS_VERSION", SSL_CIPHER_get_version(cipher) },
    { "AUTH_TYPE", client->gemctx.client_cert.cert ? "CERTIFICATE" : "NONE" },
    { "TLS_CLIENT_HASH", client->gemctx.client_cert.digests.cert_digest },
    { "TLS_CLIENT_PUBKEY_HASH", client->gemctx.client_cert.digests.pubkey_digest }
  };

  for (unsigned i = 0, n = px_n_elements(env); i < n; ++i) {
    if (out_sz == 0)
      goto ERR;

    if (!env[i].val)
      continue;

    int r = snprintf(out, out_sz, "%s=%s", env[i].key, env[i].val ? env[i].val : "");
    if (r < 0 || (size_t)r >= out_sz)
      goto ERR;
    out += r;
    out_sz -= r;
    buf_sz += r + 1;
    if (out_sz > 0) {
      --out_sz;
      ++out;
    }
  }

  ssize_t tx = send(cgi_sock, buf, buf_sz, MSG_NOSIGNAL);
  if (tx < 0 || (size_t)tx != buf_sz) {
    int e = errno;
    px_log_error("%s: could not send cgi environment: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    goto ERR;
  }

  free(url_str);
  free(buf);
  return true;
ERR:
  free(url_str);
  free(buf);
  return false;
}

_Bool polluxd_cgi_cb(struct polluxd_client* client) {

  int cgi_sock = -1;
  if (!polluxd_recv_cgi_fd(client, client->route->cgi_helper_sock, &cgi_sock) || cgi_sock < 0) {
    char* encoded_path = px_url_encode_str(client->routed_path_str, escape_nongraph);
    char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);
    px_log_info("%s: requesting cgi script %s -> %s but could not receive cgi pipe",
                client->gemctx.conn.peer_addr_str,
                encoded_url_path ? encoded_url_path : "(none)",
                encoded_path ? encoded_path : "(none)");
    free(encoded_url_path);
    free(encoded_path);
    return false;
  }

  {
    char* encoded_path = px_url_encode_str(client->routed_path_str, escape_nongraph);
    char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);
    px_log_info("%s: requesting cgi script %s -> %s via fd %d",
                client->gemctx.conn.peer_addr_str,
                encoded_url_path ? encoded_url_path : "(none)",
                encoded_path ? encoded_path : "(none)",
                cgi_sock);
    free(encoded_url_path);
    free(encoded_path);
  }

  if (!send_cgi_environment(client, cgi_sock)) {
    close(cgi_sock);
    return false;
  }

  FILE* cgif = fdopen(cgi_sock, "r");
  if (!cgif) {
    int e = errno;
    px_log_warn("%s: could not fdopen fd %d: %s", client->gemctx.conn.peer_addr_str, cgi_sock, strerror(e));
    close(cgi_sock);
    return false;
  }
  client->fileinfo.file = cgif;
  client->data_cb = polluxd_send_fileinfo_cb;
  return true;
}
