#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <polluxd.h>
#include <polluxd_cgi.h>
#include <polluxd_config.h>
#include <polluxd_log.h>
#include <polluxd_actions.h>
#include <polluxd_route.h>
#include <libpxd/px_common.h>
#include <libpxd/px_connection.h>
#include <libpxd/px_event.h>
#include <libpxd/px_gemini.h>
#include <libpxd/px_listen.h>
#include <libpxd/px_log.h>
#include <libpxd/px_network.h>
#include <libpxd/px_path.h>
#include <libpxd/px_route.h>
#include <libpxd/px_url.h>

#include <openssl/err.h>

#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <netdb.h>
#include <netinet/in.h>
#include <poll.h>
#include <pwd.h>
#include <signal.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <unistd.h>

#define DEFAULT_GEMINI_PORT_STR "1965"
#define POLLUXD_ADDR INADDR_ANY
#define POLLUXD_DOCROOT "/var/gemini/"

// for stopping the polluxd server
int g_needs_stop = 0;

#define polluxd_client_from_queue(q) container_of(q, struct polluxd_client, clientq)

void signal_shutdown(int sig) { (void)sig; g_needs_stop = 1; }

void usage(FILE* f) {
  fprintf(f, "usage: polluxd -f <config_file> [-D] [-V]\n");
}

void help(FILE* f) {
  usage(f);
  fprintf(f, "  -f <config_file>  the configuration file to use\n");
  fprintf(f, "  -D                do not daemonize, stay in the foreground\n");
  fprintf(f, "  -V                print verison information and exit\n");
}

void print_version(void) {
  px_log_info("polluxd gemini server rev. 2, built with libpxd %s", px_get_libpxd_version());
}

void parse_options_and_config(int argc, char** argv, struct polluxd_context* pxd) {
  struct polluxd_config* conf = &pxd->conf;
  polluxd_config_reset(conf);

  int opt;
  char const* conffile = NULL;
  char const* argstr = "f:hDV";
  opterr = 0;
  while ((opt = getopt(argc, argv, argstr)) != -1) {
    switch (opt) {
      case 'f' :
        conffile = optarg;
        break;
      case 'h' :
        help(stdout);
        exit(0);
      case 'D' :
        conf->foreground = true;
        break;
      case 'V' :
        print_version();
        exit(0);
      default:
        px_log_error("bad option '%c'", (char)optopt);
        usage(stderr);
        exit(-1);
    }
  }

  if (!conffile) {
    px_log_error("no configuration file provided");
    usage(stderr);
    exit(-1);
  }

  FILE* f = fopen(conffile, "r");
  if (!f) {
    int e = errno;
    px_log_error("could not open config file %s: %s", conffile, strerror(e));
    exit(-1);
  }

  if (!polluxd_config_read_file(conf, f, conffile)) {
    px_log_error("could not update configuration from file");
    fclose(f);
    exit(-1);
  }
  fclose(f);
}

static void daemonize(char const* docroot) {
  pid_t p = fork();
  if (p < 0)
    px_log_fatal("could not daemonize, error on fork: %s", strerror(errno));

  if (p > 0) {
    px_log_info("forked, parent exitting");
    exit(0);
  }

  if (chdir(docroot ? docroot : "/") < 0)
    px_log_fatal("could not chdir to %s: %s", docroot ? docroot : "/", strerror(errno));

  pid_t sid = setsid();
  if (sid < 0)
    px_log_fatal("could not setsid: %s", strerror(errno));
}

static void openbsd_security_stuff(char const* docroot, _Bool need_cgi) {
#if defined(__OpenBSD__)
  {
    if (unveil(docroot, need_cgi ? "rwx" : "rw") != 0
        || unveil(NULL, NULL) != 0)
    {
      px_log_fatal("could not unveil: %s", strerror(errno));
    }
  }
#else
  // keep the compiler quiet about unused vars
  (void)docroot;
  (void)need_cgi;
#endif
}

void polluxd_close_client(struct polluxd_client* client) {
  px_log_info("%s: closing connection.  received %lu bytes, sent %lu bytes",
              client->gemctx.conn.peer_addr_str,
              client->gemctx.conn.stats.total_recv,
              client->gemctx.conn.stats.total_send);

  // no more work to do, either error or finished.  tear down the client
  px_queue_extract(&client->clientq);
  --client->pxd->n_clients; // decrease the active connection count

  // make sure the net and file events aren't in any queues (they should not be)
  //px_log_assert(px_queue_is_singleton(&client->net_ev.eventq), "net event should not be in a queue");
  //px_log_assert(px_queue_is_singleton(&client->file_ev.eventq), "file event should not be in a queue");
  px_queue_extract(&client->net_ev.eventq);
  px_queue_extract(&client->file_ev.eventq);
  px_queue_extract(&client->timeout_ev.eventq);

  px_gemini_context_reset(&client->gemctx);
  if (client->fileinfo.file) {
    fclose(client->fileinfo.file);
    client->fileinfo.file = NULL;
  }

  if (client->fileinfo.dir) {
    closedir(client->fileinfo.dir);
    client->fileinfo.dir = NULL;
  }

  free(client->fileinfo.pathname);
  client->fileinfo.pathname = NULL;

  px_path_reset(&client->routed_path);
  free(client->routed_path_str);

  free(client);
}

static void close_on_timeout(void* data) {
  struct polluxd_client* client = (struct polluxd_client*)data;
  px_log_info("%s: connection timeout", client->gemctx.conn.peer_addr_str);
  polluxd_close_client(client);
}

static void handle_socket_error(int fd, int revents, void* data) {
  // skip over read/write events, we only need to handle errors
  if ((revents & ~(PX_EVENT_READ | PX_EVENT_WRITE)) == 0)
    return;

  struct polluxd_client* client = (struct polluxd_client*)data;
  px_log_assert(fd == client->gemctx.conn.fd, "data value set incorrectly");

  if (!px_gemini_context_did_server_shutdown(&client->gemctx)) {
    char const* modifier = "";
    if (revents & PX_EVENT_HUP)
      if (px_gemini_context_did_client_shutdown(&client->gemctx))
        px_log_info("%s: client shutdown", client->gemctx.conn.peer_addr_str);
      else
        px_log_info("%s: client disconnect", client->gemctx.conn.peer_addr_str);
    else
      px_log_info("%s: %ssocket error", client->gemctx.conn.peer_addr_str, modifier);
  }
  polluxd_close_client(client);
}

void polluxd_drop_privileges(
    char const* drop_user,
    char const* drop_group,
    char const* chroot_dir,
    char const* docroot,
    _Bool       need_cgi)
{

  int target_uid = -1;
  int target_gid = -1;

#if defined(__OpenBSD__)
  if (!docroot) // used for the unveil() later
    px_log_fatal("no docroot set");
#endif // defined(__OpenBSD__)

  if (drop_user) {
    struct passwd* pwent = getpwnam(drop_user);
    if (!pwent) {
      int e = errno;
      px_log_fatal("pid %d: could not find pw entry for user %s: %s", (int)getpid(), drop_user, strerror(e));
    }
    target_uid = pwent->pw_uid;
  }

  if (drop_group) {
    struct group* grent = getgrnam(drop_group);
    if (!grent) {
      int e = errno;
      px_log_fatal("pid %d: could not find pw entry for group %s: %s", (int)getpid(), drop_group, strerror(e));
    }
    target_gid = grent->gr_gid;
  }

  if (chroot_dir) {
    int r = chroot(chroot_dir);
    if (r != 0)
      px_log_fatal("pid %d: could not chroot to %s: %s", (int)getpid(), chroot_dir, strerror(errno));

    px_log_info("chrooted to %s", chroot_dir);

    if (chdir("/") < 0) {
      int e = errno;
      px_log_fatal("could not chdir to /: %s", strerror(e));
    }
  }

#if defined(__NetBSD__)
#define SETUID_FUNC(uid) setreuid(uid, uid)
#define SETGID_FUNC(gid) setregid(gid, gid)
#else
#define SETUID_FUNC(uid) setresuid(uid, uid, uid)
#define SETGID_FUNC(gid) setresgid(gid, gid, gid)
#endif
  if (SETGID_FUNC(target_gid) != 0) {
    int e = errno;
    px_log_fatal("pid %d: could not setresgid to %d: %s", (int)getpid(), (int)target_gid, strerror(e));
  }

  if (SETUID_FUNC(target_uid) != 0) {
    int e = errno;
    px_log_fatal("pid %d: could not setresuid to %d: %s", (int)getpid(), (int)target_uid, strerror(e));
  }

  int running_uid = geteuid();
  int running_gid = getegid();
  px_log_info("pid %d running as user:group %d:%d", (int)getpid(), (int)running_uid, (int)running_gid);

  openbsd_security_stuff(docroot, need_cgi);
}

// translate a string into a enum value representing an action to take for a request
enum polluxd_action str_to_action(char const* str) {

  struct action_map_entry { char const* action_name; enum polluxd_action action; };
  static struct action_map_entry const g_action_map[] = {
    { .action_name = "deny",  .action = POLLUXD_DO_DENY },
    { .action_name = "file",  .action = POLLUXD_DO_FILE },
    { .action_name = "cgi",   .action = POLLUXD_DO_CGI  }
    //{ .action_name = "pipe",  .action = POLLUXD_DO_PIPE }
  };

  // translate the action function string to an actual function
  static const unsigned n_avail_actions = sizeof(g_action_map) / sizeof(g_action_map[0]);
  for (unsigned i = 0; i < n_avail_actions; ++i) {
    if (strcmp(g_action_map[i].action_name, str) == 0)
      return g_action_map[i].action;
  }
  return POLLUXD_N_ACTIONS;
}

// verify the route information and hook it into something we can call to
// initialize our gemini response
static void polluxd_setup_routes(struct polluxd_context* pxd) {

  size_t const n_avail_routes = px_n_elements(pxd->route_data);

  // the first route_data entry in pxd is the default route, which denies
  // access.  we don't give access to stuff unless someone means it
  pxd->route_data[0].action = POLLUXD_DO_DENY;
  unsigned n_used_routes = 1; // the default route takes up the first route_data slot

  // iterate though each location in the configuration structure and add them
  // to the route info
  struct polluxd_config_location* loc = pxd->conf.locations;
  while (loc) {
    // some sanity checks
    px_log_assert(loc->path && loc->path[0] != '\0',
                  "null/empty paths are not allowed, and this should have been checked already");
    px_log_assert(n_used_routes < n_avail_routes, "too many routes, adjust the max number of route_data");
    if (n_used_routes >= n_avail_routes)
      px_log_fatal("too many locations configured.  maximum is %u", (unsigned)n_avail_routes);

    // we're adding a route, consume a polluxd_route_data
    unsigned idx = n_used_routes++;

    // if we can't find an action function then someone typo'd something.  abort
    enum polluxd_action action = str_to_action(loc->action);
    if (action < 0 || action >= POLLUXD_N_ACTIONS)
      px_log_fatal("unrecognized action \"%s\" for location %s", loc->action, loc->path);

    struct px_path prefix = px_path_from_str(loc->docroot ? loc->docroot : pxd->conf.docroot);
    {
      struct px_path loc_prefix = px_path_from_str(loc->prefix);
      prefix = px_path_concat_consume(&prefix, &loc_prefix);
      px_path_make_abs(&prefix);
    }

    // get the name of the file to use for an index (probably index.gmi).
    // directories don't get served (file not found) if an index file isn't
    // specified/found and autoindexing isn't enabled
    char* index_file = NULL;
    if (loc->index_file)
      index_file = strdup(loc->index_file);
    else if (pxd->conf.default_index_file)
      index_file = strdup(pxd->conf.default_index_file);

    if ((loc->index_file || pxd->conf.default_index_file) && !index_file)
      px_log_fatal("no memory");

    char* allowed_cert_file = NULL;
    if (loc->allowed_cert_file) {
      allowed_cert_file = strdup(loc->allowed_cert_file);
      if (!allowed_cert_file)
        px_log_fatal("no memory");
    }

    char* docroot = NULL;
    if (loc->docroot) {
      docroot = strdup(loc->docroot);
      if (!docroot)
        px_log_fatal("no memory");
    }

    char* cgi_chroot = NULL;
    char* cgi_user = NULL;
    char* cgi_group = NULL;
    char* cgi_script = NULL;
    if (action == POLLUXD_DO_CGI) {
      if (loc->cgi_user) {
        cgi_user = strdup(loc->cgi_user);
        if (!cgi_user)
          px_log_fatal("no memory");
      }

      if (loc->cgi_group) {
        cgi_group = strdup(loc->cgi_group);
        if (!cgi_group)
          px_log_fatal("no memory");
      }

      if (loc->cgi_chroot) {
        cgi_chroot = strdup(loc->cgi_chroot);
        if (!cgi_chroot)
          px_log_fatal("no memory");
      }

      if (loc->cgi_script) {
        cgi_script = strdup(loc->cgi_script);
        if (!cgi_script)
          px_log_fatal("no memory");
      }
    }

    pxd->route_data[idx].action = action;
    pxd->route_data[idx].strip = loc->strip;
    pxd->route_data[idx].prefix = prefix;
    pxd->route_data[idx].autoindex = loc->autoindex;
    pxd->route_data[idx].autoindex_skip_file_sizes = loc->autoindex_skip_file_sizes;
    pxd->route_data[idx].index_file = index_file;
    pxd->route_data[idx].require_cert = loc->require_cert;
    pxd->route_data[idx].allowed_cert_file = allowed_cert_file;
    pxd->route_data[idx].cgi_user = cgi_user;
    pxd->route_data[idx].cgi_group = cgi_group;
    pxd->route_data[idx].cgi_chroot = cgi_chroot;
    pxd->route_data[idx].docroot = docroot;
    pxd->route_data[idx].cgi_script = cgi_script;

    struct px_route_node* node = px_route_add(&pxd->routes, loc->path, false);
    if (!node)
      px_log_fatal("could not add route node %s", loc->path ? loc->path : "(null)");

    node->priv = &pxd->route_data[idx];
    loc = loc->next;
  }
}

// @brief called when a listening socket has an action (i.e. can accept a connection)
void accept_connection(int fd, int revents, void* data) {
  struct polluxd_context* pxd = (struct polluxd_context*)data;

  if ((revents & ~PX_EVENT_READ) != 0) {
    px_log_error("error condition on listening socket");
    goto FAIL;
  }
  errno = 0;

  int sock = accept4(fd, NULL, 0, SOCK_CLOEXEC);
  if (sock < 0) {
    int e = errno;
    // no more connections available
    if (e == EAGAIN || e == EWOULDBLOCK)
      return;

    // these errors are ok, temporary issues or some problem accepting a single connection
    if (e == EPROTO
        || e == ENOMEM
        || e == ENOBUFS
        || e == EMFILE
        || e == ENFILE
        || e == ECONNABORTED)
    {
      px_log_error("could not accept connection: %s", strerror(e));
    } else {
      goto FAIL;
    }
  } else { // other errors are problems with the listening socket
    px_log_info("accepted connection on socket %d from listening socket %d", sock, fd);
  }

  struct polluxd_client* client = (struct polluxd_client*)calloc(1, sizeof(*client));
  if (!client) {
    px_log_error("could not accept connection, no memory");
    close(sock);
    return;
  }
  px_queue_init(&client->clientq);
  client->pxd = pxd;
  px_event_init(&client->net_ev);
  px_event_init(&client->file_ev);
  px_event_init(&client->timeout_ev);

  // initialize the gemini context first and setup the connection to save some work if it fails
  px_gemini_context_init(&client->gemctx);

  // TODO callbacks, settings
  if (!px_gemini_context_setup_exchange(&client->gemctx, sock, NULL, NULL, NULL)) {
    px_log_error("could not set up gemini exchange");
    px_gemini_context_reset(&client->gemctx);
    close(sock);
    free(client);
    return;
  }

  ++client->pxd->n_clients; // decrease the active connection count
  px_queue_append(&pxd->client_head, &client->clientq);

  return;
FAIL:
  {
    int e = errno;
    px_log_error("error on accept, stopping server: %s", strerror(e));
  }
  pxd->stop = true;
  close(fd);
  pxd->listen.fd = -1;
  pxd->accept_ev.fd = -1;
}

static int pollfd_events_to_px_events(int events) {
  int r = (events & POLLIN) ? PX_EVENT_READ : PX_EVENT_NONE;
  r |= (events & POLLOUT) ? PX_EVENT_WRITE : PX_EVENT_NONE;
  return r;
}

static _Bool null_data_cb(struct polluxd_client* client) {
  struct px_conn_buffer* cb = px_conn_buffer_get();
  if (!cb) // no memory
    return false;

  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;

  int r = snprintf((char*)cb->data, cb_max_sz, "51 Not found\r\n");
  if (r > 0)
    cb->data_sz = r;
  (void)px_gemini_context_queue_data(&client->gemctx, cb);

  return false;
}

int escape_nongraph(unsigned char c); // defined in polluxd_actions.c

static void route_request(struct polluxd_client* client) {

  struct px_path real_path = px_path_from_str(client->gemctx.request.url.path);
  px_log_assert(px_path_is_abs(&real_path), "assumption violation, gemini step didn't clean path");
  px_path_make_abs(&real_path);

  struct px_route_node* node = px_route_find_path(&client->pxd->routes, &real_path);
  while (node && node->parent) {
    if (!node->priv) {
      node = node->parent;
      continue;
    }
    break;
  }

  if (node && node->priv)
    client->route = (struct polluxd_route_data*)node->priv;
  else
    client->route = &client->pxd->route_data[0];

  // combine the route prefix with the stripped request path to get the route path
  struct px_path const* prefix = &client->route->prefix;
  struct px_path stripped = px_path_strip_leading(&real_path, client->route->strip);
  client->routed_path = px_path_concat_consume1(prefix, &stripped);
  px_path_make_abs(&client->routed_path);
  px_path_make_file(&client->routed_path);
  px_path_reset(&real_path);

  // make sure it's absolute
  client->routed_path_str = px_path_to_str(&client->routed_path);

  {
    char* encoded_path = px_url_encode_str(client->routed_path_str, escape_nongraph);
    char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);
    px_log_info("%s: requesting %s -> %s",
        client->gemctx.conn.peer_addr_str,
        encoded_url_path ? encoded_url_path : "(none)",
        encoded_path ? encoded_path : "(none)");
    free(encoded_url_path);
    free(encoded_path);
  }

  if (client->routed_path_str) {
    switch (client->route->action) {
      case POLLUXD_DO_DENY :
        client->data_cb = polluxd_deny_cb;
        break;
      case POLLUXD_DO_FILE :
        client->data_cb = polluxd_open_file_cb;
        break;
      case POLLUXD_DO_CGI :
        client->data_cb = polluxd_cgi_cb;
        break;
        //      case POLLUXD_DO_PIPE :
        //        client->data_cb = NULL; // TODO
        //        break;
      default:
        break;
    }
  } else {
    px_log_warn("no memory");
  }

  if (!client->data_cb)
    client->data_cb = null_data_cb;
}

// TODO check_client_cert_file requires that the certificate whitelist is
// inside of the chroot of the server.  if we can get away with it, we should
// move this out of the chroot
// TODO could we use unveil to set read-only access to this path?
static _Bool check_client_cert_file(char const* allowed_cert_file, char const* cert_digest) {
  if (!allowed_cert_file || !cert_digest || allowed_cert_file[0] == '\0' || cert_digest[0] == '\0')
    return false;

  FILE* certf = fopen(allowed_cert_file, "r");
  if (!certf)
    return false;

  char* line = NULL;
  size_t alloc_sz = 0;
  while (true) {
    ssize_t line_sz = getline(&line, &alloc_sz, certf);
    if (line_sz < 0)
      break;
    if (line_sz == 0) // what?
      continue;
    if (line[line_sz - 1] == '\n')
      line[line_sz - 1] = '\0';

    // allow for comments in the certificate file i.e. # comment at the end of line
    // given a line: xxxxxxx     # comment
    // this next block will put a nul character immediately after the final x
    char* comment_start = strchr(line, '#');
    if (comment_start) {
      while (comment_start != line) {
        char* prev_char = comment_start - 1;
        if (*prev_char != '#' && *prev_char != ' ' && *prev_char != '\t')
          break;
        comment_start = prev_char;
      }
      *comment_start = '\0';
    }

    // convert ascii hex digits to lowercase
    for (char* citr = line; citr && *citr != '\0'; ++citr) {
      char c = *citr;
      if (c >= 'A' && c <= 'F')
        *citr = 'a' + (c - 'A');
    }

    if (strcmp(line, cert_digest) == 0)
      goto SUCCESS;
  }
  free(line);
  fclose(certf);
  return false;

SUCCESS:
  free(line);
  fclose(certf);
  return true;
}

static _Bool check_client_certificate(struct polluxd_client* client) {
  if (client->route && (client->route->require_cert || client->route->allowed_cert_file)) {
    if (!client->gemctx.client_cert.cert || !client->gemctx.client_cert.digests.cert_digest) {
      px_log_info("%s: no client cert", client->gemctx.conn.peer_addr_str);
      return false;
    }

    if (client->route->allowed_cert_file) {
      if (check_client_cert_file(client->route->allowed_cert_file,
                                 client->gemctx.client_cert.digests.cert_digest))
      {
        char const* cert_hash = client->gemctx.client_cert.digests.cert_digest
                                ? client->gemctx.client_cert.digests.cert_digest
                                : "(none)";
        px_log_info("%s: authorized certificate %s", client->gemctx.conn.peer_addr_str, cert_hash);
        return true;
      }

      {
        char const* cert_hash = client->gemctx.client_cert.digests.cert_digest
                                ? client->gemctx.client_cert.digests.cert_digest
                                : "(none)";
        px_log_info("%s: unauthorized certificate %s", client->gemctx.conn.peer_addr_str, cert_hash);
      }
      return false;
    }
  }
  return true;
}

px_op_status client_iterate(struct polluxd_client* client) {

  // if the context needs data (i.e. received a valid request) then we need to supply it
  if (px_gemini_context_needs_data(&client->gemctx)) {
    // remove the timeout event from the workq
    px_queue_extract(&client->timeout_ev.eventq);

    // if we have tried to send data and it's been more than 10s since the last
    // successful transfer then we need to close the connection.  something is
    // wrong.
    time_t send_tdiff = client->gemctx.conn.stats.last_attempted_send.tv_sec
      - client->gemctx.conn.stats.last_send.tv_sec;

    if (send_tdiff >= 10) {
      px_log_info("%s: timeout while waiting to send data", client->gemctx.conn.peer_addr_str);
      return PX_OP_DONE;
    }

    // if the request hasn't been routed then do so
    if (!client->route) {
      route_request(client); // if this fails then client->data_cb will be NULL

      // make sure we have a route
      if (client->route) {
        if (!check_client_certificate(client)) {
          if (!client->gemctx.client_cert.cert) {
            px_connection_queue_string(&client->gemctx.conn, "60 Certificate required\r\n");
          } else {
            px_connection_queue_string(&client->gemctx.conn, "61 Certificate not authorized\r\n");
          }
          client->data_cb = NULL;
        }
      } else {
        client->data_cb = NULL;
      }
    }

    _Bool more_data = false;

    // make sure we have a route
    if (client->route) {
      // if there's a callback then grab data from there.  if no callback gets
      // defined then we assume we're ending the response
      if (client->data_cb)
        more_data = client->data_cb(client);
    }

    // update the context's stage if we're done with adding data (and
    // force-terminate the connection if something went wrong)
    if (!more_data) {
      if (!px_gemini_context_end_data(&client->gemctx)) {
        px_log_error("%s error while terminating response", client->gemctx.conn.peer_addr_str);
        return PX_OP_DONE;
      }
    }

    return px_gemini_context_iterate(&client->gemctx);
  }

  // if we're not waiting for data then give us a 10s timeout.  that's almost
  // certainly long enough to either receive a request or process some data
  // if the client is honest
  if (px_queue_is_singleton(&client->timeout_ev.eventq)) {
    client->timeout_ev.priv = (void*)client;
    client->timeout_ev.has_timeout = true;
    client->timeout_ev.on_timeout = close_on_timeout;
    px_event_set_timeout(&client->timeout_ev, 10000);
    px_queue_append(&client->pxd->workq.events_head, &client->timeout_ev.eventq);
  }

  return px_gemini_context_iterate(&client->gemctx);
}

static void handle_client(struct polluxd_client* client, int* timeout) {
  px_op_status r = client_iterate(client); // step through the gemini stages and add data if needed
  if (r != PX_OP_RETRY) { // !retry implies done or error => close client
    if (r == PX_OP_ERROR) {
      if (client->gemctx.stage >= PX_GEMINI_SHUTDOWN)
        px_log_info("%s: unclean shutdown", client->gemctx.conn.peer_addr_str);
      else
        px_log_info("%s: error on connection", client->gemctx.conn.peer_addr_str);
    }
    polluxd_close_client(client);
    return;
  }

  // we want to wait on the network as long as the connection is not closed
  // if the SSL layer is waiting on the network then set up the net event to
  // break out of the wait on read/write events.  we don't actually have to do
  // anything since we'll loop back through the server_loop when that happens.
  // if the connection is not waiting on the network then still throw the
  // net_ev on the event queue to wait for error conditions and hangups
  client->net_ev.fd = client->gemctx.conn.fd;
  client->net_ev.on_event = handle_socket_error; // read/write is left to gemini context
  client->net_ev.has_timeout = false; // wait until something happens
  client->net_ev.priv = (void*)client;

  if (px_gemini_context_waiting_on_network(&client->gemctx)) {
    struct pollfd pfd = px_gemini_context_get_network_pollfd(&client->gemctx);
    int events = pollfd_events_to_px_events(pfd.events);
    client->net_ev.events = events;
  } else {
    client->net_ev.events = 0; // only wait for socket error conditions

    // if we're here the network isn't blocking us, and if we have more stuff
    // to send then we can do it immediately.  alternatively, if we're not
    // waiting on a file then we can grab more data or try to open one
    // immediately
    if (px_connection_needs_send(&client->gemctx.conn)
        || client->file_ev.fd < 0
        || px_queue_is_singleton(&client->file_ev.eventq))
    {
      *timeout = 0;
    }
  }

  // if the net event isn't already on our workq then put it on there
  if (px_queue_is_singleton(&client->net_ev.eventq))
    px_queue_append(&client->pxd->workq.events_head, &client->net_ev.eventq);


}

void server_loop(struct polluxd_context* pxd) {
  while (!g_needs_stop && !pxd->stop) {

    int timeout = -1;

    // add the accept event to the queue (if it's not there already, which it shouldn't be)
    if (px_queue_is_singleton(&pxd->accept_ev.eventq) && pxd->accept_ev.fd >= 0)
      px_queue_append(&pxd->workq.events_head, &pxd->accept_ev.eventq);

    // go through each client and do some work
    struct px_queue *q = pxd->client_head.next;
    while (!pxd->stop && q != &pxd->client_head) {
      struct polluxd_client* client = polluxd_client_from_queue(q);
      q = q->next; // grab the next client so that we can remove the current client if it gets closed
      handle_client(client, &timeout);
    }

    if (g_needs_stop || pxd->stop)
      continue;

    if (!px_workq_is_empty(&pxd->workq))
      if (!px_workq_once(&pxd->workq, timeout))
        break;
  }

  px_workq_reset(&pxd->workq);
}

static void polluxd_context_init(struct polluxd_context* pxd) {
  *pxd = (struct polluxd_context) { 0 };
   pxd->listen = (struct px_listen_socket) { .fd = -1 };
   px_event_init(&pxd->accept_ev);
   px_queue_init(&pxd->client_head);
   px_workq_init(&pxd->workq);
   for (unsigned i = 0, n = px_n_elements(pxd->route_data); i < n; ++i) {
      polluxd_route_data_init(&pxd->route_data[i]);
   }
}

static void polluxd_context_reset(struct polluxd_context* pxd) {

  // any clients that are left over should be forcibly terminated.  we're shutting down
  while (!px_queue_is_singleton(&pxd->client_head)) {
    struct px_queue* q = px_queue_pop_front(&pxd->client_head);
    struct polluxd_client* client = polluxd_client_from_queue(q);
    polluxd_close_client(client);
  }

  polluxd_config_reset(&pxd->conf);
  px_route_reset(&pxd->routes);
  for (unsigned i = 0; i < sizeof(pxd->route_data) / sizeof(pxd->route_data[0]); ++i)
    polluxd_route_data_reset(&pxd->route_data[i]);
  if (pxd->listen.fd >= 0)
    close(pxd->listen.fd);
}

int main(int argc, char* argv[]) {

  struct polluxd_context pxd = { 0 };
  polluxd_context_init(&pxd);
  px_route_init(&pxd.routes);
  px_queue_init(&pxd.client_head);

  // some default settings
  pxd.conf.port = strdup(DEFAULT_GEMINI_PORT_STR);
  pxd.conf.docroot = strdup("/");

  // parse command line options (basically just determines the config file and
  // sets up pxd.conf)
  parse_options_and_config(argc, argv, &pxd);

  // open up our logfile (hopefully securely)
  polluxd_setup_logfile(&pxd.conf);

  px_log_info("starting polluxd (libpxd %s) on pid = %d", px_get_libpxd_version(), (int)getpid());

  // convert the polluxd config into a libpxd route structure so that it can be
  // used by the gemini request callback
  polluxd_setup_routes(&pxd);

  if (!polluxd_launch_cgi_helpers(&pxd))
    px_log_warn("unable to launch CGI helper processes");

  // TODO do less of this as root/before drop_privileges
  SSL_CTX* polluxd_ssl_ctx = px_net_ssl_ctx();
  if (!polluxd_ssl_ctx)
    px_log_fatal("could not create SSL_CTX object, aborting");

  if (!px_net_use_ssl_cert(polluxd_ssl_ctx, pxd.conf.cert_file, pxd.conf.key_file))
    px_log_fatal("could not use configured certificate/key %s/%s, aborting",
                  pxd.conf.cert_file, pxd.conf.key_file);

  int listen_fd = px_bind(pxd.conf.listen_addr, pxd.conf.host, pxd.conf.port);
  if (listen_fd < 0)
    px_log_fatal("could not listen, exitting");

  pxd.listen.fd = listen_fd;
  px_listen_socket_update_addrstr(&pxd.listen);
  px_log_info("listening on socket %s", pxd.listen.host_str);

  // do any securing we need to
  polluxd_drop_privileges(pxd.conf.drop_user,
                          pxd.conf.drop_group,
                          pxd.conf.chroot_dir,
                          pxd.conf.docroot ? pxd.conf.docroot : "/",
                          false);

  // clean up conf.docroot with realpath
  {
    char const* target_path = pxd.conf.docroot ? pxd.conf.docroot : "/";
    char* real_docroot = realpath(target_path, NULL);
    if (!real_docroot)
      px_log_fatal("could not get the realpath of docroot %s: %s", target_path, strerror(errno));
    free(pxd.conf.docroot);
    pxd.conf.docroot = real_docroot;
  }

  // daemonize just before starting up the polluxd server processes
  if (!pxd.conf.foreground)
    daemonize(pxd.conf.docroot);

  // set up our signal dispositions so we can gracefully shutdown and not leave
  // child processes around
  // TODO: we should do some work to ensure that cgi processes get cleaned up
  // here, but it is not obvious how to do that in a portable way yet
  if (sigaction(SIGINT, &(struct sigaction) { .sa_handler = signal_shutdown }, NULL) != 0
      || sigaction(SIGHUP, &(struct sigaction) { .sa_handler = signal_shutdown }, NULL) != 0
      || sigaction(SIGTERM, &(struct sigaction) { .sa_handler = signal_shutdown }, NULL) != 0
      || signal(SIGCHLD, SIG_IGN) != 0)
  {
    px_log_error("could not set up signal handlers");
    return -1;
  }

  pxd.accept_ev = (struct px_event) { .eventq = PX_QUEUE_INITIALIZER(&pxd.accept_ev.eventq),
                                      .fd = pxd.listen.fd,
                                      .events = PX_EVENT_READ,
                                      .has_timeout = false,
                                      .on_event = accept_connection,
                                      .priv = (void*)&pxd };
  px_queue_init(&pxd.accept_ev.eventq);

  px_gemini_context_set_default_sslctx(polluxd_ssl_ctx);

  // do the real work
  server_loop(&pxd);

  // clean up whatever resources are laying around
  px_gemini_context_set_default_sslctx(NULL);
  polluxd_context_reset(&pxd);
  SSL_CTX_free(polluxd_ssl_ctx);

  // sync outputs so we make sure we have everything
  fflush(stderr);
  fflush(stdout);

  char cmdbuf[128];
#if defined(__linux__)
  snprintf(cmdbuf, sizeof(cmdbuf), "ls -l /proc/%d/fd/\n", (int)getpid());
  if (system(cmdbuf) < 0)
    (void)0; // quiet the compiler, this is only for debugging
#elif defined(__OpenBSD__)
  snprintf(cmdbuf, sizeof(cmdbuf), "fstat -p %d\n", (int)getpid());
  system(cmdbuf);
#else
  (void)cmdbuf;
#endif

  return 0;
}
