#ifndef PX_GEMINI_CTX_H__
#define PX_GEMINI_CTX_H__

#include <libpxd/px_common.h>
#include <libpxd/px_connection.h>
#include <libpxd/px_event.h>
#include <libpxd/px_gemini.h>
#include <libpxd/px_url.h>

#ifndef PX_NO_POLL
#include <poll.h>
#endif // PX_NO_POLL

struct px_gemini_context;

// @brief numerical representations of the state of a gemini exchange
enum px_gemini_stage_ {
  PX_GEMINI_START = 0,
  PX_GEMINI_HANDSHAKE,
  PX_GEMINI_REQUEST_RECEIVE,
  PX_GEMINI_RESPONSE_SEND,
  PX_GEMINI_RESPONSE_FLUSH,
  PX_GEMINI_SHUTDOWN,
  PX_GEMINI_DONE,
  PX_GEMINI_ERROR,
  PX_GEMINI_NUM_STAGES
};
typedef enum px_gemini_stage_ px_gemini_stage;

// @brief annotates how and when TLS client certificates are required
enum px_gemini_client_cert_requirement_ {
  PX_GEMINI_CLIENT_CERT_OPTIONAL = 0, ///< client certs are optional

  PX_GEMINI_CLIENT_CERT_REQUIRE_SSL,  ///< client certs are required at the SSL level
                                      ///< if a client cert is not received then the TLS connection
                                      ///< is terminated immediately with a SSL_AD_CERTIFICATE_REQUIRED
                                      ///< TLS alert message

  PX_GEMINI_CLIENT_CERT_REQUIRE_GEMINI, ///< client certs are required at the Gemini level
                                        ///< this will wait for a request to be made before sending
                                        ///< back a proper gemini response (61 Not Authorized)
};
typedef enum px_gemini_client_cert_requirement_ px_gemini_client_cert_requirement;

struct px_x509_digests {
  char* cert_digest;
  char* pubkey_digest;
};

struct px_gemini_callbacks {
  // @brief called on TCP connection-established, pre-SSL handshake
  px_op_status  (*on_start)    (struct px_gemini_context* gemctx);

  // @brief called when SNI data is available.  use this function to select the certificate etc.
  // @detail this is done midway through the SSL handshake, so limited data will be available.
  // namely client cert/keys will not be there.  @return true if the connection should proceed,
  // false if the connection should be terminated
  _Bool         (*on_sni)       (struct px_gemini_context* gemctx, char const* sni_name);

  // @brief called when the SSL handshake is complete
  px_op_status  (*on_handshake_complete)     (struct px_gemini_context* gemctx);

  // @brief called when a portion of the gemini request has been received.
  // this overrides any default checks that are done, including checking for a
  // proper request.
  // @param buf a buffer containing data read from the TLS connection
  // @param buf_sz the size of buf in bytes
  // @return PX_OP_DONE if the request phase is considered complete
  // PX_OP_RETRY if more request data should be expected
  // PX_OP_ERROR if an error occurs that requires non-gracefully terminating the exchange
  px_op_status  (*on_request_receive) (struct px_gemini_context* gemctx, uint8_t*  buf, size_t buf_sz);

  // @brief only called when on_request_receive isn't set.  this is called when
  // a complete request (\r\n terminated, less than 1024 bytes, valid url) is
  // received.  if a client cert has been sent it will be available in
  // gemctx->cert_digests
  // @return true if the exchange should continue, false otherwise
  _Bool   (*on_request_complete) (struct px_gemini_context* gemctx);
};

// @brief these control the behavior of the gemini exchange
struct px_gemini_settings {
  // @brief if an sni hostname must be received.  an empty string, as long as it it sent in the
  // TLS extension, will satisfy this requirement
  _Bool sni_required;

  // @brief hostname sent by sni must match the request hostname
  // NOTE: if sni_required is false, a null SNI may still match an empty request hostname
  _Bool sni_must_match_request;

  // @brief selects if/at which level client certificates are enforced
  // see the comments on px_gemini_client_cert_requirement for more info
  px_gemini_client_cert_requirement client_cert_requirement;

  // @brief if true, do not enforce checking that a header was sent at the beginning of a
  // response.
  // @detail it this is not set and the first response data that is sent is not formatted as a
  // proper gemini header the data will be overwritten with an error response and the connection
  // terminated.
  _Bool disable_header_check;

  // @brief disables checking whether a particular status code sent in the response needs a body
  // or not.
  // @detail by default the gemini context will check to see if a header response code returns a
  // body or not (i.e. is one of the 2x codes) and will truncate the response to only the header
  // line if so.  set this field to true to disable that check
  _Bool disable_body_requirement_check;

  // @brief do not send a close-notify on connection shutdown.
  // @detail set this to indicate that data was truncated or some other non-fatal error on the
  // server end requires an indication to the client that the data it received is not complete
  _Bool skip_close_notify;
};

struct px_gemini_context {
  // @brief the stage of the gemini.  gets mapped to a function in px_gemini_context_iterate
  px_gemini_stage             stage;

  struct px_connection        conn;     // connection state and buffers (ssl, socket, etc)

  // when a new connection is made, tls negotiated, and gemini request
  // received, this function is called to determine what the rest of the
  // behavior is.  this is where the request's url should be parsed, and the
  // proper behaviors loaded into gemctx->actions
  // for devs implementing callbacks, this is called when
  // px_gemini->stage == PX_GEMINI_RESP_SETUP (i.e. no header has been written at all)
  struct px_gemini_callbacks  callbacks;  // operations at the different stages of the gemini request

  // these modify different requirements and behaviors of the gemini exchange
  struct px_gemini_settings settings;

  // @brief the cached value of the hostname received from the TLS SNI extension (if any)
  char* sni_hostname;

  // @brief information about the client certificate
  struct {
    // @brief the client certificate itself
    X509*                   cert;

    // @brief the subject field of the cert
    char*                   subject_name;

    // @brief sha256 hashes of the certificate and public key
    struct px_x509_digests  digests;
  } client_cert;

  // if true, a properly-formatted header was sent.  if disable_header_check is false, this will be
  // set to true once on_response_send has placed a proper header response in the response buffer.  it
  // may then also be used by on_response_send to determine if a response has already been added
  struct {
    _Bool         valid; ///< whether the request is a valid request
    struct px_url url;  ///< url received from client
    size_t        reqbuf_sz;
    uint8_t       reqbuf[PX_TLS_MAX_PLAINTEXT_SZ]; // 16k so that we can read a whole tls packet
  } request;

  struct {
    _Bool   header_sent;
    uint8_t header_status;
    char    header_meta[PX_GEMINI_HEADER_SZ + 1];
  } response;

  _Bool shutdown_ungracefully;

  // @brief implementer-specific private data for use by callbacks
  void* priv;
};

void          px_gemini_context_set_default_sslctx(SSL_CTX* def);

void          px_gemini_context_init(struct px_gemini_context* gemctx);
void          px_gemini_context_reset(struct px_gemini_context* gemctx);
_Bool         px_gemini_context_setup_exchange(struct px_gemini_context* gemctx,
                                               int fd,
                                               SSL_CTX* sslctx,
                                               struct px_gemini_callbacks* callbacks,
                                               struct px_gemini_settings* settings);

px_op_status  px_gemini_context_iterate(struct px_gemini_context* gemctx);

_Bool         px_gemini_context_pre_request(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_handshake_complete(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_received_request(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_header_sent(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_needs_data(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_queue_data(struct px_gemini_context* gemctx, struct px_conn_buffer* buf);
_Bool         px_gemini_context_end_data(struct px_gemini_context* gemctx);
_Bool         px_gemini_context_data_transmitted(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_is_finished(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_in_error(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_did_partial_shutdown(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_did_server_shutdown(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_did_client_shutdown(struct px_gemini_context const* gemctx);
_Bool         px_gemini_context_did_full_shutdown(struct px_gemini_context const* gemctx);

_Bool         px_gemini_context_waiting_on_network(struct px_gemini_context const* gemctx);

#ifndef PX_NO_POLL
int           px_gemini_context_get_net_poll_events(struct px_gemini_context const* gemctx);
struct pollfd px_gemini_context_get_network_pollfd(struct px_gemini_context const* gemctx);
px_op_status  px_gemini_context_wait_for_network(struct px_gemini_context const* gemctx, int timeout_ms);
#endif // PX_NO_POLL

// for stalling or terminating the connection
void px_gemini_context_terminate_exchange(struct px_gemini_context* gemctx);
_Bool px_gemini_context_buffer_to_header(struct px_gemini_context* gemctx, uint8_t const* buf, size_t buf_sz);

// some default callbacks so that they can be used
px_op_status on_request_receive_default_cb(struct px_gemini_context* gemctx);

#endif // PX_GEMINI_CTX_H__
