#include <libpxd/px_common.h>
#include <libpxd/px_route.h>
#include <libpxd/px_log.h>
#include <libpxd/px_url.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

_Bool urls_equal(struct px_url const* u1, struct px_url const* u2) {
  struct cmp {
    char const* str1;
    char const* str2;
  };

  // list of strings to compare
  struct cmp fields[] = {
     { u1->scheme, u2->scheme },
     { u1->userinfo, u2->userinfo },
     { u1->host, u2->host },
     { u1->port, u2->port },
     { u1->path, u2->path },
     { u1->query, u2->query },
     { u1->fragment, u2->fragment }
  };

  for (size_t i = 0; i < px_n_elements(fields); ++i) {
    if (!fields[i].str1 != !fields[i].str2 // make sure they're both null or both valued
        || (fields[i].str1 && strcmp(fields[i].str1, fields[i].str2) != 0))
      return false;
  }

  return true;
}

int main(void) {

  struct test_case {
    char const*   request;
    _Bool         valid_url;
    struct px_url expected;
  };

  struct test_case test_cases[] = {
    {
      "/absolute/path",
      true,
      { .path = "/absolute/path" }
    },
    {
      "gemini:/absolute/path",
      true,
      { .scheme = "gemini", .path = "/absolute/path" }
    },
    {
      "gemini://host/absolute/path",
      true,
      { .scheme = "gemini", .host = "host", .path = "/absolute/path" }
    },
    {
      "//host/absolute/path",
      true,
      { .host = "host", .path = "/absolute/path" }
    },
    {
      "gemini:relative/path",
      true,
      { .scheme = "gemini", .path = "relative/path" }
    },
    {
      "gemini:relative/path",
      true,
      { .scheme = "gemini", .path = "relative/path" }
    },
    {
      "gemini://user:password@host:1234/absolute/path?query",
      true,
      { .scheme = "gemini",
        .userinfo = "user:password",
        .host = "host",
        .port = "1234",
        .path = "/absolute/path",
        .query = "query" }
    },
    {
      "gemini:///absolute/path",
      true,
      { .scheme = "gemini", .host = "", .path = "/absolute/path" }
    },
    {
      "//:1234/absolute/path?",
      true,
      { .host = "", .port = "1234", .path = "/absolute/path", .query = "" }
    },
    { // real example
      "gemini://mysite.ai/robots.txt?robot=true&uri=gemini%3A%2F%2Ffreeshell.de%2Ftags%2Ffaq.gmi",
      true,
      { .scheme = "gemini",
        .host = "mysite.ai",
        .path = "/robots.txt",
        .query = "robot=true&uri=gemini%3A%2F%2Ffreeshell.de%2Ftags%2Ffaq.gmi" }
    }
  };

  unsigned n_passed = 0;
  for (unsigned test_num = 0; test_num < px_n_elements(test_cases); ++test_num) {
    struct px_url test_url = { 0 };
    struct test_case* test = &test_cases[test_num];
    _Bool did_parse = px_url_from_buffer(&test_url, (uint8_t const*)test->request, strlen(test->request));
    if (did_parse != test->valid_url) {
      fprintf(stderr, "fail: test case %u: %s: %s\n", test_num, test->request,
              test->valid_url ? "failed to parse but should have" : "parsed when it should not");
      continue;
    }

    if (urls_equal(&test_url, &test->expected)) {
      fprintf(stderr, "pass: test case %u: %s\n", test_num, test->request);
      ++n_passed;
    } else {
      fprintf(stderr, "fail: test case %u: %s: mis-parsed url\n", test_num, test->request);
    }
  }

  fprintf(stderr, "passed %u/%u tests\n", n_passed, (unsigned)px_n_elements(test_cases));

  // some of the original tests
  //char const* test_urls[] = {
  //  "/basic/path/to/something?query#fragment",
  //  "gemini://somehost.com/path/to/something?query#fragment",
  //  "gemini:///path/to/something?query#fragment",
  //  "gemini://user@somehost.com/path/to/something?query1=this&query2=that",
  //  "gemini://mysite.ai/robots.txt?robot=true&uri=gemini%3A%2F%2Ffreeshell.de%2Ftags%2Ffaq.gmi",
  //  "gemini://some_user@some.host:1234/some/path?some&query#some_fragment",
  //  "gemini://some_user@some.host:1234/some/path?some&query=/unencoded/slash#some_fragment/unencoded/slash"
  //};

}
