#include <libpxd/px_path.h>
#include <libpxd/px_log.h>
#include <string.h>

struct test_path {
  char const* path;
  size_t      n_components;
  char const* cleaned_path;
};

struct test_concat {
  char const* lpath;
  char const* rpath;
  unsigned    expected_components;
  char const* expected_concat;
};

void test_concat(void) {
  struct test_concat tests[] = {
    { "/lpath", "/rpath", 3, "/lpath/rpath" }
  };
  unsigned n_tests = sizeof(tests) / sizeof(tests[0]);

  for (unsigned i = 0; i < n_tests; ++i) {
    struct px_path l = px_path_from_str(tests[i].lpath);
    struct px_path r = px_path_from_str(tests[i].rpath);
    struct px_path c = px_path_concat(&l, &r);
    char* s = px_path_to_str(&c);
    if (s) {
      if (strcmp(s, tests[i].expected_concat) != 0) {
        px_log_error("concat differs: %s != %s\n", s, tests[i].expected_concat);
      } else if (c.components_sz != tests[i].expected_components) {
        px_log_error("component size differs: %lu != %u\n", c.components_sz, tests[i].expected_components);
      } else {
        px_log_info("concatenated path %s is correct\n", s);
      }
    } else {
      px_log_info("could not create string for %u\n", i);
    }
    free(s);
    px_path_reset(&c);
    px_path_reset(&l);
    px_path_reset(&r);
  }
}

int main(void) {
  struct test_path test_paths[] = {
    { "/", 0, "/" },
    { "//", 0, "/" },
    { "/test", 1, "/test" },
    { "/testdir/", 1, "/testdir/" },
    { "//test", 1, "/test" },
    { "//testdir//", 1, "/testdir/" },
    { "/test/dir/////", 2, "/test/dir/" },
    { "/some/path/..", 1, "/some/" },
    { "/some/path/../", 1, "/some/" },
    { "/some/path/.", 2, "/some/path/" },
    { "/some/path/../..", 0, "/" },
    { "/some/path/../../..", 0, "/" },
    { "some/path/../..", 0, "" },
    { "/../some/path/../..", 0, "/" },
    { "/../some/path/../../..", 0, "/" },
    { "../some/path/../..", 0, "" },
    { "/../some/path", 2, "/some/path" },
    { "/../some/path/", 2, "/some/path/" },
    { "/../some/path/./", 2, "/some/path/" },
    { "/some/../path", 1, "/path" },
    { "../some/path/", 2, "some/path/" },
    { "../some/../path/", 1, "path/" },
    { "../path/..", 0, "" }
  };

  int ret = 0;

  for (size_t i = 0, n = sizeof(test_paths) / sizeof(test_paths[0]); i < n; ++i) {
    struct test_path* test = &test_paths[i];
    struct px_path path = px_path_from_str(test->path);

    px_log_info("checking path %lu: %s", i, test->path ? test->path : "(null)");

    if (path.components_sz != test->n_components) {
      px_log_warn("  path size fail: %s: %lu != expected %lu",
                  test->path, path.components_sz, test->n_components);
      size_t nc = path.components_sz;
      for (size_t x = 0; x < nc; ++x) {
        px_log_warn("  component %lu: %s", x, path.components[x]);
      }
      goto NEXT;
    }

    size_t n_components = test->n_components;
    for (size_t j = 0; j < n_components; ++j) {
      if (path.components[j] == NULL) {
        px_log_warn("  path fail: %s component %lu: %s",
            test->path ? test->path : "(null)", j, path.components[j] ? path.components[j] : "(null)");
        ret = -1;
        goto NEXT;
      }
    }

    char* path_str = px_path_to_str(&path);
    if (path_str) {
      px_log_info("  reconstituted path string: %s", path_str);

      if (!test->cleaned_path) {
        px_log_warn("  non-null reconstituted path when null expected");
        free(path_str);
        goto NEXT;
      }

      int r = strcmp(path_str, test->cleaned_path);
      free(path_str);
      if (r != 0) {
        px_log_warn("  mismatch with expected path, should be: %s", test->cleaned_path);
        goto NEXT;
      }
    } else {
      if (test->cleaned_path) {
        px_log_warn("  null reconstituted path when non-null expected");
        goto NEXT;
      }
    }

    px_log_info("  success");
NEXT:
    px_path_reset(&path);
  }

  return ret;
}
