#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <polluxd_config.h>
#include <libpxd/px_log.h>
#include <stdbool.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include <errno.h>

#define MAX_LINE_SZ 1024

// remove leading whitespace from strp
// modifies strp
static void lstrip(char* strp) {
  // if strp doesn't start with a space character then there's nothing to do
  if (!strp || *strp == '\0' || !isspace((unsigned char)*strp))
    return;

  // find the first non-space character
  char* str_itr = strp;
  while (*str_itr != '\0' && isspace((unsigned char)*str_itr))
    ++str_itr;

  while (*str_itr != '\0') {
    *strp = *str_itr;
    ++strp;
    ++str_itr;
  }
  *strp = '\0';
}

// remove trailing whitespace from strp
// modifies strp
static void rstrip(char* strp) {
  if (!strp || *strp == '\0')
    return;

  char* end = strchr(strp, '\0');
  if (!end) // shouldn't happen?
    return;

  --end; // guaranteed ok, we have at least 1 character in strp
  while (isspace((unsigned char)*end)) {
    *end = '\0';
    if (end == strp)
      break;
    --end;
  }
}

// remove leading and trailing whitespace from strp
// modifies strp
static void lrstrip(char* strp) {
  lstrip(strp);
  rstrip(strp);
}

struct conf_parse_context {
  char const*                     conf_filename;
  struct polluxd_config*          conf;
  struct polluxd_config_location* loc;
};

// key-value pair
struct kv {
  char const* key;      // name of the configuration key
  void*       val_dest; // location to store the value
  _Bool       (*convert)(char const* val, void* dest);
};

static _Bool copy_string(char const* val, void* dest) {
  if (!val || *val == '\0' || !dest)
    return false;

  char* dup = strdup(val);
  if (!dup)
    return false;
  *(char**)dest = dup;
  return true;
}

static _Bool to_unsigned(char const* val, void* dest) {
  if (!val || *val == '\0' || !dest)
    return false;
  char* endp = NULL;
  long r = strtol(val, &endp, 10);
  if (!endp || *endp != '\0')
    return false;
  if (r < 0 || r > UINT_MAX)
    return false;
  *(unsigned*)dest = (unsigned)r;
  return true;
}

static _Bool to_bool(char const* val, void* dest) {
  if (!val || *val == '\0' || !dest)
    return false;

  if (strcasecmp(val, "yes") == 0
      || strcasecmp(val, "true") == 0
      || strcmp(val, "1") == 0)
  {
    *(_Bool*)dest = true;
    return true;
  } else if (strcasecmp(val, "no") == 0
      || strcasecmp(val, "false") == 0
      || strcmp(val, "0") == 0)
  {
    *(_Bool*)dest = false;
    return true;
  }
  return false;

}

static _Bool parse_kv_line(int lineno, char* line, struct kv const* kv_opts,
                            unsigned const kv_opts_len, struct conf_parse_context* parse_ctx)
{
  char const* conf_filename = parse_ctx->conf_filename ? parse_ctx->conf_filename : "(no filename)";

  char* eq = strchr(line, '=');
  if (eq == NULL) {
    px_log_error("%s line %d: expected key-value pair but got: %s", conf_filename, lineno, line);
    return false;
  }

  *eq = '\0';
  char* key = line;
  char* val = (eq + 1);
  lrstrip(key);
  lrstrip(val);
  if (*val == '\0') {
    px_log_error("%s line %d: key %s has an empty value", conf_filename, lineno, key);
    return false;
  }

  // assign the key
  for (unsigned i = 0; i < kv_opts_len; ++i) {
    if (strcmp(kv_opts[i].key, key) == 0) {
      // free any old value if we hold strings in this option
      if (kv_opts[i].convert == copy_string && *(char**)kv_opts[i].val_dest)
        free(*(char**)kv_opts[i].val_dest);

      _Bool did_convert = kv_opts[i].convert(val, (void*)kv_opts[i].val_dest);
      if (!did_convert) {
        px_log_error("%s line %d: failed to convert value %s for %s", conf_filename, lineno, val, key);
        return false;
      }
      return true;
    }
  }
  px_log_error("%s line %d: unrecognized key %s", conf_filename, lineno, key);
  return false;
}

// return 1 on success, 0 on parse failure
static _Bool parse_location_line(int lineno, char* line, struct conf_parse_context* parse_ctx) {

  if (strcmp(line, "}") == 0) { // end of location block
    struct polluxd_config* conf = parse_ctx->conf;

    // sanity checks, make sure necessary components are specified
    if (!parse_ctx->loc->action) {
      char const* conf_filename = parse_ctx->conf_filename ? parse_ctx->conf_filename : "(no filename)";
      px_log_error("%s line %d: location block is missing an action", conf_filename, lineno);
      return false;
    }

    // add the new location block to the conf's list.  new_loc becomes the new
    // first link in the list so that later entries are prioritized over
    // earlier ones
    parse_ctx->loc->next = conf->locations; // will be NULL if it's the first location
    conf->locations = parse_ctx->loc;
    parse_ctx->loc = NULL; // no longer in a location block
    return true;
  }

  // any other block in a location is a key-value pair
  struct polluxd_config_location* loc = parse_ctx->loc;
  const struct kv loc_opts[] = {
    { "action",             &loc->action,             copy_string },
    { "cgi_user",           &loc->cgi_user,           copy_string },
    { "cgi_group",          &loc->cgi_group,          copy_string },
    { "cgi_chroot",         &loc->cgi_chroot,         copy_string },
    { "cgi_script",         &loc->cgi_script,         copy_string },
    { "docroot",            &loc->docroot,            copy_string },
    { "strip",              &loc->strip,              to_unsigned },
    { "prefix",             &loc->prefix,             copy_string },
    { "require_cert",       &loc->require_cert,       to_bool },
    { "allowed_cert_file",  &loc->allowed_cert_file,  copy_string },
    { "index_file",         &loc->prefix,             copy_string },
    { "autoindex",          &loc->autoindex,          to_bool },
    { "autoindex_skip_file_sizes", &loc->autoindex_skip_file_sizes, to_bool }
  };

  return parse_kv_line(lineno, line, loc_opts, sizeof(loc_opts) / sizeof(loc_opts[0]), parse_ctx);
}

static _Bool parse_new_location_line(int lineno, char* line, struct conf_parse_context* parse_ctx) {
  // guaranteed that 'line' begins with 'location'
  char const* conf_filename = parse_ctx->conf_filename ? parse_ctx->conf_filename : "(no filename)";

  size_t line_len = strlen(line);
  px_log_assert(line_len >= strlen("location "), "programming error, line should start with 'location'");
  line += strlen("location ");

  // find the open brace, set it to the nul character, then strip any space off
  // of the intervening characters.  that is our location
  char* open_brace = strchr(line, '{');
  if (open_brace == NULL) {
    px_log_error("%s line %d: malformed location block, expected '{' before EOL", conf_filename, lineno);
    return false;
  }
  *open_brace = '\0';

  lrstrip(line);
  if (line[0] == '\0') {
    px_log_error("%s line %d: malformed location block, empty location", conf_filename, lineno);
    return false;
  }

  char* loc_path_copy = strdup(line);
  if (!loc_path_copy) {
    px_log_error("no memory");
    return false;
  }

  struct polluxd_config_location* new_loc = (struct polluxd_config_location*)calloc(1, sizeof(*new_loc));
  if (!new_loc) {
    px_log_error("no memory");
    free(loc_path_copy);
    return false;
  }

  new_loc->path = loc_path_copy;
  parse_ctx->loc = new_loc;
  return true;
}

static _Bool parse_main_line(int lineno, char* line, struct conf_parse_context* parse_ctx) {

  struct polluxd_config* conf = parse_ctx->conf;

  if (strncmp(line, "location ", strlen("location ")) == 0) {
    return parse_new_location_line(lineno, line, parse_ctx);
  }

  const struct kv main_opts[] = {
    { "host",         &conf->host,        copy_string },
    { "listen_addr",  &conf->listen_addr, copy_string },
    { "port",         &conf->port,        copy_string },
    { "cert_file",    &conf->cert_file,   copy_string },
    { "key_file",     &conf->key_file,    copy_string },
    { "docroot",      &conf->docroot,     copy_string  },
    { "autoindex",    &conf->autoindex,   to_bool },
    { "index_file",   &conf->default_index_file,  copy_string },
    { "chroot_dir",   &conf->chroot_dir,  copy_string  },
    { "drop_user",    &conf->drop_user,   copy_string  },
    { "drop_group",   &conf->drop_group,  copy_string  },
    { "logfile",      &conf->logfile,     copy_string  }
  };

  return parse_kv_line(lineno, line, main_opts, sizeof(main_opts) / sizeof(main_opts[0]), parse_ctx);
}

// parse a single line
// line should be lrstripped
// return true if line parsed successfully
static _Bool parse_line(int lineno, char* line, struct conf_parse_context* parse_ctx) {
  // if we're in a location block, first check if the line is and end-block marker '}'
  if (parse_ctx->loc)
    return parse_location_line(lineno, line, parse_ctx);
  return parse_main_line(lineno, line, parse_ctx);
}

void polluxd_config_location_reset(struct polluxd_config_location* loc) {
  if (!loc)
    return;
  free(loc->path);
  free(loc->cgi_user);
  free(loc->cgi_group);
  free(loc->cgi_chroot);
  free(loc->cgi_script);
  free(loc->docroot);
  free(loc->prefix);
  free(loc->allowed_cert_file);
  free(loc->action);
  free(loc->index_file);
  *loc = (struct polluxd_config_location) { 0 };
}

int polluxd_config_read_file(struct polluxd_config* conf, FILE* cfile, char const* conf_filename) {
  if (!conf || !cfile)
    return 0;

  struct conf_parse_context parse_ctx = {
    .conf_filename = conf_filename,
    .conf = conf,
    .loc = NULL
  };

  char* line = NULL;
  size_t alloc_sz = 0;

  int lineno = 0;
  while (1) {
    ++lineno;
    errno = 0;
    ssize_t line_sz = getline(&line, &alloc_sz, cfile);
    if (line_sz < 0) {
      if (errno != 0) {
        px_log_error("could not read configuration file: line %d: %s", lineno, strerror(errno));
        free(line);
        polluxd_config_location_reset(parse_ctx.loc);
        free(parse_ctx.loc);
        return false;
      }
      break;
    }

    lrstrip(line);

    if (*line == '\0' || *line == '#') // skip comments and empty lines
      continue;

    _Bool r = parse_line(lineno, line, &parse_ctx);
    if (!r) {
      free(line);
      return false;
    }
  }

  free(line);
  if (parse_ctx.loc) {
    px_log_error("unterminated location block");
    polluxd_config_location_reset(parse_ctx.loc);
    free(parse_ctx.loc);
    return false;
  }
  return true;
}

void polluxd_config_reset(struct polluxd_config* conf) {
  if (!conf)
    return;

  char* fields[] = {
    conf->host,
    conf->listen_addr,
    conf->port,
    conf->cert_file,
    conf->key_file,
    conf->docroot,
    //conf->autoindex,
    conf->default_index_file,
    conf->chroot_dir,
    conf->drop_user,
    conf->drop_group,
    conf->logfile
  };

  const int nentries = sizeof(fields) / sizeof(fields[0]);
  for (int i = 0; i < nentries; ++i)
    free(fields[i]);

  struct polluxd_config_location* loc = conf->locations;
  while (loc) {
    struct polluxd_config_location* next = loc->next;
    polluxd_config_location_reset(loc);
    free(loc);
    loc = next;
  }
  *conf = (struct polluxd_config) { 0 };
}
