#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <polluxd.h>
#include <polluxd_cgi.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

void print_open_fds(void) {
#ifdef __linux__
  char cmdbuf[64] = { 0 };
  snprintf(cmdbuf, sizeof(cmdbuf), "ls -lah /proc/%d/fd", (int)getpid());
  if (system(cmdbuf) < 0)
    (void)0;
#endif // __linux__
}

_Bool polluxd_recv_cgi_fd(struct polluxd_client* client, int rxsock, int* out_f) {

  *out_f = -1;

  const uint32_t check_value = 0x55aa55aa;
  uint32_t echo = check_value;
  int r = send(rxsock, &echo, sizeof(echo), MSG_NOSIGNAL);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_error("%s: cgi request socket closed", client->gemctx.conn.peer_addr_str);
    else
      px_log_error("%s: error requesting new cgi socket: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    return false;
  }

  union {
    struct cmsghdr cm;
    char control[CMSG_SPACE(sizeof(int))];
  } ctrl = { 0 };

  struct iovec msgiov = { .iov_base = &echo, .iov_len = sizeof(echo) };
  struct msghdr mhdr = {  .msg_name = NULL,
    .msg_namelen = 0,
    .msg_iov = &msgiov,
    .msg_iovlen = 1,
    .msg_control = ctrl.control,
    .msg_controllen = sizeof(ctrl.control) };

  r = recvmsg(rxsock, &mhdr, 0);
  if (r < 0) {
    int e = errno;
    px_log_error("%s: could not recvmsg: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    return false;
  } else if (r == 0) {
    px_log_error("%s: cgi helper socket closed", client->gemctx.conn.peer_addr_str);
    return false;
  }

  if (echo != check_value)
    px_log_warn("%s: check value error, expected %u but got %u",
                client->gemctx.conn.peer_addr_str, check_value, echo);

  // the only other echo right now is PX_WORKER_MSG_SENDF so we should by default expect a file
  struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
  if (cmsgp != NULL
      && cmsgp->cmsg_len == CMSG_LEN(sizeof(int))
      && cmsgp->cmsg_level == SOL_SOCKET
      && cmsgp->cmsg_type == SCM_RIGHTS)
  {
    *out_f = *((int*)CMSG_DATA(cmsgp));
  }

  return true;
}

static _Bool send_cgi_fd(int txsock, int fd) {
  // send a file descriptor using SCM_RIGHTS over a unix socket
  uint32_t echo = 0;
  int r = recv(txsock, &echo, sizeof(echo), 0);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_info("cgi helper %d: cgi request socket closed, shutting down", (int)getpid());
    else
      px_log_info("cgi helper %d: shutting down, error when waiting for socket request: %s",
                  (int)getpid(), strerror(e));
    return false;
  }

  union {
    struct cmsghdr cm; // alignment
    char control[CMSG_SPACE(sizeof(int))];
  } ctrl = { 0 };

  // we have to transmit a small amount of 'real' data to send a file descriptor
  struct iovec msgiov = { .iov_base = &echo, .iov_len = sizeof(echo) };
  struct msghdr mhdr = {  .msg_name = NULL,
                          .msg_namelen = 0,
                          .msg_iov = &msgiov,
                          .msg_iovlen = 1,
                          .msg_control = ctrl.control,
                          .msg_controllen = sizeof(ctrl.control) };

  struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
  cmsgp->cmsg_len = CMSG_LEN(sizeof(int));
  cmsgp->cmsg_level = SOL_SOCKET;
  cmsgp->cmsg_type = SCM_RIGHTS;
  *((int*)CMSG_DATA(cmsgp)) = fd;

  r = sendmsg(txsock, &mhdr, MSG_NOSIGNAL);
  close(fd); // always close fd whether we send it or not
  if (r < 0) {
    int e = errno;
    px_log_warn("cgi helper %d: could not send cgi communication socket: %s", (int)getpid(), strerror(e));
    return false;
  }
  px_log_info("cgi helper %d: sent cgi file descriptor %d", (int)getpid(), fd);
  return true;
}

static _Bool is_allowed_cgi_env(char const* key) {
  static char const* const allowed_cgi_fields[] = {
    "GATEWAY_INTERFACE",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "GEMINI_URL",
    "PATH_INFO",
    "PATH_TRANSLATED",
    "SCRIPT_NAME",
    "QUERY_STRING",
    "HOSTNAME",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_PORT",
    "REMOTE_USER",
    "TLS_CIPHER",
    "TLS_VERSION",
    "AUTH_TYPE",
    "TLS_CLIENT_HASH",
    "TLS_CLIENT_PUBKEY_HASH" };

  for (unsigned i = 0, n = px_n_elements(allowed_cgi_fields); i < n; ++i) {
    if (strcmp(key, allowed_cgi_fields[i]) == 0)
      return true;
  }
  return false;
}

static _Bool unpack_envp(char* buf, size_t buf_sz, char** envp, size_t envp_sz) {

  size_t n_env = 0;
  px_log_assert(buf_sz > 0 && buf[buf_sz - 1] == '\0', "assumption violation");
  while (buf_sz > 0) {
    char* nul = memmem(buf, buf_sz, &(char) { '\0' }, sizeof(char));
    if (!nul) { // this should never happen
      memset(buf, 0, buf_sz);
      break;
    }

    do {
      char* eq = strchr(buf, '=');
      if (!eq) {
        // zero out the invalid data
        memset(buf, 0, nul - buf);
        break;
      }

      *eq = '\0';
      char* key = buf;
      if (!is_allowed_cgi_env(key)) {
        px_log_debug("cgi worker %d: invalid key %s", (int)getpid(), key);
        memset(buf, 0, nul - buf);
        break;
      }
      *eq = '=';

      if (n_env >= (envp_sz - 1)) {
        // out of space
        return false;
      }
      envp[n_env++] = buf;
    } while (0);

    buf_sz -= nul - buf;
    buf = nul;
    if (buf_sz > 0) {
      --buf_sz;
      ++buf;
    }
  }

  if (n_env < envp_sz) {
    envp[n_env] = NULL;
  } else {
    return false;
  }

  return true;
}

static _Bool execve_from_docroot(char const* script, char** envp, char const* docroot) {

  char* dots_buf = NULL;
  int ancestor_fd = -1, parent_fd = -1; // temporaries used to traverse the directory ancestors

  if (!script) {
    px_log_info("cgi worker %d: no script", (int)getpid());
    return false;
  }

  char* path_rp = realpath(script, NULL);
  if (!path_rp)
    goto ERR_OR_DONE;

  // summary of what we do here.  at any error we release resources and return -1
  // - open a file descriptor to the dirname as 'parent_fd'.
  // - initialize the 'dots buffer' with the string "."
  // - loop:
  //      open up some ancestor of parent_fd via openat(parent_fd, dots_buffer, ...)
  //      if the ancestor's inode matches the docroot inode, then break the loop (success)
  //      if the ancestor is the root dir, exit as error (file not in docroot)
  //      otherwise, append "/.." to the dots buffer and loop
  // - if we break from the loop (success, we're in docroot) then we use
  //   openat(parent_fd, basename, O_NOFOLLOW...) to open the real file.  return this file descriptor

  // separate dirname and basename;
  char* dirname = NULL;
  char* basename = NULL;
  char* last_slash = strrchr(path_rp, '/');
  if (last_slash) {
    *last_slash = '\0';
    dirname = path_rp;
    basename = last_slash + 1;
  } else {
    dirname = ".";
    basename = path_rp;
  }

  parent_fd = open(dirname, O_RDONLY | O_DIRECTORY | O_CLOEXEC);
  if (parent_fd < 0)
    goto ERR_OR_DONE;

  _Bool do_execve = false;
  if (docroot && strcmp(docroot, "/") != 0) {
    struct stat root_s;
    struct stat docroot_s;
    if (stat("/", &root_s) < 0 || stat(docroot ? docroot : "", &docroot_s) < 0)
      goto ERR_OR_DONE;

    // allocate
    const int dots_buf_sz = 1024; // arbitrarily: size of gemini request
    dots_buf = (char*)calloc(1, dots_buf_sz);
    if (!dots_buf)
      goto ERR_OR_DONE;

    int dots_len = 0;

    ancestor_fd = parent_fd; // start with the parent
    while (ancestor_fd >= 0) {
      struct stat s;
      if (fstat(ancestor_fd, &s) < 0)
        goto ERR_OR_DONE;

      // is the ancestor the docroot?  if so then we can open the file!
      if (s.st_ino == docroot_s.st_ino
          && s.st_dev == docroot_s.st_dev)
      {
        if (ancestor_fd != parent_fd)
          close(ancestor_fd);
        ancestor_fd = -1;
        do_execve = true;
        break; // success
      }

      // check if the ancestor is the filesystem root.  if so then the file is not
      // in the docroot and we are done
      if (s.st_ino == root_s.st_ino
          && s.st_dev == root_s.st_dev)
      {

        // put the slash back for the message, so that path_rp reads as the whole
        // path instead of just the dirname
        if (last_slash)
          *last_slash = '/';

        px_log_warn("cgi worker %d: %s (realpath %s) is outside of the docroot (%s)",
            (int)getpid(), script, path_rp, docroot);
        errno = EPERM;
        goto ERR_OR_DONE;
      }

      if (dots_len + (int)sizeof("../") > dots_buf_sz) // out of space?
        break; // can't do any more dots!  premature end

      memcpy(dots_buf + dots_len, "../", sizeof("../"));
      dots_len += sizeof("../") - 1;

      if (ancestor_fd != parent_fd)
        close(ancestor_fd);

      // open the next dir up
      ancestor_fd = openat(parent_fd, dots_buf, O_RDONLY | O_DIRECTORY | O_CLOEXEC);
    }
  } else {
    do_execve = true;
  }

  if (do_execve) {
    if (fchdir(parent_fd) == 0) {
      char* path[2] = { basename, NULL };
      execve(basename, path, envp);
    } else {
      px_log_warn("cgi worker %d: could not fchdir: %s", (int)getpid(), strerror(errno));
    }
  }

ERR_OR_DONE:

  {
    int e = errno;
    if (parent_fd >= 0)
      close(parent_fd);
    if (ancestor_fd >= 0)
      close(ancestor_fd);
    free(dots_buf);
    free(path_rp);
    errno = e;
  }
  return false;
}

#define CGI_BUFFER_SZ 4096
static void cgi_worker(int cgi, char const* script, char const* docroot) {
  px_log_info("cgi worker launched on pid %d", (int)getpid());

  char* buf = calloc(1, CGI_BUFFER_SZ + 1);
  if (!buf)
    return;

  int r = recv(cgi, buf, CGI_BUFFER_SZ, 0);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_error("cgi worker %d: cgi output socket closed prematurely", (int)getpid());
    else
      px_log_error("cgi worker %d: could not read cgi startup data from %d: %s",
                   (int)getpid(), cgi, strerror(e));
    print_open_fds();
    free(buf); // make valgrind happy
    exit(r == 0 ? 0 : -1);
  }
  if (buf[r-1] != '\0') {
    buf[r] = '\0'; // ensure we're always nul-terminated
    ++r;
  }

  char** envp = (char**)calloc(64, sizeof(char*));
  _Bool env_ok = unpack_envp(buf, (size_t)r, envp, 64);
  if (!env_ok) {
    px_log_info("cgi worker %d: could not set up cgi environment", (int)getpid());
    free(envp);
    close(cgi);
    exit(-1);
  }

  px_log_info("cgi worker %d: read %d bytes of cgi data", (int)getpid(), r);

  char* script_path = NULL;
  if (script) { // script was provided in the config file
    px_log_info("cgi worker %d: executing fixed script %s", (int)getpid(), script);
    script_path = strdup(script);
  } else { // polluxd translated the script name from a path
    px_log_info("cgi worker %d: translating script from path", (int)getpid());
    for (unsigned i = 0; i < 64 && envp[i]; ++i) {
      if (strncmp(envp[i], "PATH_TRANSLATED", sizeof("PATH_TRANSLATED") - 1) == 0) {
        char* eq = strchr(envp[i], '=');
        if (eq)
          script_path = strdup(eq + 1);
        break;
      }
    }
  }

  if (!script_path) {
    px_log_info("cgi worker %d: could not find script (no PATH_TRANSLATED?), cannot execute cgi", (int)getpid());
    close(cgi);
    free(envp);
    exit(-1);
  }

  if (dup2(cgi, STDOUT_FILENO) < 0) {
    int e = errno;
    px_log_fatal("cgi helper %d: could not dup2 cgi fd %d to stdout: %s",
                 (int)getpid(), cgi, strerror(e));
  }
  close(cgi); // don't need cgi anymore since it's now on stdout

  if (!execve_from_docroot(script_path, envp, docroot)) {
    int e = errno;
    px_log_error("cgi worker %d: could not execve %s: %s", (int)getpid(), script_path, strerror(e));
  }

  fprintf(stdout, "51 Not found\r\n");

  free(script_path);
  free(envp);
  free(buf);
  exit(0);
}

// the meat of the cgi helper process
static _Bool cgi_helper(int cgi_sock, char const* script, char const* docroot) {

  struct pollfd pfd = { .fd = cgi_sock, .events = POLLOUT, .revents = 0 };
  int r = poll(&pfd, 1, -1);
  if (r < 0) {
    int e = errno;
    px_log_error("cgi helper %d: error on poll: %s", (int)getpid(), strerror(e));
    return false;
  }

  int socks[] = { -1, -1 };

  if (socketpair(AF_UNIX, SOCK_STREAM | SOCK_NONBLOCK, 0, socks) != 0) {
    int e = errno;
    px_log_error("cgi helper %d: could not pipe(): %s", (int)getpid(), strerror(e));
    return false;
  }

  if (!send_cgi_fd(cgi_sock, socks[0])) {
    close(socks[1]);
    return false;
  }
  socks[0] = -1;

  pid_t p = fork();
  if (p < 0) {
    int e = errno;
    px_log_error("cgi helper %d: could not fork: %s", (int)getpid(), strerror(e));
    close(socks[1]);
    return true;
  }

  if (p > 0) {
    close(socks[1]);
    return true;
  } else {
    cgi_worker(socks[1], script, docroot); // should not return
    close(socks[1]);
    exit(-1);
  }

  return true;
}

// @brief launch the cgi helper process.
// @details the helper process is what forks new processes and hands
// communcations channels back to the main polluxd process.  the helper
// receives a file descriptor from the main communication channel, then reads
// 1) a uint32_t 'x' from the received file and then 2) reads 'x' bytes of data
// from the file descriptor into a buffer.  if 'x' is too large then the new
// file descriptor is closed and no cgi is executed.  the helper then forks.
// the parent process then closes its copy of the new file descriptor, frees
// the buffered data, and waits for another message.  the child process parses
// the cgi data received during the original request.  if the cgi data is
// properly formatted the child then sets up the necessary environment
// variables from the received data, calls dup2 to move the received fd to
// stdout, then calls execve on the requested script.  at that point the cgi
// script will be executed and can write gemini data back to the stdout pipe
// (including the header) which will be read and forwarded in the main polluxd
// process
_Bool launch_cgi_helper(struct polluxd_context* pxd, struct polluxd_route_data* rtd) {
  int spair[2] = { -1, -1 };

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, spair) != 0) {
    int e = errno;
    px_log_fatal("cgi helper: socketpair failed: %s", strerror(e));
    return false;
  }

  char const* cgi_user = rtd->cgi_user ? rtd->cgi_user : pxd->conf.drop_user;
  char const* cgi_group = rtd->cgi_group ? rtd->cgi_group : pxd->conf.drop_group;
  char const* cgi_chroot = rtd->cgi_chroot ? rtd->cgi_chroot : pxd->conf.chroot_dir;
  char const* cgi_script = rtd->cgi_script;
  char const* docroot = rtd->docroot ? rtd->docroot : pxd->conf.docroot;

  // to set up th
  pid_t p = fork();
  if (p < 0) {
    int e = errno;
    px_log_fatal("cgi helper: fork failed: %s", strerror(e));
    return false;
  }

  if (p > 0) {
    close(spair[1]);
    int cpid_status = 0;

    // wait for the first child process of the double-fork to exit
    pid_t cpid = waitpid(p, &cpid_status, 0);
    if (cpid != p
        || WEXITSTATUS(cpid_status) != 0
        || read(spair[0], &(char) { 0 }, sizeof(char)) != sizeof(char))
    {
      px_log_error("could not wait for child cgi helper process: %s", strerror(errno));
      close(spair[0]);
      return false;
    }
    px_log_assert(rtd->cgi_helper_sock == -1, "assumption violation");
    rtd->cgi_helper_sock = spair[0];
    return true;
  }

  // set stderr to O_APPEND mode, or print a warning
  // if we don't do this then a CGI script could truncate the log file
  const int set_append_fds[] = { fileno(stdin), fileno(stdout), fileno(stderr) };
  static char const* const strnames[] = { "stdin", "stdout", "stderr" };
  for (unsigned i = 0; i < px_n_elements(set_append_fds); ++i) {
    int fd = set_append_fds[i];
    int flags = fcntl(fd, F_GETFL, 0);
    if (flags < 0 || fcntl(fd, F_SETFL, flags | O_APPEND) < 0) {
      int e = errno;
      px_log_warn("cgi child %d: could not set O_APPEND on %s: %s", (int)getpid(), strnames[i], strerror(e));
    }
  }

  // TODO add cgi chroot and docroot dirs
  polluxd_drop_privileges(cgi_user, cgi_group, cgi_chroot, docroot ? docroot : "/", true);

  // child process - we need to break into a new session, re-fork (parent exits),
  // and then serve cgi stuff on the new child
  if (chdir("/") != 0)
    px_log_fatal("cgi helper %d: could not chdir: %s", (int)getpid(), strerror(errno));

  close(spair[0]);

  if (setsid() < 0)
    px_log_warn("cgi helper %d: could not setsid: %s", (int)getpid(), strerror(errno));

  // double fork so that the main process doesn't wait on us
  pid_t p1 = fork();
  if (p1 < 0)
    px_log_fatal("cgi helper first stage: could not re-fork: %s", strerror(errno));

  if (p1 > 0) // parent exits sucessfully
    exit(0);

  if (signal(SIGCHLD, SIG_IGN) != 0) {
    int e = errno;
    px_log_fatal("cgi helper %d: could not ignore SIGCHLD: %s", (int)getpid(), strerror(e));
  }

  // write one byte of data to the socket so the parent can continue
  char out = 0;
  if (write(spair[1], &out, sizeof(out)) != sizeof(out))
    px_log_fatal("cgi: could not write confirmatory data: %s", strerror(errno));
  px_log_info("cgi started on pid %d", (int)getpid());

  while (true) {
    if (!cgi_helper(spair[1], cgi_script, docroot))
      break;
  }

  exit(0);
}

_Bool polluxd_launch_cgi_helpers(struct polluxd_context* pxd) {
  for (unsigned i = 0, n = px_n_elements(pxd->route_data); i < n; ++i) {
    if (pxd->route_data[i].action == POLLUXD_DO_CGI) {
      if (!launch_cgi_helper(pxd, &pxd->route_data[i]))
        px_log_fatal("could not launch cgi helper process");
    }
  }
  return true;
}
