#!/bin/sh

trim_nl() {
  tr -d '\n'
}

echo_version() {
  DEFAULT_VERSION="unknown-$(date +%Y%m%d)"

  # we require that we're in a git repo here
  if ! git status >/dev/null 2>&1 ; then
    echo "$DEFAULT_VERSION" | trim_nl
    return 0
  fi

  # first check if any tracked files have been modified.
  # if so we're on a development branch of a commit
  if [ "$(git status -uno --porcelain 2>/dev/null | wc -l 2>/dev/null)" -ne 0 ] ; then
    echo "development version based on " | trim_nl

    # we need to strip the newline off of the end of this output so that it doesn't show up in the header
    git rev-list --date='format:%Y%m%d' --no-commit-header --max-count=1 \
      --pretty="format:revision %h-" HEAD 2>/dev/null | trim_nl
    date +"%Y%m%d" | trim_nl

  else # <- no files have changed, so we can use a tag or un-munged revision
    # check if there's a tag associated with this commit, and use it for the version if so
    TAG="$(git tag --contains HEAD 2>/dev/null)"
    if [ -n "$TAG" ] ; then # <- tag exists on our commit, cool
      echo "version $TAG" | trim_nl
    else # <- no tag associated with commit, use revision and date
      # we're sitting on an unmodified commit, use its revision
      if ! git rev-list --date='format:%Y%m%d' --no-commit-header --max-count=1 \
        --pretty="format:revision %h-%cd" HEAD 2>/dev/null | trim_nl ; then
            echo "$DEFAULT_VERSION" | trim_nl
      fi
    fi
  fi
  return 0
}

write_version_header() {
  outfile="${1}"
  version="${2}"

  if [ -z "$outfile" ] ; then
    echo "No output file specified"
    return -1
  fi
  echo "Writing version header to $outfile"

  # write out the px_version.h header
  cat <<EOF > "$outfile"
#ifndef PX_VERSION_H__
#define PX_VERSION_H__

#define LIBPXD_VERSION "$version"

#endif // PXC_VERSION_H__
EOF
}

VERSION_HDR_FILE="${1:-include/libpxd/px_version.h}"
VERSION="$(echo_version)"

if ! git status >/dev/null 2>&1 ; then
  # source tree is not a git repo
  if [ ! -f "$VERSION_HDR_FILE" ] ; then
    echo "No version header present, writing the default: $VERSION"
    write_version_header "$VERSION_HDR_FILE" "$VERSION"
  else
    echo "Not updating version information"
  fi
  exit 0
fi


TMP_HDR="$(mktemp ./.px_version.h.XXXXXXXX)"
write_version_header "$TMP_HDR" "$VERSION"

if ! cmp "$VERSION_HDR_FILE" "$TMP_HDR" >/dev/null 2>&1 ; then
  echo "Updating version info to: $VERSION"
  mv "$TMP_HDR" "$VERSION_HDR_FILE"
else
  echo "Version info is up to date: $VERSION"
fi

rm -f "$TMP_HDR"
