#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_log.h>
#include <libpxd/px_network.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <arpa/inet.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

_Bool px_net_addr_to_str(
    int sock,
    char* addrbuf, size_t addrbuf_sz, // <host ip>:<port> or unix:<inode>
    char* hostbuf, size_t hostbuf_sz, // <host ip> or unix:<inode>
    char* portbuf, size_t portbuf_sz, // <port>
    _Bool get_peer)
{
  char loc_addrbuf[PX_ADDRSTR_LEN] = { 0 };
  char loc_portbuf[8] = { 0 };

  // get the sockaddr data for whatever end of the connection we want
  struct sockaddr_storage ss;
  socklen_t ss_len = sizeof(ss);
  memset(&ss, 0, ss_len);
  int r = get_peer
    ? getpeername(sock, (struct sockaddr*)&ss, &ss_len)
    : getsockname(sock, (struct sockaddr*)&ss, &ss_len);
  if (r < 0)
    return false;

  _Bool ret = true;

  switch (ss.ss_family) {
    case AF_INET :
      {
        struct sockaddr_in const* sin = (struct sockaddr_in const*)&ss;
        char const* rc = inet_ntop(AF_INET, &sin->sin_addr, loc_addrbuf, sizeof(loc_addrbuf));
        if (rc != loc_addrbuf) {
          px_log_warn("overflow, buffers need adjustment");
          loc_addrbuf[0] = '\0';
          ret = false;
          break;
        }

        unsigned short port = ntohs(sin->sin_port);

        int r = snprintf(loc_portbuf, sizeof(loc_portbuf), "%hu", port);
        if (r < 0 || (unsigned)r >= sizeof(loc_portbuf)) {
          px_log_warn("overflow, bufers need adjustment");
          loc_portbuf[0] = '\0';
          ret = false;
          break;
        }
      }
      break;
    case AF_INET6 :
      {
        struct sockaddr_in6 const* sin6 = (struct sockaddr_in6 const*)&ss;
        char const* rc = inet_ntop(AF_INET6, &sin6->sin6_addr, loc_addrbuf, sizeof(loc_addrbuf));
        if (rc != loc_addrbuf) {
          px_log_warn("overflow, buffers need adjustment");
          loc_addrbuf[0] = '\0';
          ret = false;
          break;
        }

        unsigned short port = ntohs(sin6->sin6_port);

        int r = snprintf(loc_portbuf, sizeof(loc_portbuf), "%hu", port);
        if (r < 0 || (unsigned)r >= sizeof(loc_portbuf)) {
          px_log_warn("overflow, bufers need adjustment");
          loc_portbuf[0] = '\0';
          ret = false;
          break;
        }
      }
      break;
    case AF_UNIX :
      {
        loc_portbuf[0] = '\0'; // no port for unix sockets
        struct sockaddr_un const* un = (struct sockaddr_un const*)&ss;

        _Static_assert(sizeof(loc_addrbuf) > sizeof(un->sun_path), "address string buffers need to be adjusted");
        if (get_peer) {
          // for a peer connection we print unix:<inode number> since
          // un->sun_path will be empty
          struct stat st;
          if (fstat(sock, &st) != 0) {
            ret = false;
            break;
          }

          r = snprintf(loc_addrbuf, sizeof(loc_addrbuf), "unix:%llu", (unsigned long long)st.st_ino);
          if (r < 0 || (unsigned)r >= sizeof(loc_addrbuf)) {
            ret = false;
            break;
          }
        } else { // for the local address we copy sun_path
          size_t un_path_len = strnlen(un->sun_path, sizeof(un->sun_path));
          memcpy(loc_addrbuf, un->sun_path, un_path_len);
          loc_addrbuf[un_path_len] = '\0';
        }
      }
      break;
    default :
      px_log_assert(0, "unsupported connection medium");
  }

  if (addrbuf) {
    if (loc_portbuf[0] != '\0')
      r = snprintf(addrbuf, addrbuf_sz, "%s:%s", loc_addrbuf, loc_portbuf);
    else
      r = snprintf(addrbuf, addrbuf_sz, "%s", loc_addrbuf);

    if (r < 0 || (unsigned)r >= addrbuf_sz) {
      *addrbuf = '\0';
      ret = false;
    }
  }

  if (hostbuf && hostbuf_sz > 0) {
    r = snprintf(hostbuf, hostbuf_sz, "%s", loc_addrbuf);
    if (r < 0 || (unsigned)r >= hostbuf_sz) {
      *hostbuf = '\0';
      ret = false;
    }
  }

  if (portbuf && portbuf_sz > 0) {
    r = 0;
    if (loc_portbuf[0] != '\0')
      r = snprintf(portbuf, portbuf_sz, "%s", loc_portbuf);
    else
      *portbuf = '\0';

    if (r < 0 || (unsigned)r >= portbuf_sz) {
      *portbuf = '\0';
      ret = false;
    }
  }

  return ret;
}

_Bool px_net_local_addr_to_str(int sock, char* bufip, size_t bufip_sz) {
  return px_net_addr_to_str(sock, bufip, bufip_sz, NULL, 0, NULL, 0, false);
}

_Bool px_net_peer_addr_to_str(int sock, char* bufip, size_t bufip_sz) {
  return px_net_addr_to_str(sock, bufip, bufip_sz, NULL, 0, NULL, 0, true);
}

_Bool px_net_local_host_to_str(int sock, char* hostbuf, size_t hostbuf_sz) {
  return px_net_addr_to_str(sock, NULL, 0, hostbuf, hostbuf_sz, NULL, 0, false);
}

_Bool px_net_local_port_to_str(int sock, char* portbuf, size_t portbuf_sz) {
  return px_net_addr_to_str(sock, NULL, 0, NULL, 0, portbuf, portbuf_sz, false);
}

_Bool px_net_peer_host_to_str(int sock, char* hostbuf, size_t hostbuf_sz) {
  return px_net_addr_to_str(sock, NULL, 0, hostbuf, hostbuf_sz, NULL, 0, true);
}

_Bool px_net_peer_port_to_str(int sock, char* portbuf, size_t portbuf_sz) {
  return px_net_addr_to_str(sock, NULL, 0, NULL, 0, portbuf, portbuf_sz, true);
}

static int accept_all_client_certs(int preverify_ok, X509_STORE_CTX *ctx) {
  // Gemini operates with a trust-on-first-use scheme, so we do not care what
  // the certificate information is at all, we only care that the client has
  // access to the client cert's private key.  validating the client's control
  // of the private key happens during the SSL handshake and any further
  // validation is beyond thes cope of libpxd.

  (void)preverify_ok;
  (void)ctx;
  // accept all certificates
  return 1;
}

static int __attribute__((unused)) verify_client_certs(int preverify_ok, X509_STORE_CTX *ctx) {
  char    buf[256];

  X509* err_cert = X509_STORE_CTX_get_current_cert(ctx);
  int err = X509_STORE_CTX_get_error(ctx);
  int depth = X509_STORE_CTX_get_error_depth(ctx);

  /*
   * Retrieve the pointer to the SSL of the connection currently treated
   * and the application specific data stored into the SSL object.
   */
  //ssl = X509_STORE_CTX_get_ex_data(ctx, SSL_get_ex_data_X509_STORE_CTX_idx());
  //mydata = SSL_get_ex_data(ssl, mydata_index);

  X509_NAME_oneline(X509_get_subject_name(err_cert), buf, 256);
  buf[255] = '\0';

  /*
   * Catch a too long certificate chain. The depth limit set using
   * SSL_CTX_set_verify_depth() is by purpose set to "limit+1" so
   * that whenever the "depth>verify_depth" condition is met, we
   * have violated the limit and want to log this error condition.
   * We must do it here, because the CHAIN_TOO_LONG error would not
   * be found explicitly; only errors introduced by cutting off the
   * additional certificates would be logged.
   */
  //if (depth > mydata->verify_depth) {
  //  preverify_ok = 0;
  //  err = X509_V_ERR_CERT_CHAIN_TOO_LONG;
  //  X509_STORE_CTX_set_error(ctx, err);
  //}

  if (!preverify_ok && err != X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) {
    if (err == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) {
      X509_NAME_oneline(X509_get_issuer_name(err_cert), buf, 256);
      px_log_info("client certificate: bad issuer: %s", buf);
    }
    px_log_info("client certificate: verify error: num=%d:%s:depth=%d:%s",
                err, X509_verify_cert_error_string(err), depth, buf);
  }

  return 1;
}

SSL_CTX* px_net_ssl_ctx(void) {

  SSL_CTX* ctx = SSL_CTX_new(TLS_server_method());
  if (!ctx) {
    px_log_warn("could not allocate ssl context");
    return NULL;
  }

  // TODO allow user-specified verify callbacks.  someone may want to check and
  // reject connections
  //
  SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, accept_all_client_certs);

#if defined(SSL_OP_NO_RENEGOTIATION)
  // no renegotiation, it's stupid
  SSL_CTX_set_options(ctx, SSL_OP_NO_RENEGOTIATION);
#endif

  // we do a minimum of TLS 1.2.  TLS 1.3 would be better
  if (SSL_CTX_set_min_proto_version(ctx, TLS1_2_VERSION) != 1) {
    px_log_warn("could not set minimum TLS version");
    SSL_CTX_free(ctx);
    return NULL;
  }

  const char* const PREFERRED_CIPHERS = "HIGH:!aNULL:!kRSA:!SRP:!PSK:!CAMELLIA:!RC4:!MD5:!DSS";
  if (SSL_CTX_set_cipher_list(ctx, PREFERRED_CIPHERS) == 0) {
    px_log_warn("could not set ciphers");
    SSL_CTX_free(ctx);
    return NULL;
  }

  // we need to set a sesssion id context.  from
  // SSL_CTX_set_session_id_context(3): If the session id context is not set on
  // an SSL/TLS server and client certificates are used, stored sessions will
  // not be reused but a fatal error will be flagged and the handshake will
  // fail.
  // so - if we don't set this, clients that try to re-use a session will fail
  // automatically
  static const char ctxname[] = "libpxd";
  if (SSL_CTX_set_session_id_context(ctx, (unsigned char const*)ctxname, sizeof(ctxname) - 1) != 1) {
    px_log_warn("could not set SSL session id context");
    SSL_CTX_free(ctx);
    return NULL;
  }

  return ctx;
}

_Bool px_net_use_ssl_cert(SSL_CTX* ctx, char const* certfile, char const* keyfile) {
  px_log_assert(ctx, "SSL_CTX should not be null");
  px_log_assert(certfile, "certificate file should not be null");
  px_log_assert(keyfile, "key file file should not be null");

  int r = SSL_CTX_use_certificate_chain_file(ctx, certfile);
  if (r != 1) {
    unsigned long e = ERR_get_error();
    px_log_warn("could not use cert chain file %s: %s (%lu)", certfile, ERR_error_string(e, NULL), e);
    return false;
  }

  // load the private key from file, default "key.pem"
  ERR_clear_error();
  r = SSL_CTX_use_PrivateKey_file(ctx, keyfile, SSL_FILETYPE_PEM);
  if (r != 1) {
    int e = ERR_get_error();
    px_log_warn("could not load PEM private key (%s): %s\n", keyfile, ERR_error_string(e, NULL));
    SSL_CTX_free(ctx);
    return false;
  }

  // make sure the private key matches the certificate
  ERR_clear_error();
  r = SSL_CTX_check_private_key(ctx);
  if (r != 1) {
    px_log_warn("private key does not match certificate chain: %s", ERR_error_string(ERR_get_error(), NULL));
    return false;
  }
  return true;
}
