#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_log.h>
#include <errno.h>
#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

pthread_mutex_t* get_mtx(void) {
  static pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;
  return &mtx;
}

void px_log_msg(char const* file, int line, enum px_log_level lvl, char const* fmt, ...) {
  char buf[256] = { 0 };
  if (file && line > 0) {
    char const* prefix;
    switch (lvl) {
      case px_log_level_error :
        prefix="error ";
        break;
      case px_log_level_warn :
        prefix="warn  ";
        break;
      case px_log_level_info :
        prefix="info  ";
        break;
      case px_log_level_debug :
        prefix="debug ";
        break;
      default:
        prefix="unk";
    }
    int r = snprintf(buf, sizeof(buf), "%s %s: %d: ", prefix, file, line);
    if (r < 0 || (unsigned)r >= sizeof(buf)) {
      int e = errno;
      fprintf(stderr, "%s line %d: warning, could not print full line prefix (r=%d, err=%s)",
              __FILE__, __LINE__, r, strerror(e));
      r = (r >= 0) ? sizeof(buf)-1 : 0;
      buf[r] = '\0';
    }
  }
  va_list va;
  va_start(va, fmt);
  char* msgbuf = NULL;
  int r = vasprintf(&msgbuf, fmt, va);
  int e = errno;
  va_end(va);

  if (r >= 0) {
    pthread_mutex_t* mtx = get_mtx();
    pthread_mutex_lock(mtx);
    fprintf(stderr, "%s%s\n", buf, msgbuf ? msgbuf : "");
    pthread_mutex_unlock(mtx);
  } else {
    fprintf(stderr, "%s line %d: warning, could not vasprintf: %s", __FILE__, __LINE__, strerror(e));
  }

  free(msgbuf);
}
