#if defined(__linux__)
#define _GNU_SOURCE
#endif //  defined(__linux__)

#include <libpxd/px_connection.h>
#include <libpxd/px_event.h>
#include <libpxd/px_log.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <openssl/err.h>

// @brief the free list of px_conn_buffers that are available for use.  this
// will maintain up to a maximum number of free buffers for use in buffering
// and sending data across a TLS connection. buffers are 16k in size by default
// which is the maximum TLS packet size.
struct {
  struct px_queue buf_head;
  size_t          buf_count;
} buffer_freelist = { .buf_head = PX_QUEUE_INITIALIZER(&buffer_freelist.buf_head),
                      .buf_count = 0 };

struct px_conn_buffer* px_conn_buffer_get(void) {
 // provide a compile-time hook to disable cached conn_buffers to aid in memory
 // leak tracing
#if !defined(PX_DEBUG_CONN_BUFFER)
  struct px_queue* q = px_queue_pop_front(&buffer_freelist.buf_head);

  if (q) { // got one from the free list, decrement the free count
    --buffer_freelist.buf_count;
    return px_conn_buffer_from_queue(q);
  }
#endif // !defined(PX_DEBUG_CONN_BUFFER)

  // make a new buffer
  px_log_assert(buffer_freelist.buf_count == 0, "programming error, inconsistent state");
  struct px_conn_buffer* newbuf = (struct px_conn_buffer*)calloc(1, sizeof(*newbuf));
  if (!newbuf) {
    px_log_error("no memory");
    return NULL;
  }
  px_queue_init(&newbuf->bufq);
  return newbuf;
}

void px_conn_buffer_free(struct px_conn_buffer* buf) {
  if (!buf)
    return;

#if defined(PX_DEBUG_CONN_BUFFER)
  free(buf);
  return;
#endif // defined(PX_DEBUG_CONN_BUFFER)

  if (!px_queue_is_singleton(&buf->bufq))
    px_log_error("WARNING: buffer should be extracted before freeing");

  px_queue_extract(&buf->bufq);
  *buf = (struct px_conn_buffer) { 0 }; // clear old data
  px_queue_init(&buf->bufq);

  // prepend the free'd buffer to the free list
  px_queue_prepend(&buffer_freelist.buf_head, &buf->bufq);
  ++buffer_freelist.buf_count;

  // remove buffers from the tail
  while (buffer_freelist.buf_count > PX_BUFFER_FREELIST_HIGH_WATERMARK) {
    struct px_queue* q = px_queue_pop_back(&buffer_freelist.buf_head);
    --buffer_freelist.buf_count;
    px_log_assert(q, "assumption violation, queue should never be empty here");
    free(px_conn_buffer_from_queue(q));
  }
}

_Bool px_conn_buffer_split(struct px_queue* dst_head, struct px_conn_buffer* buf, size_t max_frag_sz) {
  if (!buf || !dst_head)
    return false;

  size_t orig_sz = buf->data_sz;

  // if the buffer fits in a single packet then just stick it onto dst and be done
  if (orig_sz <= max_frag_sz) {
    // pull buf out of any queue it's in, stick it on dst_head, reset bufp
    px_queue_extract(&buf->bufq);
    px_queue_append(dst_head, &buf->bufq);
    return true;
  }

  // temporary in case something goes wrong
  struct px_queue tmp_head;
  px_queue_init(&tmp_head);

  // we will use buf as the first element in the array of dst_head, but first
  // we need to pull out the data that overflows max_frag_sz
  size_t off = max_frag_sz;
  while (off < orig_sz) {
    struct px_conn_buffer* cb = px_conn_buffer_get();
    if (!cb)
      goto ERR;

    size_t this_sz = (orig_sz - off);
    if (this_sz > max_frag_sz)
      this_sz = max_frag_sz;

    cb->data_sz = this_sz;
    memcpy(cb->data, buf->data + off, this_sz);
    px_queue_append(&tmp_head, &cb->bufq);
    off += this_sz;
  }

  // now the only data left in buf is what will fit in a single max_frag_sz
  // packet.  prepend it to the rest of the data
  buf->data_sz = max_frag_sz;

  // pull buf out of any queue it's in, stick it at the front of tmp_head, reset bufp
  px_queue_extract(&buf->bufq);
  px_queue_prepend(&tmp_head, &buf->bufq);
  px_queue_splice(dst_head, &tmp_head);  // move tmp buffer to dst
  return true;
ERR:
  while (!px_queue_is_singleton(&tmp_head)) {
    struct px_queue* q = px_queue_pop_front(&tmp_head);
    struct px_conn_buffer* cb = px_conn_buffer_from_queue(q);
    px_conn_buffer_free(cb);
  }
  return false;
}

static void update_timespec(struct timespec* ts_dest) {
  struct timespec ts;
  if (clock_gettime(CLOCK_MONOTONIC, &ts) == 0)
    *ts_dest = ts;
}

enum nbio_status_ {
  PX_IO_ERROR = -1,
  PX_IO_AGAIN = 0,
  PX_IO_OK = 1
};
typedef enum nbio_status_ nbio_status;

void px_connection_init(struct px_connection* c) {
  *c = (struct px_connection) { 0 };
  c->fd = -1;
  c->is_open = false;
  px_queue_init(&c->outq_head);
  px_queue_init(&c->sendq_head);
  px_queue_init(&c->recvq.bufq);
}

static void clear_sendq(struct px_connection* c); // forward declaration

void px_connection_reset(struct px_connection* c) {
  if (!c)
    return;

  clear_sendq(c);
  px_connection_clear_outqueue(c);

  if (c->ssl) {
    SSL_free(c->ssl);
    c->ssl_in = NULL;
    c->ssl_out = NULL;
  }

  if (c->fd >= 0) {
    if (close(c->fd) < 0) {
      px_log_warn("could not close fd %d: %s", c->fd, strerror(errno));
    }
  }
  c->fd = -1;
  c->is_open = false;

  px_connection_init(c);
}

// @brief transfer data between the SSL's output buffer and the socket
static _Bool sslout_to_sendq(struct px_connection* c) {
  int pending = BIO_pending(c->ssl_out);

  if (pending < 0) // what?  why would this ever happen?
    return false;

  if (pending == 0)
    return true;

  while (pending > 0) {

    size_t upending = (size_t)pending;
    struct px_conn_buffer* sendbuf = px_conn_buffer_get();

    if (!sendbuf)
      return false;

    // technically we should incorporate sendbuf->data_offt and
    // sendbuf->data_sz into the size and destination poiner calculations but
    // since we got sendbuf from px_conn_buffer_get the data_offt and data_sz
    // members are zero

    size_t read_sz = upending > sizeof(sendbuf->data) ? sizeof(sendbuf->data) : upending;

    int r = BIO_read(c->ssl_out, &sendbuf->data[0] + sendbuf->data_offt, read_sz);
    if (r < 0 || (size_t)r != read_sz) { // it's a memory buffer, this shouldn't happen unless OOM?
      px_conn_buffer_free(sendbuf);
      return false;
    }

    sendbuf->data_sz = (size_t)r;
    px_queue_append(&c->sendq_head, &sendbuf->bufq);
    pending = BIO_pending(c->ssl_out);
  }

  if (pending < 0)
    return false;

  return true;
}

// @brief write as much data from the sendq buffer to the socket as possible
static nbio_status sendq_to_socket(struct px_connection* c) {
  size_t total_send = 0;
  while (!px_queue_is_singleton(&c->sendq_head)) { // write the buffer to the socket
    struct px_queue* q =  c->sendq_head.next;
    struct px_conn_buffer* sendbuf = px_conn_buffer_from_queue(q);

    update_timespec(&c->stats.last_attempted_send);
    int r = send(c->fd, sendbuf->data + sendbuf->data_offt, sendbuf->data_sz, MSG_NOSIGNAL | MSG_DONTWAIT);
    if (r < 0) {
      int e = errno;
      if (e != EAGAIN && e != EWOULDBLOCK) {
        c->is_open = false;
        if (sendbuf->data_sz > 0)
          px_log_error("%s: could not write to network: %s", c->peer_addr_str, strerror(e));
        errno = e;
        return PX_IO_ERROR;
      }
      r = 0;
    }

    if (r == 0)
      break;

    sendbuf->data_offt += r;
    sendbuf->data_sz -= r;

    if (sendbuf->data_sz == 0) {
      px_queue_extract(&sendbuf->bufq);
      px_conn_buffer_free(sendbuf);
    }

    total_send += r;
    c->stats.total_send += r;
    update_timespec(&c->stats.last_send);
  }

  return total_send > 0 ? PX_IO_OK : PX_IO_AGAIN;
}

// transfer data between the socket and SSL's input buffer
static nbio_status recvq_to_sslin(struct px_connection* c) {
  size_t total_bio_write = 0;
  while (c->recvq.data_sz > 0) {
    int r = BIO_write(c->ssl_in, c->recvq.data + c->recvq.data_offt, c->recvq.data_sz);
    if (r < 0) {
      if (!BIO_should_retry(c->ssl_in)) {
        px_log_error("%s: could not write input data to buffer: %s",
                      c->peer_addr_str, ERR_reason_error_string(ERR_get_error()));
        return PX_IO_ERROR;
      }
      r = 0;
    }

    if (r == 0) {
      // move any un-copied data to the front of the buffer
      if (c->recvq.data_offt > 0) {
        memmove(&c->recvq.data[0], c->recvq.data + c->recvq.data_offt, c->recvq.data_sz);
        c->recvq.data_offt = 0;
      }
      break;
    }

    total_bio_write += r;
    c->recvq.data_sz -= r;
    if (c->recvq.data_sz == 0)
      c->recvq.data_offt = 0;
    else
      c->recvq.data_offt += r;
  }
  return total_bio_write > 0 ? PX_IO_OK : PX_IO_AGAIN;
}

static _Bool socket_to_recvq(struct px_connection* c) {

  // move remaining data to the front of the buffer
  if (c->recvq.data_offt > 0) {
    memmove(&c->recvq.data[0], c->recvq.data + c->recvq.data_offt, c->recvq.data_sz);
    c->recvq.data_offt = 0;
  }

  size_t new_data_start_offt;
  while ((new_data_start_offt = c->recvq.data_offt + c->recvq.data_sz) < sizeof(c->recvq.data)) {
    update_timespec(&c->stats.last_attempted_recv);
    errno = 0;
    int r = recv(c->fd,
                 c->recvq.data + new_data_start_offt,
                 sizeof(c->recvq.data) - new_data_start_offt,
                 MSG_DONTWAIT);

    if (r <= 0) {
      if (r < 0) {
        int e = errno;
        if (e != EAGAIN && e != EWOULDBLOCK) {
          c->is_open = false;
          px_log_error("%s: could not read input data from fd: %s", c->peer_addr_str, strerror(e));
          errno = e;
          return false;
        }
        // EAGAIN/EWOULDBLOCK leave the socket open
      } else { // r == 0 means EOF, no more data
        c->is_open = false;
      }
      r = 0; // no data read, set to zero since we're not dealing with a 'real' error
    }

    c->recvq.data_sz += r;
    if (r == 0)
      break;
    c->stats.total_recv += r;
    update_timespec(&c->stats.last_recv);
  }
  return true;
}

// @brief marshall data from the socket to the ssl input BIO
// @return PX_IO_OK if some data was transferred
//         PX_IO_AGAIN if no data was transferred but the socket is still ok
//         PX_IO_ERROR on error
static nbio_status network_in(struct px_connection* c) {
  if (!socket_to_recvq(c))
    return PX_IO_ERROR;
  return recvq_to_sslin(c);
}

// @brief marshall data from the ssl output BIO to the socket
// @return PX_IO_OK if some data was transferred
//         PX_IO_AGAIN if no data was transferred but the socket is still ok
//         PX_IO_ERROR on error
static nbio_status network_out(struct px_connection* c) {
  if (!sslout_to_sendq(c))
    return PX_IO_ERROR;
  return sendq_to_socket(c);
}

_Bool px_connection_start(struct px_connection* c, int sock, SSL_CTX* initial_ctx) {
  if (sock < 0) {
    px_log_error("invalid file descriptor");
    return false;
  }

  if (!initial_ctx) {
    px_log_warn("no SSL_CTX, this is necessary for connection");
    return false;
  }

  // copy the local and peer addresses to a printable buffer
  if (!px_net_local_addr_to_str(sock, c->host_addr_str, sizeof(c->host_addr_str))
      || !px_net_peer_addr_to_str(sock, c->peer_addr_str, sizeof(c->peer_addr_str)))
  {
    px_log_error("overflow, buffer sizes need adjustment");
    return false;
  }

  if (strlen(c->host_addr_str) >= PX_ADDRSTR_LEN_EXPECTED)
    px_log_error("WARNING: host string is longer than expected (%lu > %lu): %s",
                 (size_t)strlen(c->host_addr_str), (size_t)PX_ADDRSTR_LEN_EXPECTED, c->host_addr_str);

  if (strlen(c->peer_addr_str) >= PX_ADDRSTR_LEN_EXPECTED)
    px_log_error("WARNING: peer string is longer than expected (%lu > %lu): %s",
        (size_t)strlen(c->peer_addr_str), (size_t)PX_ADDRSTR_LEN_EXPECTED, c->peer_addr_str);

  SSL* ssl = SSL_new(initial_ctx);
  if (!ssl) {
    px_log_warn("could not set up ssl: %s", ERR_error_string(ERR_get_error(), NULL));
    return false;
  }

  if (SSL_get_ssl_method(ssl) == TLS_server_method()) {
    SSL_set_accept_state(ssl);
  } else {
    SSL_set_connect_state(ssl);
  }

  BIO* inbio = BIO_new(BIO_s_mem());
  if (!inbio) {
    px_log_error("could not create input BIO: %s", ERR_error_string(ERR_get_error(), NULL));
    SSL_free(ssl);
    return false;
  }

  BIO* outbio = BIO_new(BIO_s_mem());
  if (!outbio) {
    px_log_error("could not create output BIO: %s", ERR_error_string(ERR_get_error(), NULL));
    BIO_free(inbio);
    SSL_free(ssl);
    return false;
  }

  SSL_set_bio(ssl, inbio, outbio);

  c->fd = sock;
  c->is_open = true;
  c->ssl = ssl;
  c->ssl_in = inbio;
  c->ssl_out = outbio;

  // TODO SSL_set_app_data(c->ssl, c);

  // initialize the statistics timestamps
  update_timespec(&c->stats.last_queue);
  update_timespec(&c->stats.last_sslread);
  update_timespec(&c->stats.last_sslwrite);
  update_timespec(&c->stats.last_recv);
  update_timespec(&c->stats.last_send);

  return true;
}

_Bool px_connection_set_nonblocking(struct px_connection* c) {
  int flags = fcntl(c->fd, F_GETFL);
  if (flags < 0 || fcntl(c->fd, F_SETFL, flags | O_NONBLOCK) != 0) {
    int e = errno;
    px_log_error("could not set fd %d to nonblocking: %s", c->fd, strerror(e));
    errno = e;
    return false;
  }
  return true;
}

_Bool px_connection_outqueue_is_empty(struct px_connection const* c) {
  return px_queue_is_singleton(&c->outq_head);
}

size_t px_connection_queue_string(struct px_connection* c, char const* str) {
  size_t slen = str ? strlen(str) : 0;
  return px_connection_queue_write(c, str, slen);
}

size_t px_connection_queue_write(struct px_connection* c, char const* buf, size_t buf_sz) {
  size_t total_queued = 0;
  while (buf_sz > 0) {
    struct px_conn_buffer* cb = px_conn_buffer_get();
    if (!cb) // no memory
      return total_queued;

    size_t to_copy = buf_sz <= sizeof(cb->data) ? buf_sz : sizeof(cb->data);
    memcpy(&cb->data[0], buf, to_copy);
    cb->data_sz = to_copy;
    buf += to_copy;
    buf_sz -= to_copy;
    if (!px_connection_queue_buffer(c, cb)) {
      px_conn_buffer_free(cb);
      break;
    }
    total_queued += to_copy;
  }

  return total_queued;
}

_Bool px_connection_queue_buffer(struct px_connection* c, struct px_conn_buffer* buf) {
  if (!c->ssl || !c->is_open) {
    px_log_error("connection not active, cannot queue anything");
    return false;
  }

  // if we're shutting down then don't add the data
  int status = SSL_get_shutdown(c->ssl);
  if ((status & (SSL_SENT_SHUTDOWN | SSL_RECEIVED_SHUTDOWN)) != 0)
      return false;

  px_log_assert(px_queue_is_singleton(&buf->bufq), "programming error");
  if (buf->data_offt != 0) {
    px_log_error("WARNING: buffer data should be reset here");
    buf->data_offt = 0;
  }

  // for empty buffers discard the buffer but update the queue stamp.
  // we wouldn't do anything with the data anyway
  if (buf->data_sz == 0) {
    update_timespec(&c->stats.last_queue);
    px_conn_buffer_free(buf);
    return true;
  }

  px_queue_append(&c->outq_head, &buf->bufq);

  c->stats.total_queued += buf->data_sz;
  update_timespec(&c->stats.last_queue);
  return true;
}

void clear_sendq(struct px_connection* c) {
  while (!px_queue_is_singleton(&c->sendq_head)) {
    struct px_queue* q = px_queue_pop_front(&c->sendq_head);
    struct px_conn_buffer* cb = px_conn_buffer_from_queue(q);
    px_conn_buffer_free(cb);
  }
}

void px_connection_clear_outqueue(struct px_connection* c) {
  while (!px_queue_is_singleton(&c->outq_head)) {
    struct px_queue* q = px_queue_pop_front(&c->outq_head);
    struct px_conn_buffer* txbuf = px_conn_buffer_from_queue(q);
    px_conn_buffer_free(txbuf);
  }
}

_Bool px_connection_needs_send(struct px_connection const* c) {
  if (!c->ssl)
    return false;

  return !px_queue_is_singleton(&c->outq_head)
         || !px_queue_is_singleton(&c->sendq_head)
         || BIO_pending(c->ssl_out) > 0;
}

_Bool px_connection_needs_recv(struct px_connection const* c) {
  if (!c->ssl)
    return false;

  return c->recvq.data_sz > 0 || SSL_pending(c->ssl) > 0 || BIO_pending(c->ssl_in) > 0;
}

unsigned px_connection_sendqueue_length(struct px_connection* c) {
  unsigned len = 0;
  struct px_queue* q = c->outq_head.next;
  while (q != &c->outq_head) {
    struct px_conn_buffer* txbuf = px_conn_buffer_from_queue(q);
    q = q->next;
    unsigned new_len = len + txbuf->data_sz;
    if (new_len < len) // overflow
      return (unsigned)-1;
    len  = new_len;
  }

  return len;
}

static nbio_status tls_handshake(struct px_connection* c) {
  if (!c->ssl || !c->is_open)
    return PX_IO_ERROR;

  if (!SSL_in_init(c->ssl))
    return PX_IO_OK;

  while (true) {
    ERR_clear_error();
    int r = SSL_do_handshake(c->ssl);
    if (r == 1) {
      return PX_IO_OK;
    }

    int e = SSL_get_error(c->ssl, r);
    switch (e) {
    case SSL_ERROR_WANT_READ :
      {
        if (px_connection_needs_send(c)) {
          nbio_status out_status = network_out(c);
          if (out_status == PX_IO_ERROR)
            return out_status;
        }

        nbio_status in_status = network_in(c);
        if (in_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLIN;
        else
          c->wait_on_network &= ~POLLIN;

        if (in_status == PX_IO_OK)
          continue;
        return in_status;
      }
    case SSL_ERROR_WANT_WRITE :
      {
        nbio_status out_status = network_out(c);
        if (out_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLOUT;
        else
          c->wait_on_network &= ~POLLOUT;

        if (out_status == PX_IO_OK)
          continue;
        return out_status;
      }
    default :
      for (unsigned long err = ERR_get_error(), depth = 0; err != 0; err = ERR_get_error(), ++depth)
        px_log_info("%s: SSL error during handshake %lu: %s",
            c->peer_addr_str, depth, ERR_error_string(err, NULL));
      return PX_IO_ERROR;
    }
    break;
  }
}

static _Bool conn_buffer_to_ssl(struct px_connection* c, struct px_conn_buffer* buf) {
  if (buf->data_sz == 0)
    return true;

  const size_t max_data_sz = PX_TLS_MAX_PLAINTEXT_SZ;
#if defined(SSL_SESSION_get_max_fragment_length) // needed because libressl doesn't have this
  // if the client requested smaller packets then use the smaller size (as long
  // as it doesn't drop below the arbitrary minimum packet threshold)
  SSL_SESSION* sess = SSL_get_session(gemctx->conn.ssl);
  if (sess) {
    long issl_frag_sz = SSL_SESSION_get_max_fragment_length(sess);
    size_t ssl_frag_sz = issl_frag_sz >= 0 ? issl_frag_sz : 0;
    max_data_sz = ssl_frag_sz >= max_data_sz ? max_data_sz : ssl_frag_sz;
  }
#endif // defined(SSL_SESSION_get_max_fragment_length)

  while (buf->data_sz > 0) {
    ERR_clear_error();

    // SSL packets have a maximum transmission size, write it to the SSL in
    // compatible sized chunks
    size_t out_sz = (buf->data_sz <= max_data_sz) ? buf->data_sz : max_data_sz;
    int r = SSL_write(c->ssl, buf->data + buf->data_offt, out_sz);
    if (r > 0) {
      c->stats.total_sslwrite += r;
      update_timespec(&c->stats.last_sslwrite);
      buf->data_offt += r;
      buf->data_sz -= r;
      nbio_status nbr = network_out(c);
      if (nbr != PX_IO_OK) {
        if (nbr == PX_IO_AGAIN)
          c->wait_on_network |= POLLOUT;
        break;
      }
      continue;
    }

    int e = SSL_get_error(c->ssl, r);
    switch (e) {
    case SSL_ERROR_NONE : // wat?
      break;
    case SSL_ERROR_WANT_READ :
    {
      if (px_connection_needs_send(c)) {
        nbio_status out_status = network_out(c);
        if (out_status == PX_IO_ERROR)
          return false;
      }

      nbio_status in_status = network_in(c);
      if (in_status == PX_IO_AGAIN)
        c->wait_on_network |= POLLIN;
      else
        c->wait_on_network &= ~POLLIN;

      if (in_status == PX_IO_OK)
        continue;
      return (in_status != PX_IO_ERROR);
    }
    case SSL_ERROR_WANT_WRITE :
    {
      nbio_status out_status = network_out(c);
      if (out_status == PX_IO_AGAIN)
        c->wait_on_network |= POLLOUT;
      else
        c->wait_on_network &= ~POLLOUT;

      if (out_status == PX_IO_OK)
        continue;
      return (out_status != PX_IO_ERROR);
    }
    default :
      for (unsigned long err = ERR_get_error(), depth = 0; err != 0; err = ERR_get_error(), ++depth)
        px_log_info("%s: SSL error during send %lu: %s", c->peer_addr_str, depth, ERR_error_string(err, NULL));
      return false;
    }
  }
  return true;
}

static _Bool outq_to_ssl(struct px_connection* c) {
  while (!px_queue_is_singleton(&c->outq_head)) {
    struct px_queue* q = c->outq_head.next;
    struct px_conn_buffer* cb = px_conn_buffer_from_queue(q);

    if (!conn_buffer_to_ssl(c, cb))
      return false;

    if (cb->data_sz > 0)
      return true;

    if (cb->data_sz == 0) {
      px_queue_extract(&cb->bufq);
      px_conn_buffer_free(cb);
    }
  }
  return true;
}

// send queued data
px_op_status px_connection_send(struct px_connection* c) {
  if (!c->ssl || !c->is_open)
    return PX_OP_ERROR;

  if (SSL_in_init(c->ssl)) {
    nbio_status r = tls_handshake(c);
    if (r != PX_IO_OK)
      return r == PX_IO_AGAIN ? PX_OP_RETRY : PX_OP_ERROR;
  }

  if (!outq_to_ssl(c))
    return PX_OP_ERROR;

  if (network_out(c) == PX_IO_ERROR)
    return PX_OP_ERROR;

  return px_connection_needs_send(c) ? PX_OP_RETRY : PX_OP_DONE;
}

_Bool px_connection_discard_readable(struct px_connection* c) {

  if (!px_connection_needs_recv(c))
    return false;

  struct px_conn_buffer* tempcb = px_conn_buffer_get();
  if (!tempcb)
    return false;

  size_t buf_sz = sizeof(tempcb->data);
  (void)px_connection_recv(c, &tempcb->data[0], &buf_sz);
  _Bool did_read = (buf_sz > 0);
  px_conn_buffer_free(tempcb);
  return did_read;
}

px_op_status  px_connection_do_handshake(struct px_connection* conn) {
  if (!conn->ssl)
    return PX_OP_ERROR;

  nbio_status r = tls_handshake(conn);
  switch (r) {
  case PX_IO_AGAIN :
    return PX_OP_RETRY;
  case PX_IO_OK :
    return PX_OP_DONE;
  case PX_IO_ERROR :
  default :
    return PX_OP_ERROR;
  }
}

px_op_status px_connection_recv(struct px_connection* c, uint8_t* buf, size_t* buf_sz) {

  size_t full_buf_sz = *buf_sz;
  *buf_sz = 0;

  // we don't check is_open here because we may be able to read something from
  // the buffered data
  if (!c->ssl) {
    px_log_error("ssl context not set up");
    return PX_OP_ERROR;
  }

  if (SSL_in_init(c->ssl)) {
    nbio_status r = tls_handshake(c);
    if (r != PX_IO_OK)
      return (r == PX_IO_AGAIN) ? PX_OP_RETRY : PX_OP_ERROR;
  }

  while (true) {
    ERR_clear_error();
    int r = SSL_read(c->ssl, buf, full_buf_sz);
    if (r > 0) {
      c->stats.total_sslread += r;
      update_timespec(&c->stats.last_sslread);
      *buf_sz = (size_t)r;
      return PX_OP_DONE;
    }

    int e = SSL_get_error(c->ssl, r);
    switch (e) {
      case SSL_ERROR_NONE : // wat?
      case SSL_ERROR_ZERO_RETURN : // zero return == closed connection
        return PX_OP_RETRY;
      case SSL_ERROR_WANT_READ :
      {
        if (px_connection_needs_send(c)) {
          nbio_status out_status = network_out(c);
          if (out_status == PX_IO_ERROR)
            return PX_OP_ERROR;
        }

        nbio_status in_status = network_in(c);
        if (in_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLIN;
        else
          c->wait_on_network &= ~POLLIN;

        if (in_status == PX_IO_OK) // some data is queued for ssl read
          continue;
        return (in_status == PX_IO_ERROR) ? PX_OP_ERROR : PX_OP_RETRY;
      }
      case SSL_ERROR_WANT_WRITE :
      {
        nbio_status out_status = network_out(c);
        if (out_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLOUT;
        else
          c->wait_on_network &= ~POLLOUT;

        // note: this check/branch was left out for a while.  I'm not sure if that was
        // on purpose or if it was an oversight
        if (out_status == PX_IO_OK)
          continue;
        return (out_status == PX_IO_ERROR) ? PX_OP_ERROR : PX_OP_RETRY;
      }
      default :
        for (unsigned long err = ERR_get_error(), depth = 0; err != 0; err = ERR_get_error(), ++depth)
          px_log_info("SSL error during send %lu: %s", depth, ERR_error_string(err, NULL));
    }
    break;
  }
  return PX_OP_ERROR;
}

static inline _Bool close_notify_finished(SSL* ssl) {
  int status = SSL_get_shutdown(ssl);
  return (status & SSL_SENT_SHUTDOWN) && (status & SSL_RECEIVED_SHUTDOWN);
}

_Bool px_connection_sent_shutdown(struct px_connection const* c) {
  if (!c || !c->ssl)
    return true;
  int status = SSL_get_shutdown(c->ssl);
  return (status & SSL_SENT_SHUTDOWN) != 0;
}

_Bool px_connection_received_shutdown(struct px_connection const* c) {
  if (!c || !c->ssl)
    return true;
  int status = SSL_get_shutdown(c->ssl);
  return (status & SSL_RECEIVED_SHUTDOWN) != 0;
}

_Bool px_connection_is_closed(struct px_connection const* c) {
  return c->fd < 0 || !c->is_open;
}

px_op_status px_connection_shutdown(struct px_connection* c) {
  // we may have closed the socket, but the SSL will still be around
  if (!c->ssl || !c->is_open)
    return PX_OP_ERROR;

  if (close_notify_finished(c->ssl))
    return PX_OP_DONE;

  px_connection_clear_outqueue(c);

  if (SSL_in_init(c->ssl)) {
    nbio_status hs = tls_handshake(c);
    if (hs != PX_IO_OK)
      return hs == PX_IO_AGAIN ? PX_OP_RETRY : PX_OP_ERROR;
  }

  while (true) {
    ERR_clear_error();
    int r = SSL_shutdown(c->ssl);
    if (r == 1) { // ssl shutdown complete
      return PX_OP_DONE;
    } else if (r == 0) { // ssl shutdown hasn't completed yet
      if (px_connection_needs_send(c)) {
        nbio_status out_status = network_out(c);
        if (out_status == PX_IO_ERROR)
          return PX_OP_ERROR;
      }

      nbio_status in_status = network_in(c);
      if (in_status == PX_IO_OK)
        continue;
      return (in_status == PX_IO_ERROR) ? PX_OP_ERROR : PX_OP_RETRY;
    } else { // r < 0, real error

      // error.  why?
      int e = SSL_get_error(c->ssl, r);
      switch (e) {
      case SSL_ERROR_WANT_READ :
      {
        if (px_connection_needs_send(c)) {
          nbio_status out_status = network_out(c);
          if (out_status == PX_IO_ERROR)
            return PX_OP_ERROR;
        }

        nbio_status in_status = network_in(c);
        if (in_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLIN;
        else
          c->wait_on_network &= ~POLLIN;

        if (in_status == PX_IO_OK)
          continue;
        return (in_status == PX_IO_ERROR) ? PX_OP_ERROR : PX_OP_RETRY;
      }
      case SSL_ERROR_WANT_WRITE :
      {
        nbio_status in_status = network_in(c);
        if (in_status == PX_IO_ERROR)
          return PX_OP_ERROR;

        nbio_status out_status = network_out(c);
        if (out_status == PX_IO_AGAIN)
          c->wait_on_network |= POLLOUT;
        else
          c->wait_on_network &= ~POLLOUT;

        if (out_status == PX_IO_OK)
          continue;
        return (out_status == PX_IO_ERROR) ? PX_OP_ERROR : PX_OP_RETRY;
      }
      default :
        for (unsigned long err = ERR_get_error(), depth = 0; err != 0; err = ERR_get_error(), ++depth)
          px_log_info("SSL error during shutdown %lu: %s", depth, ERR_error_string(err, NULL));
      }
    }
    break;
  }
  return PX_OP_ERROR;
}

void px_connection_close(struct px_connection* c) {
  // NOTE: we keep the SSL around so that we can get data from it if needed
  if (c->fd >= 0) {
    if (close(c->fd) < 0)
      px_log_warn("could not close fd %d: %s", c->fd, strerror(errno));
    c->fd = -1;
    c->is_open = false;
  }
  c->wait_on_network = 0;
}

_Bool px_connection_waiting_on_network(struct px_connection const* c) {
  return px_connection_needs_send(c) || c->wait_on_network != 0;
}

int px_connection_get_poll_events(struct px_connection const* c) {
  int events = POLLIN;
  // is there something that's ready to be sent over the wire?
  if (px_connection_needs_send(c))
    events |= POLLOUT;
  return events;
}

