#ifndef PX_QUEUE_H__
#define PX_QUEUE_H__

#include <stddef.h>
#include <stdint.h>

#ifndef container_of
#define container_of(ptr, type, member) ((type*)((intptr_t)ptr - offsetof(type, member)))
#endif // container_of

#define PX_QUEUE_INITIALIZER(ptr) { .next = (ptr), .prev = (ptr) }

// a circular queue.  a single struct px_queue should

struct px_queue {
  struct px_queue*  next;
  struct px_queue*  prev; // pointer to the previous entry's 'next' item
};

static inline void px_queue_init(struct px_queue* q) {
  *q = (struct px_queue) PX_QUEUE_INITIALIZER(q);
}

static inline _Bool px_queue_is_singleton(struct px_queue const* q) {
  return q->next == q;
}

static inline void px_queue_extract(struct px_queue* q) {
  q->prev->next = q->next;
  q->next->prev = q->prev;
  px_queue_init(q);
}

static inline void px_queue_insert_before(struct px_queue* target, struct px_queue* to_insert) {
  to_insert->next = target;
  to_insert->prev = target->prev;
  target->prev->next = to_insert;
  target->prev = to_insert;
}

static inline void px_queue_insert_after(struct px_queue* target, struct px_queue* to_insert) {
  to_insert->prev = target;
  to_insert->next = target->next;
  target->next->prev = to_insert;
  target->next = to_insert;
}

static inline void px_queue_append(struct px_queue* q_head, struct px_queue* to_insert) {
  px_queue_insert_before(q_head, to_insert);
}

static inline void px_queue_prepend(struct px_queue* q_head, struct px_queue* to_insert) {
  px_queue_insert_after(q_head, to_insert);
}

static inline unsigned px_queue_size(struct px_queue const* head) {
  if (!head)
    return 0;

  unsigned sz = 0;
  struct px_queue const* q = head->next;
  while (q != head) {
    ++sz;
    q = q->next;
  }
  return sz;
}

// appends the entries in src_head to dst_head
static inline void px_queue_splice(struct px_queue* dst_head, struct px_queue* src_head) {
  if (px_queue_is_singleton(src_head))
    return;

  dst_head->prev->next  = src_head->next;
  src_head->next->prev  = dst_head->prev;
  dst_head->prev        = src_head->prev;
  dst_head->prev->next  = dst_head;
  px_queue_init(src_head);
}

static inline struct px_queue* px_queue_pop_front(struct px_queue* head) {
  if (px_queue_is_singleton(head))
    return NULL;
  struct px_queue* q = head->next;
  px_queue_extract(q);
  return q;
}

static inline struct px_queue* px_queue_pop_back(struct px_queue* head) {
  if (px_queue_is_singleton(head))
    return NULL;
  struct px_queue* q = head->prev;
  px_queue_extract(q);
  return q;
}

static inline struct px_queue* px_queue_front(struct px_queue* head) {
  return head->next;
}

static inline struct px_queue* px_queue_back(struct px_queue* head) {
  return head->prev;
}

#endif // PX_QUEUE_H__
