#ifndef PX_EVENT_H__
#define PX_EVENT_H__

#include <libpxd/px_time.h>

#include <stdbool.h>
#include <libpxd/px_queue.h>

struct px_event;

#define PX_EVENT_DISABLE_TIMEOUT 0
#define PX_EVENT_ENABLE_TIMEOUT 1
#define PX_EVENT_IMMEDIATE_TIMEOUT (struct timespec) { .tv_sec = (time_t)-1, .tv_nsec = LONG_MIN }

enum px_event_flag {
  PX_EVENT_NONE      = 0,
  PX_EVENT_READ      = 0x01,
  PX_EVENT_WRITE     = 0x02,
  PX_EVENT_SHUTDOWN  = 0x04,
  PX_EVENT_ERROR     = 0x08,
  PX_EVENT_HUP       = 0x10
};

#define px_event_from_queue(q) container_of(q, struct px_event, eventq)

typedef void (*px_event_fxn_on_event)    (int fd, int revents, void* data);
typedef void (*px_event_fxn_on_timeout)  (void* data);
typedef void (*px_event_fxn_on_dequeue)  (void* data);

struct px_event {
  struct px_queue           eventq;

  int             fd; // file descriptor to wait on.  <0 indicates no fd
  int             events; // events to wait for on fd
  int             revents; // received events
  _Bool           has_timeout; // if nonzero then timeout_end represents the event timeout point
  struct timespec timeout_end; // monotonic clock time for the end of the event

  // called when an event happens on a file descriptor
  // fd - the file descriptor which has an event value
  // revents - a bitwise-or of the px_event_flag values that are relevant
  // data - the value of priv
  px_event_fxn_on_event   on_event;

  // called when a timeout occurs
  px_event_fxn_on_timeout on_timeout;

  // called when dequeued without a timeout or event happening
  // e.g. on eventq shutdown
  px_event_fxn_on_dequeue on_dequeue;

  void* priv; // private data, passed to event callbacks
};

void px_event_init(struct px_event* ev); // for new events
void px_event_reset(struct px_event* ev); //
void px_event_setup(struct px_event* ev,
                    int fd,
                    int events,
                    px_event_fxn_on_event event_fxn,
                    int timeout,
                    px_event_fxn_on_timeout timeout_fxn,
                    px_event_fxn_on_dequeue dequeue_fxn,
                    void* data);

void px_event_set_timeout(struct px_event* ev, int ms);
void px_event_set_immediate_timeout(struct px_event* ev);
void px_event_disable_timeout(struct px_event* ev);
void px_event_clear_timeout(struct px_event* ev);

_Bool px_event_has_fd(struct px_event const* ev);
_Bool px_event_has_timeout(struct px_event const* ev);
_Bool px_event_timeout_expired(struct px_event const* ev, struct timespec const* now);

void px_event_enqueue(struct px_event* ev, struct px_queue* head); // move an event onto a queue
void px_event_dequeue(struct px_event* ev); // remove an event from a queue

void px_event_trigger_event(struct px_event* ev, int revents); // trigger the event callback
void px_event_trigger_timeout(struct px_event* ev); // trigger the timeout callback
void px_event_trigger_dequeue(struct px_event* ev); // remove an event from its queue, triggering its callback

void px_eventq_clear(struct px_queue* eventq);

void px_dbg_check_queue(struct px_queue* queue);

#endif // PX_EVENT_H__
