#ifndef PX_CONNECTION_H__
#define PX_CONNECTION_H__

#include <libpxd/px_common.h>
#include <libpxd/px_network.h>
#include <libpxd/px_queue.h>
#include <stdlib.h>
#include <openssl/ssl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netdb.h>

#define px_conn_buffer_from_queue(q) container_of(q, struct px_conn_buffer, bufq)
#define PX_BUFFER_FREELIST_HIGH_WATERMARK 32

#ifndef PX_NO_POLL
#include <poll.h>
#endif // PX_NO_POLL

#ifndef POLLIN
#define POLLIN 0x01
#endif // POLLIN

#ifndef POLLOUT
#define POLLOUT 0x04
#endif // POLLOUT

// a buffer for queueing data to a px_connection
struct px_conn_buffer {
  struct px_queue bufq;       // queue entry for storing the queued data
  size_t          data_offt;  // offset to beginning of data
  size_t          data_sz;    // length of data
  uint8_t         data[PX_TLS_MAX_PACKET_SZ]; // data
};

// @brief get a pointer to a connection buffer
// @return a valid pointer, or null on out-of-memory
struct  px_conn_buffer* px_conn_buffer_get(void);

// @brief free a connection buffer allocated with px_conn_buffer_get()
void                    px_conn_buffer_free(struct px_conn_buffer* buf);

//! @brief split a connection buffer into max_frag_sz pieces
//! @param dst_head the queue head to place the split buffers onto
//! @param buf the input buffer.  on failure buf will be unmodified.
//! @param max_frag_sz maximum size of each buffer
//! @return true on success, buf is no longer valid, dst_head has the split buffers appended to it.
//          false on failure, buf is unmodified, dst_head is unmodified
_Bool                   px_conn_buffer_split(struct px_queue* dst_head,
                                             struct px_conn_buffer* buf,
                                             size_t max_frag_sz);

struct px_connection {
  int                       fd;       ///< socket
  _Bool                     is_open;  ///< whether socket is open (note: may be false and fd >= 0)
  SSL*                      ssl;      ///< the tls state

  // we put an intermediate layer of memory BIOs between the socket and the SSL object so that we
  // can call send with MSG_NOSIGNAL so that we do not have to mask SIGPIPE from the whole process
  // (this is a library and we try to be as unobtrusive as possible, after all)
  BIO*                      ssl_in;   ///< BIO which buffers input from socket, owned by 'ssl'
  BIO*                      ssl_out;  ///< BIO which buffers output from socket, owned by 'ssl'

  // we use sockaddr_storage to make sure we have enough space for any protocol we'd use
  struct sockaddr_storage   host_ss; ///< sockaddr for the server socket's address
  char                      host_addr_str[PX_ADDRSTR_LEN]; ///< human-readable address <IP>:<PORT>\0

  struct sockaddr_storage   peer_ss; ///< sockaddr for the peer socket's address
  char                      peer_addr_str[PX_ADDRSTR_LEN]; ///< human-readable address <IP>:<PORT>\0

  struct {
    struct timespec           last_queue;   ///< the last time data was queued for write
    size_t                    total_queued; ///< how many plaintext bytes were queued

    struct timespec           last_sslwrite;  ///< the last time >0 bytes were written via SSL_write
    size_t                    total_sslwrite; ///< total bytes written via SSL_write

    struct timespec           last_attempted_send;  ///< the last time a send() was attempted
    struct timespec           last_send;  ///< the last time >0 bytes were sent over the socket
    size_t                    total_send; ///< total bytes transmitted via send

    struct timespec           last_attempted_recv;  ///< the last time a recv() was attempted
    struct timespec           last_recv;  ///< the last time >0 bytes were received on the socket
    size_t                    total_recv; ///< total number of bytes received on the socket

    struct timespec           last_sslread;   ///< the last time >0 plaintext bytes were read from the SSL
    size_t                    total_sslread;  ///< total bytes that have been read from SSL
  } stats;

  int                       wait_on_network;

  struct px_queue           outq_head; ///< pre-TLS outbound data

  // queues with TLS-encrypted data
  struct px_conn_buffer     recvq; ///< TLS-encrypted data read from socket
  struct px_queue           sendq_head; ///< post-TLS outbound data to be written to socket
};

void          px_connection_init(struct px_connection* c);
void          px_connection_reset(struct px_connection* c);

_Bool         px_connection_start(struct px_connection* c, int s, SSL_CTX* initial_ctx);

px_op_status  px_connection_do_handshake(struct px_connection* conn);
px_op_status  px_connection_recv(struct px_connection* conn, uint8_t* buf, size_t* buf_sz);
_Bool         px_connection_discard_readable(struct px_connection* c);

_Bool         px_connection_outqueue_is_empty(struct px_connection const* c);
size_t        px_connection_queue_string(struct px_connection* c, char const* str);
size_t        px_connection_queue_write(struct px_connection* c, char const* buf, size_t buf_sz);
_Bool         px_connection_queue_buffer(struct px_connection* c, struct px_conn_buffer* buf);
void          px_connection_clear_outqueue(struct px_connection* c);

_Bool         px_connection_needs_send(struct px_connection const* c);
_Bool         px_connection_needs_recv(struct px_connection const* c);
unsigned      px_connection_sendqueue_length(struct px_connection* c);
px_op_status  px_connection_send(struct px_connection* conn);


px_op_status  px_connection_shutdown(struct px_connection* c);
void          px_connection_close(struct px_connection* c);

_Bool         px_connection_sent_shutdown(struct px_connection const* c);
_Bool         px_connection_received_shutdown(struct px_connection const* c);
_Bool         px_connection_is_closed(struct px_connection const* c);

_Bool         px_connection_waiting_on_network(struct px_connection const* c);

int           px_connection_get_poll_events(struct px_connection const* c);

#endif // PX_CONNECTION_H__
