#include <libpxd/px_gemini_ctx.h>
#include <libpxd/px_network.h>
#include <libpxd/px_log.h>
#include <libpxd/px_listen.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <poll.h>

char const* keyfile = NULL;
char const* certfile = NULL;

void usage(char const* progname) {
  px_log_info("usage: %s <keyfile> <certfile> <file to serve> <mimetype>\n"
              "  e.g. %s key.pem cert.pem myfile.gmi text/gemini", progname, progname);
}

_Bool print_sni (struct px_gemini_context* gemctx, char const* sni_name) {
  if (sni_name)
    px_log_info("%s: client requested host %s", gemctx->conn.peer_addr_str, sni_name);
  else
    px_log_info("%s: client did not send sni", gemctx->conn.peer_addr_str);

  return true;
}

_Bool print_request_url (struct px_gemini_context* gemctx) {
  struct px_url encoded = { 0 };

  if (px_url_encode(&encoded, &gemctx->request.url)) {
    char* str = px_url_to_str(&encoded);
    if (str) {
      px_log_info("%s: request url: %s", gemctx->conn.peer_addr_str, str);
    } else {
      px_log_info("%s: could not decode requested url\n", gemctx->conn.peer_addr_str);
    }
    free(str);
    px_url_reset(&encoded);
  }
  return true;
}

_Bool fixed_response(struct px_gemini_context* gemctx, uint8_t*  resp_buf, size_t* resp_buf_sz) {
  //size_t orig_resp_buf_sz = *resp_buf_sz;
  *resp_buf_sz = 0;

  char const* responses[] = {
    "20 text/gemini\r\n",
    "Hello, world!\n",
    "We are only sending a single output,\n",
    "but we do it line by line\n",
    "as a demonstration\n",
    NULL
  };

  char const** txt = &responses[0];
  size_t skip = gemctx->conn.stats.total_queued;
  while (*txt != NULL && skip > 0) {
    size_t txt_sz = strlen(*txt);
    if (txt_sz <= skip) {
      skip -= txt_sz;
      ++txt;
      continue;
    }
  }

  if (*txt) {
    // this is always going to be true, resp_buf_sz is 16k
    memcpy(resp_buf, *txt, strlen(*txt));
    *resp_buf_sz = strlen(*txt);
  }
  return *(txt + 1) ? true : false;
}

static void print_connection_term(struct px_gemini_context* gemctx) {
  px_log_info("%s: closing connection, rx %lu, tx %lu bytes",
              gemctx->conn.peer_addr_str,
              gemctx->conn.stats.total_recv,
              gemctx->conn.stats.total_send);
}

int main(int argc, char* argv[]) {
  if (argc < 3) {
    usage(argv[0]);
    return -1;
  }

  keyfile = argv[1];
  certfile = argv[2];

  int f = px_bind("localhost", NULL, "1965");
  if (f < 0) {
    px_log_error("could not bind localhost port 1965");
    return -1;
  }

  SSL_CTX* sslctx = px_net_ssl_ctx();
  if (!sslctx || !px_net_use_ssl_cert(sslctx, certfile, keyfile)) {
    px_log_error("couldn't set up the SSL_CTX");
    close(f);
    return -1;
  }

  px_gemini_context_set_default_sslctx(sslctx);

  // this defines our gemini exchange behavior
  struct px_gemini_callbacks cbs = {
    .on_sni = print_sni,
    .on_request_complete = print_request_url,
    //.on_response_send = fixed_response,
  };

  while (1) {
    struct pollfd pfd = { .fd = f, .events = POLLIN, .revents = 0 };
    int r = poll(&pfd, 1, -1);
    if (r < 0) {
      int e = errno;
      px_log_info("error while polling fd %d: %s", f, strerror(e));
      break;
    }

    if (r == 0) {
      px_log_info("timeout on poll");
      continue;
    }

    int s = accept(f, NULL, 0);
    if (s < 0) {
      int e = errno;
      px_log_error("could not accept: %s", strerror(e));
      continue;
    }

    struct px_gemini_context gemctx;
    px_gemini_context_init(&gemctx);

    _Bool setup_ok = px_gemini_context_setup_exchange(&gemctx,
                                                      s,
                                                      NULL /* or: sslctx */,
                                                      &cbs,
                                                      NULL);
    if (!setup_ok) {
      px_log_error("could not start gemini exchange");
      close(s);
      continue;
    }

    px_log_info("connected %s", gemctx.conn.peer_addr_str);

    while (px_gemini_context_iterate(&gemctx) != PX_OP_ERROR) {
      if (px_gemini_context_is_finished(&gemctx))
        break;

      if (px_connection_outqueue_is_empty(&gemctx.conn)) {
        if (px_gemini_context_needs_data(&gemctx)) {
          struct px_conn_buffer* cb = px_conn_buffer_get();
          if (!cb)
            break;

          size_t data_sz = PX_TLS_MAX_PLAINTEXT_SZ;
          _Bool more_data = fixed_response(&gemctx, &cb->data[0], &data_sz);
          cb->data_sz = data_sz;

          if (data_sz > 0) {
            (void)px_gemini_context_queue_data(&gemctx, cb);
          }

          if (!more_data)
            px_gemini_context_end_data(&gemctx);
        }
      }

      int timeout = px_gemini_context_waiting_on_network(&gemctx) ? -1 : 0;
      if (px_gemini_context_wait_for_network(&gemctx, timeout) == PX_OP_ERROR) {
          px_log_error("error while waiting for network");
        break;
      }
    }
    print_connection_term(&gemctx);
    px_gemini_context_reset(&gemctx);
  }

  close(f);
  return 0;
}
