#!/bin/sh

set -o pipefail

filter_newline() {
  tr -d '\n'
}

write_version() {
  # check if we're in a git repo.  if not then we can't get the version info
  # and should use whatever is already there (hopefully it's still there)
  if ! git status 2>/dev/null >&2 ; then
    echo "Could not get version info from git" >&2
    return 1
  fi

  # first check if any tracked files have been modified.
  # if so we're on a development branch of a commit
  if [ "`git status -uno --porcelain 2>/dev/null | wc -l 2>/dev/null`" -ne 0 ] ; then
    echo "development version based on " | filter_newline

    # we need to strip the newline off of the end of this output so that it doesn't show up in the header
    git rev-list --date=short --no-commit-header --max-count=1 \
      --pretty="format:revision %h " HEAD 2>/dev/null | filter_newline
    date +"%Y-%m-%d" | filter_newline

  else # <- no files have changed, so we can use a tag or un-munged revision
    # check if there's a tag associated with this commit, and use it for the version if so
    TAG="`git tag --contains HEAD 2>/dev/null`"
    if [ -n "$TAG" ] ; then # <- tag exists on our commit, cool
      echo "version $TAG" | filter_newline
    else # <- no tag associated with commit, use revision and date
      # we're sitting on an unmodified commit, use its revision
      if ! git rev-list --date=short --no-commit-header --max-count=1 \
        --pretty="format:revision %h %cd" HEAD 2>/dev/null | filter_newline ; then
            echo "unknown version" | filter_newline
      fi
    fi
  fi
  return 0
}

VERSION_HEADER="${1:-include/libpxd/px_version.h}"

tmp="`mktemp ./.px_version.h.XXXXXXXX`"
version=$(write_version)
if [ "$?" -ne "0" ] ; then
  echo "Not updating version information"
  rm -f "$tmp"
  exit 0
fi

# write out the px_version.h header
cat <<EOF > "$tmp"
#ifndef PX_VERSION_H__
#define PX_VERSION_H__

#define LIBPXD_VERSION "$version"

#endif // PXC_VERSION_H__
EOF

if [ ! -f "$VERSION_HEADER" ] || ! cmp "$VERSION_HEADER" "$tmp" >/dev/null 2>&1 ; then
  echo "Updating version info to: $version"
  mv "$tmp" "$VERSION_HEADER"
else
  rm -f "$tmp"
fi
