#include <libpxd/px_route.h>
#include <libpxd/px_log.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

void print_node(struct px_route_node* node, int depth) {
  char spaces[32] = { 0 };
  memset(spaces, ' ', sizeof(spaces) - 1);
  int spaces_sz = sizeof(spaces);
  if (depth > spaces_sz / 2 - 1)
    depth = spaces_sz / 2 - 1;
  if (depth < 0)
    depth = 0;

  spaces[2 * depth] = '\0';
  fprintf(stdout, "%s%s : %lu\n", spaces, node->path ? node->path : "(no path)", (uintptr_t)node->priv);
}

int main(void) {

  struct route_add_test_data { char const* path; _Bool excl; _Bool should_succeed; };

  struct route_add_test_data add_tests[] = {
    { "/", false, true },
    { "/depth1", false, true },
    { "/depth1/depth2.1", false, true },
    { "/depth1/depth2.2", false, true },
    { "/depth1/depth2.2/depth3", false, true },
    { "/depth1/depth2.2/depth3", true , false },
    { "/depth1/depth2.2/depth3", false, true },
    { NULL, false, false },
    { "", false, false },
    { "/some/sub/path/that/has/a/bunch/of/extra/components", false, true },
    { "dont/add/relative/paths", false, false }
  };
  unsigned const n_add_tests = sizeof(add_tests) / sizeof(add_tests[0]);

  struct px_route rt;
  memset(&rt, 0x55, sizeof(rt));
  px_route_init(&rt);

  for (unsigned i = 0; i < n_add_tests; ++i) {
    struct px_route_node* node = px_route_add(&rt, add_tests[i].path, add_tests[i].excl);
    if (!!node != add_tests[i].should_succeed) {
      px_log_error("FAILURE: route: %s: route added? %s; should it have been added?  %s",
                  add_tests[i].path ? add_tests[i].path : "(null)",
                  node ? "yes" : "no",
                  add_tests[i].should_succeed ? "yes" : "no");
    }
    if (node)
      node->priv = (void*)(uintptr_t)(i + 1);
  }

  struct route_find_test_data { char const* path; _Bool should_succeed; uintptr_t expected_data; };
  struct route_find_test_data find_tests[] = {
    { "/", true, 0x1 },
    { "/depth1", true, 0x2 },
    { "/depth1/depth2.2/depth3", true, 0x7 },
    { "/depth1/depth2.2/depth3/depth4/depth5", true, 0x7 }, // this will find depth3
  };
  unsigned const n_find_tests = sizeof(find_tests) / sizeof(find_tests[0]);

  for (unsigned i = 0; i < n_find_tests; ++i) {
    struct px_route_node* node = px_route_find(&rt, find_tests[i].path);
    if (!!node != find_tests[i].should_succeed) {
      px_log_error("FAILURE: route: %s: route found? %s; should it have been found?  %s",
                  find_tests[i].path ? find_tests[i].path : "(null)",
                  node ? "yes" : "no",
                  find_tests[i].should_succeed ? "yes" : "no");
      continue;
    }

    if (node && (uintptr_t)node->priv != find_tests[i].expected_data) {
      px_log_error("FAILURE: route: %s: route data %lu but expected %lu",
                  find_tests[i].path ? find_tests[i].path : "(null)",
                  (uintptr_t)node->priv,
                  find_tests[i].expected_data);
    }
  }

  px_log_info("printing route tree:");
  px_route_visit_tree(&rt.root, 0, print_node);
  px_log_info("removing root node:");
  px_route_remove(&rt, &rt.root);
  px_route_visit_tree(&rt.root, 0, print_node);
  px_log_info("clearing tree:");
  px_route_reset(&rt);
  return 0;
}
