#include <libpxd/px_route.h>
#include <fnmatch.h>
#include <string.h>

// this ensures all non-NULL subnodes are packed at the start of the node array
static void pack_route_node_pointers(struct px_route_node* node) {
  const unsigned n_route_subnodes = MAX_ROUTE_NODES;
  size_t copy_idx = 0;
  for (unsigned i = 0; i < n_route_subnodes; ++i) {
    if (node->subnodes[i] != NULL && i != copy_idx) {
      node->subnodes[copy_idx] = node->subnodes[i];
      node->subnodes[i] = NULL;
      ++copy_idx;
    }
  }
}

void px_route_node_init(struct px_route_node* node) {
  if (!node)
    return;
  *node = (struct px_route_node) { 0 };
}

void px_route_node_reset(struct px_route_node* node) {
  // NOTE: all subnodes of node will be malloc'd, but node itself may not be
  if (!node)
    return;
  const unsigned n_route_subnodes = MAX_ROUTE_NODES;
  for (unsigned i = 0; i < n_route_subnodes; ++i) {
    if (node->subnodes[i] != NULL) {
      px_route_node_reset(node->subnodes[i]);
      free(node->subnodes[i]);
      node->subnodes[i] = NULL;
    }
  }

  if (node->path)
    free(node->path);

  px_route_node_init(node);
}

void px_route_init(struct px_route* route) {
  if (!route)
    return;
  *route = (struct px_route) { 0 };
  px_route_node_init(&route->root);
}

void px_route_reset(struct px_route* route) {
  if (!route)
    return;
  px_route_node_reset(&route->root);
  px_route_init(route);
}

struct px_route_node* px_route_add(struct px_route* route, char const* route_str, _Bool excl) {
  if (!route_str) {
    return NULL;
  }

  struct px_path route_path = px_path_from_str(route_str);
  struct px_route_node* ret = px_route_add_path(route, &route_path, excl);
  px_path_reset(&route_path);
  return ret;
}

struct px_route_node* px_route_add_path(struct px_route* route, struct px_path const* route_path, _Bool excl) {
  if (!route || !route_path)
    return NULL;

  if (!px_path_is_abs(route_path)) {
    return NULL;
  }

  // path is only an absolute path marker
  if (route_path->components_sz == 1)
    return excl ? NULL : &route->root;

  return px_route_node_add_components(&route->root, (char const**)route_path->components + 1, route_path->components_sz - 1, excl);
}

struct px_route_node* px_route_node_add_components(struct px_route_node* node,
                                                    char const** components,
                                                    size_t components_sz,
                                                    _Bool excl)
{
  unsigned const n_route_subnodes = MAX_ROUTE_NODES;
  if (components_sz == 0)
    return NULL;

  char const* cur_route_component = components[0];
  if (!cur_route_component)
    return NULL;

  // sanity checks on the path component
  px_log_assert(cur_route_component != NULL, "null route component");
  if (cur_route_component == NULL)
    return NULL;

  // find a matching subnode for the path component
  struct px_route_node* matched = NULL;
  unsigned i;
  for (i = 0; i < n_route_subnodes && node->subnodes[i]; ++i) {
    if (node->subnodes[i]->path != NULL  // null path doesn't get checked (and shouldn't happen?)
        && strcmp(cur_route_component, node->subnodes[i]->path) == 0)
    {
      matched = node->subnodes[i];
      break;
    }
  }

  // matched?  cool, go to the next component with a new node if there are
  // any more components, or return the final one (or fail if excl)
  if (matched != NULL) {
    if (components_sz == 1)
      return excl ? NULL : matched; // if the last component already exists and we have excl set,
                                    // then treat it as an error/failure
    return px_route_node_add_components(matched, components + 1, components_sz - 1, excl);
  }

  // no matching subnodes, we need to make a new subtree rooted at node
  if (i >= n_route_subnodes) { // ran out of space
    px_log_error("too many subnodes for node %s at depth %u", node->path ? node->path : "(null)", node->depth);
    return NULL;
  }

  // make a new node and assign its path/depth
  struct px_route_node* new_node = (struct px_route_node*)malloc(sizeof(*new_node));
  if (!new_node) {
    px_log_error("no memory");
    return NULL;
  }
  px_route_node_init(new_node);
  new_node->parent = node;
  new_node->depth = node->depth + 1;
  new_node->path = strdup(cur_route_component);
  if (!new_node->path) {
    px_log_error("no memory");
    free(new_node);
    return NULL;
  }
  node->subnodes[i] = new_node;

  if (components_sz == 1)
    return new_node;

  // if there are any additional components that need to be added, use the
  // new_node as their parents and recurse this function
  struct px_route_node* ret = px_route_node_add_components(new_node, components + 1, components_sz - 1, excl);

  // if we couldn't add everything then undo our changes
  if (!ret) {
    node->subnodes[i] = NULL;
    free(new_node->path);
    free(new_node);
    return NULL;
  }
  return ret;
}

static _Bool remove_subnode(struct px_route_node* cur_node, struct px_route_node* to_remove) {
  for (unsigned i = 0; i < MAX_ROUTE_NODES && cur_node->subnodes[i] != NULL; ++i) {
    if (cur_node->subnodes[i] == to_remove) {
      px_route_node_reset(cur_node->subnodes[i]);
      free(cur_node->subnodes[i]);
      cur_node->subnodes[i] = NULL;
      pack_route_node_pointers(cur_node);
      return true;
    } else {
      if (remove_subnode(cur_node->subnodes[i], to_remove))
        return true;
    }
  }
  return false;
}

_Bool px_route_remove(struct px_route* route, struct px_route_node* to_remove) {
  if (!route || !to_remove)
    return false;

  if (to_remove == &route->root) {
    px_route_node_reset(to_remove);
    return true;
  }

  return remove_subnode(&route->root, to_remove);
}

_Bool px_route_node_matches_path_component(struct px_route_node const* node, char const* component) {
  if (!node || !node->path || !component)
    return false;

  // we shouldn't need this check because all paths that house the components
  // that this function receives *should* be cleaned, but just to make extra
  // double sure we don't allow a wildcard to let a .. pass for some reason
  if (strcmp(component, "..") == 0)
    return false;
  return fnmatch(node->path, component, FNM_NOESCAPE | FNM_PATHNAME) == 0;
}

struct px_route_node* px_route_find(struct px_route* route, char const *query) {
  if (!route || !query)
    return NULL;

  struct px_path path = px_path_from_str(query);
  struct px_route_node* ret = px_route_find_path(route, &path);
  px_path_reset(&path);
  return ret;
}

struct px_route_node* px_route_find_components(struct px_route_node* node,
                                               char const** components,
                                               size_t comp_sz)
{
  if (!components || comp_sz == 0)
    return NULL;

  // does the first component match the current node?  if no, return NULL
  if (!px_route_node_matches_path_component(node, components[0]))
    return NULL;

  struct px_route_node* sub_node = NULL;
  for (unsigned i = 0; i < MAX_ROUTE_NODES && node->subnodes[i] != NULL; ++i) {
    struct px_route_node* try_subnode = px_route_find_components(node->subnodes[i],
                                                                 (char const**)components + 1,
                                                                 comp_sz - 1);
    if (try_subnode && (!sub_node || sub_node->depth < try_subnode->depth))
      sub_node = try_subnode;
  }
  return sub_node ? sub_node : node;
}

struct px_route_node* px_route_find_path(struct px_route* route, struct px_path const* query) {
  if (!route || !query)
    return NULL;

  if (!px_path_is_abs(query))
    return NULL;

  struct px_route_node* sub_node = NULL;
  for (unsigned i = 0; i < MAX_ROUTE_NODES && route->root.subnodes[i] != NULL; ++i) {
    struct px_route_node* try_subnode = px_route_find_components(route->root.subnodes[i],
                                                                          (char const**)query->components + 1,
                                                                          query->components_sz - 1);
    if (try_subnode && (!sub_node || sub_node->depth < try_subnode->depth))
      sub_node = try_subnode;
  }

  return sub_node ? sub_node : &route->root;
}

void px_route_visit_tree(struct px_route_node* node, unsigned depth, px_route_visit_fxn fxn) {
  if (fxn)
    fxn(node, depth);
  for (unsigned i = 0; i < MAX_ROUTE_NODES && node->subnodes[i] != NULL; ++i) {
    px_route_visit_tree(node->subnodes[i], depth + 1, fxn);
  }
}
