#include <libpxd/px_path.h>
#include <libpxd/px_log.h>
#include <errno.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stddef.h>

#define PARANOID
#define COMPONENT_INCREMENT 8

#define ROUNDUP(x, r) ((((x) + (r) - 1)/(r))*(r))

_Bool px_path_is_valid(struct px_path const* path) {
  if (!path)
    return 0;

  // basically:
  return !!path->components_sz == (path->components != NULL);
}

static int count_path_components(char const* str) {
  if (!str || str[0] == '\0')
    return 0;

  int n_components = 1;
  for (char const* itr = str; *itr; ++itr) {
    if (itr[0] == '/') {
      while (itr[1] == '/')
        ++itr;
      if (itr[1] != '\0')
        ++n_components;
    } else {
      while (itr[1] != '/' && itr[1] != '\0')
        ++itr;
    }
  }
  return n_components;
}

struct px_path px_path_from_str(char const* str) {
  size_t n_components = 0;
  _Bool is_abs = false;

  // null or empty string gets nothing
  if (!str || *str == '\0')
    return (struct px_path) { 0 };

  if (*str == '/') {
    ++n_components;
    is_abs = true;
    while (*str == '/')
      ++str;
  }

  // determine the number of components we need.  each new component is
  // terminated by a slash or null.  other slashes are skipped
  // loop invariant: *comp_start != '/' is guaranteed, *comp_start == '\0' may be true
  char const* comp_start = str;
  while (*comp_start != '\0') {
    char const* comp_end = comp_start;
    while (*comp_end != '/' && *comp_end != '\0')
      ++comp_end;

    px_log_assert(comp_end != comp_start, "logic error?  comp_end should never match comp_start");
    ++n_components;

    while (*comp_end == '/')
      ++comp_end;
    comp_start = comp_end;
  }

  px_log_assert(n_components > 0, "logic error, n_components is not expected to ever be 0 here");

  // sanity check
  if (n_components > PX_PATH_MAX_COMPONENTS) {
    px_log_warn("exceeded max number of components");
    return (struct px_path) { 0 };
  }

  // allocate space for the components
  // from here down on failure we need to goto FAIL to free components
  char** components = (char**)calloc(n_components, sizeof(char*));
  if (!components) {
    px_log_error("no memory");
    return (struct px_path) { 0 };
  }

  // if it's an absolute path then add a / component to the start (we skipped
  // over the characters at the beginning of the function so we won't see them
  // in the loop below)
  size_t cur_idx = 0;
  if (is_abs) {
    components[cur_idx] = strdup("/");
    if (!components[cur_idx]) {
      px_log_error("no memory");
      goto FAIL;
    }
    ++cur_idx;
  }

  // now copy the components
  comp_start = str;
  while (*comp_start != '\0') {
    char const* comp_end = comp_start;
    // find the first non-delimiting character
    while (*comp_end != '/' && *comp_end != '\0')
      ++comp_end;

    // get the component length and make a copy of it
    size_t comp_len = comp_end - comp_start;
    char* new_component = strndup(comp_start, comp_len);
    if (!new_component || *new_component == '\0') { // second one shouldn't happen
      free(new_component);
      goto FAIL;
    }

    if (comp_len == 1 && strcmp(new_component, ".") == 0) {
      free(new_component); // skip 'current directory'
    } else if (comp_len == 2 && strcmp(new_component, "..") == 0) {
      // double-dot .. removes the previous entry (if it's not the leading absolute component)
      free(new_component);
      if (cur_idx > (is_abs ? 1 : 0)) {
        --cur_idx;
        free(components[cur_idx]);
        components[cur_idx] = NULL;
      }
    } else {
      components[cur_idx] = new_component;
      ++cur_idx;
    }

    while (*comp_end == '/')
      ++comp_end;
    comp_start = comp_end;
  }

  px_log_assert(cur_idx <= n_components, "memory overrun");
  if (cur_idx != n_components) { // shorten the allocation if needed
    if (cur_idx > 0) {
      char** new_components = (char**)realloc(components, sizeof(char*) * cur_idx);
      if (new_components) // make sure the realloc succeeded
        components = new_components;
    } else { // no components
      free(components);
      components = NULL;
    }
  }

  return (struct px_path) { .components = components, .components_sz = cur_idx };

FAIL:
  if (components) {
    for (size_t i = 0; i < n_components && components[i] != NULL; ++i)
      free(components[i]);
    free(components);
  }
  return (struct px_path) { 0 };
}

struct px_path px_path_from_str_clean(char const* str) {
  struct px_path from_str = px_path_from_str(str);
  struct px_path cleaned = px_path_clean(&from_str);
  px_path_reset(&from_str);
  return cleaned;
}

static int path_needs_split(struct px_path const* path) {
  if (!path || path->components_sz == 0)
    return 0;

  for (size_t i = 0, n = path->components_sz; i < n; ++i) {
    if (!path->components[i] || path->components[i][0] == '\0')
      continue;

    // if a nonempty path component does not have one subcomponent then it
    // means it's got an internal slash and could be split into two additional
    // components
    if (count_path_components(path->components[i]) != 1)
      return 1;
  }
  return 0;
}

// compresses a path into a minimal set of acceptable components
// note this doesn't check individual components for internal slashes - those
// shouldn't be there anyway
struct px_path px_path_clean(struct px_path const* path) {
  struct px_path cleaned = { 0 };

  if (!path || path->components_sz == 0)
    return cleaned;

  // we don't handle malformed components with internal slashes
  if (path_needs_split(path))
    return cleaned;

  size_t n_components = path->components_sz;

  size_t out_i = 0;
  char** new_components = (char**)calloc(n_components, sizeof(char*));

  for (size_t in_i = 0; in_i < n_components; ++in_i) {

    // null or empty component, skip
    if (!path->components[in_i] || path->components[in_i][0] == '\0')
      continue;

    // single dot, skip
    if (strncmp(path->components[in_i], "./", 2) == 0 || strcmp(path->components[in_i], ".") == 0)
      continue;

    // double dot == previous dir, remove the last entry if it's not the first / (i.e. absolute dir marker)
    if (strncmp(path->components[in_i], "../", 3) == 0 || strcmp(path->components[in_i], "..") == 0) {
      if (out_i > 0)
        --out_i;

      // keep a leading '/' (from an abs path)
      if (out_i == 0 && new_components[out_i] && new_components[out_i][0] == '/') {
        ++out_i;
        continue;
      }

      free(new_components[out_i]);
      new_components[out_i] = NULL;
      continue;
    }

    if (path->components[in_i][0] == '/') {
      // if it's the first index for both sets then copy the / to make an
      // absolute path, otherwise skip it
      if (in_i == 0 && out_i == 0) {
        new_components[out_i] = strdup("/");
        if (!new_components[out_i])
          goto FAIL;
        ++out_i;
      }
      continue;
    }

    // no more special cases

    // find the next / or the end of string
    char* itr = path->components[in_i];
    while (*itr && *itr != '/')
      ++itr;

    // if we landed on a (series of) slash, include it/the first one
    if (*itr == '/')
      ++itr;

    char* new_comp = NULL;
    if (in_i < n_components - 1) {
      // anything that isn't the last element is done in this block

      if (itr[-1] != '/') {
        // if the last character in the string is not a slash then add one
        size_t comp_len = strlen(path->components[in_i]);
        new_comp = calloc(1, comp_len + 2);
        if (!new_comp)
          goto FAIL;
        strcpy(new_comp, path->components[in_i]);
        new_comp[comp_len] = '/';
        new_comp[comp_len + 1] = '\0';
      } else {
        // slash is already present in the intermediate component, copy it directly
        new_comp = strndup(path->components[in_i], itr - path->components[in_i]);
      }
    } else {
      // if it's the last component, then copy it without any other checking
      new_comp = strndup(path->components[in_i], itr - path->components[in_i]);
    }

    if (!new_comp)
      goto FAIL;
    new_components[out_i] = new_comp;
    ++out_i;
  }

  cleaned.components_sz = out_i;
  cleaned.components = new_components;
  return cleaned;

FAIL:
  if (new_components) {
    for (size_t i = 0; i < out_i; ++i) {
      free(new_components[i]);
    }
    free(new_components);
  }
  cleaned = (struct px_path) { 0 };
  return cleaned;
}



void px_path_init(struct px_path* path) {
  memset(path, 0, sizeof(*path));
}

void px_path_reset(struct px_path* path) {
  if (!path)
    return;
  if (path->components) {
    for (size_t i = 0, n = path->components_sz; i < n; ++i) {
      free(path->components[i]);
      path->components[i] = NULL;
    }
    free(path->components);
  }
  px_path_init(path);
}

struct px_path px_path_copy(struct px_path const* src) {
  struct px_path dst = { 0 };
  if (!src)
    return dst;

  if (!src->components || src->components_sz == 0) {
    return dst;
  }

  char** new_components = (char**)calloc(src->components_sz, sizeof(char*));
  if (!new_components) // couldn't copy
    return dst;

  for (size_t i = 0, n = src->components_sz; i < n; ++i) {
    if (!src->components[i])
      continue;
    char* str_i = strdup(src->components[i]);
    if (!str_i) { // couldn't strdup, free everything else and bail
      for (size_t j = 0; j < i; ++j)
        free(new_components[j]);
      free(new_components);
      return dst;
    }
    new_components[i] = str_i;
  }

  dst.components = new_components;
  dst.components_sz = src->components_sz;
  return dst;
}

void px_path_move(struct px_path* dst, struct px_path* src) {
  *dst = *src;
  *src = (struct px_path) { 0 };
}

struct px_path px_path_strip_leading(struct px_path const* src, unsigned n) {
  if (!src || src->components_sz == 0)
    return (struct px_path) { .components = NULL, .components_sz = 0 };

  if (n == 0)
    return px_path_copy(src);

  // TODO we should check overflows here

  // we don't count the leading '/' as a component, so if src is an abs path
  // then we need to add 1 to n to compensate
  int is_abs = px_path_is_abs(src);
  if (is_abs)
    n++;

  if ((size_t)n > src->components_sz)
    n = src->components_sz;

  // we're abusing the px_path structure a bit by constructing this on the
  // stack, but it avoids a temporary copy since we're passing it directly to
  // px_path_copy which does not care where the components are allocated
  struct px_path dummy = { .components_sz = src->components_sz - n };
  dummy.components = dummy.components_sz ? src->components + n : NULL;

  struct px_path ret = px_path_copy(&dummy);

  if (is_abs)
    px_path_make_abs(&ret);

  return ret;
}

struct px_path px_path_concat(struct px_path const* lpath, struct px_path const* rpath) {
  if (!lpath)
    return px_path_copy(rpath);

  if (!rpath)
    return px_path_copy(lpath);

  // is r is absolute?  if so we need to do stuff to it
  _Bool r_is_abs = rpath->components_sz >= 1 && rpath->components[0][0] == '/';

  // we skip the first componet, which is the '/' component, in rpath (if it's absolute)
  size_t n_total_components = lpath->components_sz + rpath->components_sz - r_is_abs;
  if (n_total_components == 0)
    return (struct px_path) { 0 };

  char** new_components = (char**)calloc(n_total_components, sizeof(char*));
  if (!new_components) {
    px_log_error("no memory");
    return (struct px_path) { 0 };
  }

  size_t comp_idx = 0;
  char* const* cur_component = lpath->components;
  char* const* end_l_component = lpath->components + lpath->components_sz;
  while (cur_component != end_l_component) {
    new_components[comp_idx] = strdup(*cur_component);
    if (!new_components[comp_idx]) {
      px_log_error("no memory");
      goto FAIL;
    }
    ++cur_component;
    ++comp_idx;
  }

  cur_component = rpath->components + r_is_abs;
  char* const* end_r_component = rpath->components + rpath->components_sz;
  while (cur_component != end_r_component) {
    new_components[comp_idx] = strdup(*cur_component);
    if (!new_components[comp_idx]) {
      px_log_error("no memory");
      goto FAIL;
    }
    ++cur_component;
    ++comp_idx;
  }

  px_log_assert(comp_idx == n_total_components, "%lu != %lu", comp_idx, n_total_components);

  return (struct px_path) { .components = new_components, .components_sz = n_total_components };
FAIL:
  for (size_t i = 0; i < n_total_components && new_components[i] != NULL; ++i) {
    free(new_components[i]);
  }
  free(new_components);
  return (struct px_path) { 0 };
}

struct px_path px_path_concat_consume(struct px_path* pre, struct px_path* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(pre);
  px_path_reset(suf);
  return concat;
}

struct px_path px_path_concat_consume0(struct px_path* pre, struct px_path const* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(pre);
  return concat;
}

struct px_path px_path_concat_consume1(struct px_path const* pre, struct px_path* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  px_path_reset(suf);
  return concat;
}

struct px_path px_path_concat_clean(struct px_path const* pre, struct px_path const* suf) {
  struct px_path concat = px_path_concat(pre, suf);
  struct px_path cleaned = px_path_clean(&concat);
  px_path_reset(&concat);
  return cleaned;
}

char* px_path_to_str(struct px_path const* path) {
  if (!path)
    return NULL;

  if (path->components_sz == 0)
    return strdup("");

  unsigned is_abs = (path->components[0][0] == '/');

  char** components = path->components + is_abs; // skip the absolute component
  size_t n_components = path->components_sz - is_abs;

  size_t nchars = is_abs + 1; // absolute / and null byte
  if (n_components > 1)         // only need separators if there are more than one component
    nchars += n_components - 1; // then add one char for each separator

  for (unsigned i = 0; i < n_components; ++i) {
    size_t new_nchars = nchars + strlen(components[i]);
    if (new_nchars < nchars) { // overflow
      px_log_error("overflow");
      return NULL;
    }
    nchars = new_nchars;
  }

  char* path_str = (char*)malloc(nchars);
  if (!path_str) {
    px_log_error("no memory");
    return NULL;
  }

  char* str_out = path_str;
  if (is_abs) {
    *str_out = '/';
    ++str_out;
  }

  for (size_t comp_idx = 0; comp_idx < n_components; ++comp_idx) {
    char const* comp_itr = components[comp_idx];
    while (*comp_itr != '\0') {
      *str_out = *comp_itr;
      ++str_out;
      ++comp_itr;
    }

    // add a separator if needed
    if (comp_idx < n_components - 1) {
      *str_out = '/';
      ++str_out;
    }
  }
  *str_out = '\0';
  px_log_assert(str_out == path_str + nchars - 1, "logic error");
  return path_str;
}

char const* px_path_get_component(struct px_path const* path, int comp) {
  if (!path)
    return NULL;

  int idx = comp >= 0 ? comp : (int)path->components_sz + comp;
  if (idx >= 0 && idx < (int)path->components_sz)
    return path->components[idx];
  return NULL;
}

_Bool px_path_is_abs(struct px_path const* path) {
  if (path && path->components_sz > 0 && path->components[0][0] == '/')
    return true;
  return false;
}

_Bool px_path_make_abs(struct px_path* path) {
  if (!path)
    return false;
  if (path->components_sz > 0 && strcmp(path->components[0], "/") == 0) // already absolute
    return true;

  char* abs_comp = strdup("/");
  if (!abs_comp) // no memory
    return false;

  size_t new_comps_sz = path->components_sz + 1;
  char** new_comps = (char**)realloc(path->components, (new_comps_sz) * sizeof(char*));
  if (!new_comps) { // no memory
    free(abs_comp);
    return false;
  }
  path->components = new_comps;

  if (path->components_sz > 0) // move all of the current components forward one
    memmove(&path->components[1], &path->components[0], sizeof(char*) * path->components_sz);
  path->components[0] = abs_comp;
  path->components_sz = new_comps_sz;
  return true;
}

_Bool px_path_make_rel(struct px_path* path) {
  if (!path)
    return false;
  if (path->components_sz == 0 || path->components[0][0] != '/') // already relative
    return true;

  size_t new_comps_sz = path->components_sz - 1;
  if (new_comps_sz == 0) {
    free(path->components);
    path->components = NULL;
    path->components_sz = 0;
    return true;
  }

  free(path->components[0]); // free the leading /
  // shift everything forward one space
  memmove(&path->components[0], &path->components[1], new_comps_sz * sizeof(char*));
  path->components[new_comps_sz] = NULL; // fill in the end value with NULL to be safe
  char** new_comps = (char**)realloc(path->components, new_comps_sz * sizeof(char*));
  if (new_comps)
    path->components = new_comps;
  return true;
}
