#include <libpxd/px_event.h>
#include <libpxd/px_log.h>
#include <limits.h>
#include <stddef.h>
#include <string.h>

void px_event_init(struct px_event* ev) {
  *ev = (struct px_event) { .fd = -1 };
  px_queue_init(&ev->eventq);
}

void px_event_reset(struct px_event* ev) {
  if (!ev)
    return;
  px_event_dequeue(ev);
  px_event_init(ev);
}

void px_event_setup(struct px_event* ev,
                    int fd,
                    int events,
                    px_event_fxn_on_event event_fxn,
                    int timeout,
                    px_event_fxn_on_timeout timeout_fxn,
                    px_event_fxn_on_dequeue dequeue_fxn,
                    void* data)
{
  if (!ev)
    return;
  px_event_reset(ev);

  // timeout
  if (timeout < 0)
    px_event_disable_timeout(ev);
  else if (timeout == 0)
    px_event_set_immediate_timeout(ev);
  else
    px_event_set_timeout(ev, timeout);

  ev->fd = fd;
  ev->events = events;
  ev->on_event = event_fxn;
  ev->on_timeout = timeout_fxn;
  ev->on_dequeue = dequeue_fxn;
  ev->priv = data;
}

void px_event_enqueue(struct px_event* ev, struct px_queue* head) {
  px_queue_extract(&ev->eventq);
  px_queue_prepend(head, &ev->eventq);
}

void px_event_dequeue(struct px_event* ev) {
  px_queue_extract(&ev->eventq);
}

void px_event_trigger_event(struct px_event* ev, int revents) {
  if (ev->on_event)
    ev->on_event(ev->fd, revents, ev->priv);
}

void px_event_trigger_timeout(struct px_event* ev) {
  if (ev->on_timeout)
    ev->on_timeout(ev->priv);
}

void px_event_trigger_dequeue(struct px_event* ev) {
  if (ev->on_dequeue)
    ev->on_dequeue(ev->priv);
}

void px_event_set_timeout(struct px_event* ev, int ms) {
  ev->has_timeout = true;
  ev->timeout_end = px_time_now_plus_milliseconds(ms);
}

void px_event_set_immediate_timeout(struct px_event* ev) {
  ev->has_timeout = true;
  ev->timeout_end.tv_sec = (time_t)-1;
  ev->timeout_end.tv_nsec = LONG_MIN;
}

void px_event_disable_timeout(struct px_event* ev) {
  ev->has_timeout = false;
}

void px_event_clear_timeout(struct px_event* ev) {
  px_event_set_immediate_timeout(ev);
  px_event_disable_timeout(ev);
}

_Bool px_event_has_fd(struct px_event const* ev) {
  return ev->fd >= 0;
}

_Bool px_event_has_timeout(struct px_event const* ev) {
  return ev->has_timeout;
}

_Bool px_event_timeout_expired(struct px_event const* ev, struct timespec const* now) {
  if (!ev->has_timeout)
    return false;

  return ev->timeout_end.tv_sec < 0
          ||ev->timeout_end.tv_sec < now->tv_sec
          || (ev->timeout_end.tv_sec == now->tv_sec && ev->timeout_end.tv_nsec < now->tv_nsec);
}

void px_eventq_clear(struct px_queue* head) {
  while (!px_queue_is_singleton(head)) {
    struct px_event* ev = px_event_from_queue(head->next);
    px_event_dequeue(ev);
    px_event_trigger_dequeue(ev);
  }
}

void dbg_check_queue(struct px_queue* head) {
  (void)head;
}

