#include <polluxd.h>
#include <polluxd_cgi.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>

void print_open_fds(void) {
#ifdef __linux__
  char cmdbuf[64] = { 0 };
  snprintf(cmdbuf, sizeof(cmdbuf), "ls -lah /proc/%d/fd", (int)getpid());
  if (system(cmdbuf) < 0)
    (void)0;
#endif // __linux__
}

_Bool polluxd_recv_cgi_fd(struct polluxd_client* client, int rxsock, int* out_f) {

  *out_f = -1;

  const uint32_t check_value = 0x55aa55aa;
  uint32_t echo = check_value;
  int r = send(rxsock, &echo, sizeof(echo), MSG_NOSIGNAL);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_error("%s: cgi request socket closed", client->gemctx.conn.peer_addr_str);
    else
      px_log_error("%s: error requesting new cgi socket: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    return false;
  }

  union {
    struct cmsghdr cm;
    char control[CMSG_SPACE(sizeof(int))];
  } ctrl = { 0 };

  struct iovec msgiov = { .iov_base = &echo, .iov_len = sizeof(echo) };
  struct msghdr mhdr = {  .msg_name = NULL,
    .msg_namelen = 0,
    .msg_iov = &msgiov,
    .msg_iovlen = 1,
    .msg_control = ctrl.control,
    .msg_controllen = sizeof(ctrl.control) };

  r = recvmsg(rxsock, &mhdr, 0);
  if (r < 0) {
    int e = errno;
    px_log_error("%s: could not recvmsg: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    return false;
  } else if (r == 0) {
    px_log_error("%s: cgi helper socket closed", client->gemctx.conn.peer_addr_str);
    return false;
  }

  if (echo != check_value)
    px_log_warn("%s: check value error, expected %u but got %u",
                client->gemctx.conn.peer_addr_str, check_value, echo);

  // the only other echo right now is PX_WORKER_MSG_SENDF so we should by default expect a file
  struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
  if (cmsgp != NULL
      && cmsgp->cmsg_len == CMSG_LEN(sizeof(int))
      && cmsgp->cmsg_level == SOL_SOCKET
      && cmsgp->cmsg_type == SCM_RIGHTS)
  {
    *out_f = *((int*)CMSG_DATA(cmsgp));
  }

  return true;
}

static _Bool send_cgi_fd(int txsock, int fd) {
  // send a file descriptor using SCM_RIGHTS over a unix socket
  uint32_t echo = 0;
  int r = recv(txsock, &echo, sizeof(echo), 0);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_info("cgi helper %d: cgi request socket closed", (int)getpid());
    else
      px_log_info("cgi helper %d: error when waiting for socket request: %s", (int)getpid(), strerror(e));
    return false;
  }

  union {
    struct cmsghdr cm; // alignment
    char control[CMSG_SPACE(sizeof(int))];
  } ctrl = { 0 };

  // we have to transmit a small amount of 'real' data to send a file descriptor
  struct iovec msgiov = { .iov_base = &echo, .iov_len = sizeof(echo) };
  struct msghdr mhdr = {  .msg_name = NULL,
                          .msg_namelen = 0,
                          .msg_iov = &msgiov,
                          .msg_iovlen = 1,
                          .msg_control = ctrl.control,
                          .msg_controllen = sizeof(ctrl.control) };

  struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
  cmsgp->cmsg_len = CMSG_LEN(sizeof(int));
  cmsgp->cmsg_level = SOL_SOCKET;
  cmsgp->cmsg_type = SCM_RIGHTS;
  *((int*)CMSG_DATA(cmsgp)) = fd;

  r = sendmsg(txsock, &mhdr, MSG_NOSIGNAL);
  close(fd); // always close fd whether we send it or not
  if (r < 0) {
    int e = errno;
    px_log_warn("cgi helper %d: could not send cgi communication socket: %s", (int)getpid(), strerror(e));
    return false;
  }
  px_log_info("cgi helper %d: sent cgi file descriptor %d", (int)getpid(), fd);
  return true;
}

static _Bool is_allowed_cgi_env(char const* key) {
  static char const* const allowed_cgi_fields[] = {
    "GATEWAY_INTERFACE",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "GEMINI_URL",
    "PATH_INFO",
    "PATH_TRANSLATED",
    "SCRIPT_NAME",
    "QUERY_STRING",
    "HOSTNAME",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_PORT",
    "TLS_CIPHER",
    "TLS_VERSION",
    "AUTH_TYPE",
    "TLS_CLIENT_HASH",
    "TLS_CLIENT_PUBKEY_HASH" };

  for (unsigned i = 0, n = px_n_elements(allowed_cgi_fields); i < n; ++i) {
    if (strcmp(key, allowed_cgi_fields[i]) == 0)
      return true;
  }
  return false;
}

static _Bool unpack_envp(char* buf, size_t buf_sz, char** envp, size_t envp_sz) {

  size_t n_env = 0;
  px_log_assert(buf_sz > 0 && buf[buf_sz - 1] == '\0', "assumption violation");
  while (buf_sz > 0) {
    char* nul = memmem(buf, buf_sz, &(char) { '\0' }, sizeof(char));
    if (!nul) { // this should never happen
      memset(buf, 0, buf_sz);
      break;
    }

    do {
      char* eq = strchr(buf, '=');
      if (!eq) {
        // zero out the invalid data
        memset(buf, 0, nul - buf);
        break;
      }

      *eq = '\0';
      char* key = buf;
      if (!is_allowed_cgi_env(key)) {
        px_log_debug("cgi %d: invalid key %s", (int)getpid(), key);
        memset(buf, 0, nul - buf);
        break;
      //} else {
      //  px_log_debug("cgi %d: found key %s", (int)getpid(), key);
      }
      *eq = '=';

      if (n_env >= (envp_sz - 1)) {
        // out of space
        return false;
      }
      envp[n_env++] = buf;
    } while (0);

    buf_sz -= nul - buf;
    buf = nul;
    if (buf_sz > 0) {
      --buf_sz;
      ++buf;
    }
  }

  if (n_env < envp_sz) {
    envp[n_env] = NULL;
  } else {
    return false;
  }

  return true;
}

#define CGI_BUFFER_SZ 4096
static void cgi_worker(int cgi, char const* script) {
  px_log_info("cgi worker launched on pid %d", (int)getpid());

  char* buf = calloc(1, CGI_BUFFER_SZ + 1);
  if (!buf)
    return;

  int r = recv(cgi, buf, CGI_BUFFER_SZ, 0);
  if (r <= 0) {
    int e = errno;
    if (r == 0)
      px_log_error("cgi %d: cgi output socket closed prematurely", (int)getpid());
    else
      px_log_error("cgi %d: could not read cgi startup data from %d: %s",
                   (int)getpid(), cgi, strerror(e));
    print_open_fds();
    free(buf); // make valgrind happy
    exit(r == 0 ? 0 : -1);
  }
  if (buf[r-1] != '\0') {
    buf[r] = '\0'; // ensure we're always nul-terminated
    ++r;
  }

  char** envp = (char**)calloc(64, sizeof(char*));
  _Bool env_ok = unpack_envp(buf, (size_t)r, envp, 64);
  if (!env_ok) {
    px_log_info("cgi %d: could not set up cgi environment", (int)getpid());
    free(envp);
    close(cgi);
    exit(-1);
  }

  px_log_info("cgi %d: read %d bytes of cgi data", (int)getpid(), r);

  char* script_path = NULL;
  if (script) { // script was provided in the config file
    px_log_info("cgi helper %d: executing fixed script %s", (int)getpid(), script);
    script_path = strdup(script);
  } else { // polluxd translated the script name from a path
    px_log_info("cgi helper %d: translating script from path", (int)getpid());
    for (unsigned i = 0; i < 64 && envp[i]; ++i) {
      if (strncmp(envp[i], "PATH_TRANSLATED", sizeof("PATH_TRANSLATED") - 1) == 0) {
        char* eq = strchr(envp[i], '=');
        if (eq)
          script_path = strdup(eq + 1);
        break;
      }
    }
  }

  if (!script_path) {
    px_log_info("cgi %d: no PATH_TRANSLATED provided, cannot execute cgi", (int)getpid());
    close(cgi);
    free(envp);
    exit(-1);
  }

  if (dup2(cgi, STDOUT_FILENO) < 0) {
    int e = errno;
    px_log_fatal("cgi helper %d: could not dup2 cgi fd %d to stdout: %s",
                 (int)getpid(), cgi, strerror(e));
  }
  close(cgi); // don't need cgi anymore since it's now on stdout

  char* path[2] = { script_path, NULL };
  if (execve(script_path, path, envp) != 0) {
    int e = errno;
    px_log_error("cgi %d: could not execve %s: %s", (int)getpid(), script_path, strerror(e));
  }

  fprintf(stdout, "51 Not found\r\n");

  free(script_path);
  free(envp);
  free(buf);
  exit(0);
}

// the meat of the cgi helper process
static _Bool cgi_helper(int cgi_sock, char const* script) {

  struct pollfd pfd = { .fd = cgi_sock, .events = POLLOUT, .revents = 0 };
  int r = poll(&pfd, 1, -1);
  if (r < 0) {
    int e = errno;
    px_log_error("cgi helper %d: error on poll: %s", (int)getpid(), strerror(e));
    return false;
  }

  int socks[] = { -1, -1 };

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, socks) != 0) {
    int e = errno;
    px_log_error("cgi helper %d: could not pipe(): %s", (int)getpid(), strerror(e));
    return false;
  }

  if (!send_cgi_fd(cgi_sock, socks[0])) {
    close(socks[1]);
    return false;
  }
  socks[0] = -1;

  pid_t p = fork();
  if (p < 0) {
    int e = errno;
    px_log_error("cgi helper %d: could not fork: %s", (int)getpid(), strerror(e));
    close(socks[1]);
    return true;
  }

  if (p > 0) {
    close(socks[1]);
    return true;
  } else {
    cgi_worker(socks[1], script); // should not return
    close(socks[1]);
    exit(-1);
  }

  return true;
}

// @brief launch the cgi helper process.
// @details the helper process is what forks new processes and hands
// communcations channels back to the main polluxd process.  the helper
// receives a file descriptor from the main communication channel, then reads
// 1) a uint32_t 'x' from the received file and then 2) reads 'x' bytes of data
// from the file descriptor into a buffer.  if 'x' is too large then the new
// file descriptor is closed and no cgi is executed.  the helper then forks.
// the parent process then closes its copy of the new file descriptor, frees
// the buffered data, and waits for another message.  the child process parses
// the cgi data received during the original request.  if the cgi data is
// properly formatted the child then sets up the necessary environment
// variables from the received data, calls dup2 to move the received fd to
// stdout, then calls execve on the requested script.  at that point the cgi
// script will be executed and can write gemini data back to the stdout pipe
// (including the header) which will be read and forwarded in the main polluxd
// process
_Bool launch_cgi_helper(struct polluxd_context* pxd, struct polluxd_route_data* rtd) {
  int spair[2] = { -1, -1 };

  if (socketpair(AF_UNIX, SOCK_STREAM, 0, spair) != 0) {
    int e = errno;
    px_log_fatal("cgi helper: socketpair failed: %s", strerror(e));
    return false;
  }

  char const* cgi_user = rtd->cgi_user ? rtd->cgi_user : pxd->conf.drop_user;
  char const* cgi_group = rtd->cgi_group ? rtd->cgi_group : pxd->conf.drop_group;
  char const* cgi_chroot = rtd->cgi_chroot ? rtd->cgi_chroot : pxd->conf.chroot_dir;
  char const* cgi_script = rtd->cgi_script;
  //char const* cgi_docroot = rtd->cgi_docroot ? rtd->cgi_docroot : pxd->conf.docroot; // unused for now

  // to set up th
  pid_t p = fork();
  if (p < 0) {
    int e = errno;
    px_log_fatal("cgi helper: fork failed: %s", strerror(e));
    return false;
  }

  if (p > 0) {
    close(spair[1]);
    int cpid_status = 0;

    // wait for the first child process of the double-fork to exit
    pid_t cpid = waitpid(p, &cpid_status, 0);
    if (cpid != p
        || WEXITSTATUS(cpid_status) != 0
        || read(spair[0], &(char) { 0 }, sizeof(char)) != sizeof(char))
    {
      px_log_error("could not wait for child cgi helper process: %s", strerror(errno));
      close(spair[0]);
      return false;
    }
    px_log_assert(rtd->cgi_helper_sock == -1, "assumption violation");
    rtd->cgi_helper_sock = spair[0];
    return true;
  }

  // TODO add cgi chroot and docroot dirs
  polluxd_drop_privileges(cgi_user, cgi_group, cgi_chroot, "/", true);

  // child process - we need to break into a new session, re-fork (parent exits),
  // and then serve cgi stuff on the new child
  if (chdir("/") != 0)
    px_log_fatal("cgi helper %d: could not chdir: %s", (int)getpid(), strerror(errno));

  close(spair[0]);

  if (setsid() < 0)
    px_log_warn("cgi helper %d: could not setsid: %s", (int)getpid(), strerror(errno));

  // double fork so that the main process doesn't wait on us
  pid_t p1 = fork();
  if (p1 < 0)
    px_log_fatal("cgi helper first stage: could not re-fork: %s", strerror(errno));

  if (p1 > 0) // parent exits sucessfully
    exit(0);

  if (signal(SIGCHLD, SIG_IGN) != 0) {
    int e = errno;
    px_log_fatal("cgi helper %d: could not ignore SIGCHLD: %s", (int)getpid(), strerror(e));
  }

  // write one byte of data to the socket so the parent can continue
  char out = 0;
  if (write(spair[1], &out, sizeof(out)) != sizeof(out))
    px_log_fatal("cgi: could not write confirmatory data: %s", strerror(errno));
  px_log_info("cgi started on pid %d", (int)getpid());

  while (true) {
    if (!cgi_helper(spair[1], cgi_script))
      break;
  }

  exit(0);
}

_Bool polluxd_launch_cgi_helpers(struct polluxd_context* pxd) {
  for (unsigned i = 0, n = px_n_elements(pxd->route_data); i < n; ++i) {
    if (pxd->route_data[i].action == POLLUXD_DO_CGI) {
      if (!launch_cgi_helper(pxd, &pxd->route_data[i]))
        px_log_fatal("could not launch cgi helper process");
    }
  }
  return true;
}

//static struct px_worker_comm_msg recv_comm_msg(int fd) {
//
//  // receive a message consisting of one 4-byte opcod3
//  uint32_t  opcode = PX_WORKER_MSG_NOP;
//  int       msg_fd = -1;
//
//  union {
//    struct cmsghdr cm;
//    char control[CMSG_SPACE(sizeof(int))];
//  } ctrl = { 0 };
//
//  struct iovec msgiov = { .iov_base = &opcode, .iov_len = sizeof(opcode) };
//  struct msghdr mhdr = {  .msg_name = NULL,
//    .msg_namelen = 0,
//    .msg_iov = &msgiov,
//    .msg_iovlen = 1,
//    .msg_control = ctrl.control,
//    .msg_controllen = sizeof(ctrl.control) };
//
//  int r = recvmsg(fd, &mhdr, MSG_DONTWAIT);
//  if (r < 0) {
//    int e = errno;
//    px_log_error("could not recvmsg: %s", strerror(e));
//    int is_real_error = (e != EAGAIN && e != EWOULDBLOCK);
//    struct px_worker_comm_msg ret = { .opcode = is_real_error
//                                                ? PX_WORKER_MSG_ERROR
//                                                : PX_WORKER_MSG_NOP,
//                                      .fd = -1 };
//    return ret;
//  }
//
//  // the only other opcode right now is PX_WORKER_MSG_SENDF so we should by default expect a file
//  struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
//  if (cmsgp != NULL
//      && cmsgp->cmsg_len == CMSG_LEN(sizeof(int))
//      && cmsgp->cmsg_level == SOL_SOCKET
//      && cmsgp->cmsg_type == SCM_RIGHTS)
//  {
//    msg_fd = *((int*)CMSG_DATA(cmsgp));
//  }
//
//  return (struct px_worker_comm_msg) { .opcode = opcode, .fd = msg_fd };
//}
//
//px_op_status px_worker_send_msg(int commsock, int fd, uint32_t opcode) {
//  // send a file descriptor over the worker's comm socket
//  // use SCM_RIGHTS over a unix socket
//  if (opcode != PX_WORKER_MSG_SENDF && opcode != PX_WORKER_MSG_SHUTDOWN) {
//    px_log_error("invalid opcode %u, this is a programming error", opcode);
//    return AS_STATUS(PX_OP_ERROR);
//  }
//
//  if (commsock < 0) {
//    px_log_error("worker communication channel is closed");
//    return AS_STATUS(PX_OP_ERROR);
//  }
//
//  union {
//    struct cmsghdr cm; // alignment
//    char control[CMSG_SPACE(sizeof(int))];
//  } ctrl = { 0 };
//
//  struct iovec msgiov = { .iov_base = &opcode, .iov_len = sizeof(opcode) };
//  struct msghdr mhdr = {  .msg_name = NULL,
//    .msg_namelen = 0,
//    .msg_iov = &msgiov,
//    .msg_iovlen = 1,
//    .msg_control = NULL,
//    .msg_controllen = 0 };
//
//  if (opcode == PX_WORKER_MSG_SENDF) {
//    if (fd < 0) {
//      px_log_error("fd must be >= 0, is %d", fd);
//      return AS_STATUS(PX_OP_ERROR);
//    }
//
//    mhdr.msg_control = ctrl.control;
//    mhdr.msg_controllen = sizeof(ctrl.control);
//
//    struct cmsghdr *cmsgp = CMSG_FIRSTHDR(&mhdr);
//    cmsgp->cmsg_len = CMSG_LEN(sizeof(int));
//    cmsgp->cmsg_level = SOL_SOCKET;
//    cmsgp->cmsg_type = SCM_RIGHTS;
//    *((int*)CMSG_DATA(cmsgp)) = fd;
//  }
//
//  int r = sendmsg(commsock, &mhdr, MSG_DONTWAIT | MSG_NOSIGNAL);
//  if (r < 0) {
//    int e = errno;
//    if (e == EAGAIN || e == EWOULDBLOCK)
//      return AS_STATUS(PX_OP_RETRY);
//    px_log_error("could not sendmsg: %s", strerror(e));
//    return AS_STATUS(PX_OP_ERROR);
//  }
//  return AS_STATUS(PX_OP_DONE);
//}
//static void exec_cgi(struct px_gemini_context* gemctx,
//                     struct px_fileat const* fa)
//{
//
//  struct px_fileinfo* fi = (struct px_fileinfo*)gemctx->priv;
//  px_log_info("%s: serving CGI script %s on pid %d", gemctx->conn.peer_addr_str, fi->filename, getpid());
//
//  char peer_host[PX_ADDRSTR_LEN] = { 0 };
//  char local_port[8] = { 0 };
//  char peer_port[8] = { 0 };
//
//  _Bool r = px_net_addr_to_str(gemctx->conn.fd, NULL, 0, peer_host, sizeof(peer_host),
//                                local_port, sizeof(local_port), true);
//  if (!r) {
//    px_log_error("%s: overflow, buffers need adjustment", gemctx->conn.peer_addr_str);
//    return;
//  }
//
//  r = px_net_addr_to_str(gemctx->conn.fd, NULL, 0, NULL, 0, local_port, sizeof(local_port), false);
//  if (!r) {
//    px_log_error("%s: overflow, buffers need adjustment", gemctx->conn.peer_addr_str);
//    return;
//  }
//
//  const SSL_CIPHER *cipher = SSL_get_current_cipher(gemctx->conn.ssl);
//  struct kv {
//    char const* key;
//    char const* val;
//    int val_sz;
//  };
//
//  // we need strdup because execve requires a char* instead of a char const*
//  // and you should not just cast away constness
//  char* const argp[] = { strdup(fa->basename), NULL };
//
//  char const* scriptname = fa->basename;
//
//  int reqbuf_sz = gemctx->gemreq.reqbuf_sz < INT_MAX ? gemctx->gemreq.reqbuf_sz : INT_MAX;
//
//  struct kv cgi_env[] = {
//    { "GATEWAY_INTERFACE","CGI/1.1", -1 },
//    { "SERVER_PROTOCOL",  "GEMINI", -1 },
//    { "SERVER_SOFTWARE",  "libpxd", -1 },
//    { "PATH_INFO",        gemctx->gemreq.decoded_url.path, -1 },
//    { "GEMINI_URL",       gemctx->gemreq.reqbuf, reqbuf_sz },
//    { "SCRIPT_NAME",      scriptname, -1 },
//    { "HOSTNAME",         gemctx->gemreq.decoded_url.path, -1 },
//    { "SERVER_NAME",      gemctx->gemreq.decoded_url.path, -1 },
//    { "SERVER_ADDR",      gemctx->conn.host_addr_str, -1 },
//    { "SERVER_PORT",      local_port, -1 },
//    { "REMOTE_ADDR",      gemctx->conn.peer_addr_str, -1 },
//    { "REMOTE_HOST",      peer_host, -1 },
//    { "REMOTE_PORT",      peer_port, -1 },
//    { "TLS_CIPHER",       SSL_CIPHER_get_name(cipher), -1 },
//    { "TLS_VERSION",      SSL_CIPHER_get_version(cipher), -1 },
//    { "AUTH_TYPE",        gemctx->cert_digests.cert_digest ? "CERTIFICATE" : NULL, -1 },
//    { "TLS_CLIENT_HASH",  gemctx->cert_digests.cert_digest, -1 },
//    { "TLS_CLIENT_PUBKEY_HASH",  gemctx->cert_digests.pubkey_digest, -1 },
//  };
//
//  size_t nenv = sizeof(cgi_env) / sizeof(cgi_env[0]);
//  char** envp = (char**)calloc(nenv + 1, sizeof(char*));
//
//  unsigned end_env = 0;
//  if (!envp) {
//    goto ERR;
//  }
//
//  for (unsigned i = 0; i < nenv; ++i) {
//    struct kv* env = &cgi_env[i];
//    if (!env->key || !env->val)
//      continue;
//
//    int val_sz = env->val_sz;
//    if (val_sz < 0)
//      val_sz = strlen(env->val);
//
//    size_t total_len = strlen(env->key) + 1 /* = */ + val_sz + 1 /* NULL */;
//    int itotal_len = total_len < INT_MAX ? total_len : INT_MAX;
//    char* full_env = (char*)calloc(1, total_len);
//    int r = snprintf(full_env, itotal_len, "%s=%.*s", env->key, val_sz, env->val);
//    if (r < 0 || r >= itotal_len) {
//      if (r < 0)
//        px_log_error("%s: could not build cgi environment variable %s: %s",
//            gemctx->conn.peer_addr_str, env->key, strerror(errno));
//      else
//        px_log_error("%s: cgi environment variable %s too long: %s",
//            gemctx->conn.peer_addr_str, env->key, strerror(errno));
//      goto ERR;
//    }
//    envp[end_env] = full_env;
//    ++end_env;
//  }
//
//  (void)execve(fa->basename, argp, envp);
//  px_log_error("%s: could not execve %s: %s", gemctx->conn.peer_addr_str, fa->basename, strerror(errno));
//
//ERR:
//
//  free(argp[0]);
//  if (envp)
//    for (unsigned i = 0; i < end_env; ++i)
//      free(envp[i]);
//  free(envp);
//
//  // execve failed or wasn't done, so write out a server error to the file descriptor
//  px_gemini_set_header(&gemctx->gemreq, PX_PERMFAIL_NOT_FOUND, "File not found");
//  if (gemctx->gemreq.hdr_sz > 0) {
//    int nw = write(STDOUT_FILENO, gemctx->gemreq.hdr, gemctx->gemreq.hdr_sz);
//    if (nw < 0) {
//      int e = errno;
//      px_log_error("%s: could not write header line after execve failure: %s",
//                    gemctx->conn.peer_addr_str, strerror(e));
//    }
//  }
//  exit(-1);
//}
