#include <polluxd.h>
#include <polluxd_actions.h>
#include <polluxd_cgi.h>
#include <libpxd/px_log.h>
#include <libpxd/px_path.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

static char const* px_get_mimetype_by_extension(char const* filename) {
  if (!filename)
    return NULL;
  char const* dot = strrchr(filename, '.');
  if (dot != NULL) {
    struct mimetype_map_entry { char const* ext; char const* mimetype; };

    // these were taken from the Mozilla page about mimetypes
    // https://developer.mozilla.org/en-US/docs/Web/HTTP/MIME_types/Common_types
    static const struct mimetype_map_entry map[] = {
      { .ext = ".gmi", .mimetype = "text/gemini" },
      { .ext = ".aac", .mimetype = "audio/aac" },
      { .ext = ".abw", .mimetype = "application/x-abiword" },
      { .ext = ".apng", .mimetype = "image/apng" },
      { .ext = ".arc", .mimetype = "application/x-freearc" },
      { .ext = ".avif", .mimetype = "image/avif" },
      { .ext = ".avi", .mimetype = "video/x-msvideo" },
      { .ext = ".azw", .mimetype = "application/vnd.amazon.ebook" },
      { .ext = ".bin", .mimetype = "application/octet-stream" },
      { .ext = ".bmp", .mimetype = "image/bmp" },
      { .ext = ".bz", .mimetype = "application/x-bzip" },
      { .ext = ".bz2", .mimetype = "application/x-bzip2" },
      { .ext = ".cda", .mimetype = "application/x-cdf" },
      { .ext = ".csh", .mimetype = "application/x-csh" },
      { .ext = ".css", .mimetype = "text/css" },
      { .ext = ".csv", .mimetype = "text/csv" },
      { .ext = ".doc", .mimetype = "application/msword" },
      { .ext = ".docx", .mimetype = "application/vnd.openxmlformats-officedocument.wordprocessingml.document" },
      { .ext = ".eot", .mimetype = "application/vnd.ms-fontobject" },
      { .ext = ".epub", .mimetype = "application/epub+zip" },
      { .ext = ".gz", .mimetype = "application/x-gzip" },
      { .ext = ".gif", .mimetype = "image/gif" },
      { .ext = ".html", .mimetype = "text/html" },
      { .ext = ".ico", .mimetype = "image/vnd.microsoft.icon" },
      { .ext = ".ics", .mimetype = "text/calendar" },
      { .ext = ".jar", .mimetype = "application/java-archive" },
      { .ext = ".jpeg", .mimetype = "image/jpeg" },
      { .ext = ".jpg", .mimetype = "image/jpeg" },
      { .ext = ".js", .mimetype = "text/javascript" },
      { .ext = ".json", .mimetype = "application/json" },
      { .ext = ".jsonld", .mimetype = "application/ld+json" },
      { .ext = ".midi", .mimetype = "audio/x-midi" },
      { .ext = ".mjs", .mimetype = "text/javascript" },
      { .ext = ".mp3", .mimetype = "audio/mpeg" },
      { .ext = ".mp4", .mimetype = "video/mp4" },
      { .ext = ".mpeg", .mimetype = "video/mpeg" },
      { .ext = ".mpkg", .mimetype = "application/vnd.apple.installer+xml" },
      { .ext = ".odp", .mimetype = "application/vnd.oasis.opendocument.presentation" },
      { .ext = ".ods", .mimetype = "application/vnd.oasis.opendocument.spreadsheet" },
      { .ext = ".odt", .mimetype = "application/vnd.oasis.opendocument.text" },
      { .ext = ".oga", .mimetype = "audio/ogg" },
      { .ext = ".ogv", .mimetype = "video/ogg" },
      { .ext = ".ogx", .mimetype = "application/ogg" },
      { .ext = ".opus", .mimetype = "audio/ogg" },
      { .ext = ".otf", .mimetype = "font/otf" },
      { .ext = ".png", .mimetype = "image/png" },
      { .ext = ".pdf", .mimetype = "application/pdf" },
      { .ext = ".php", .mimetype = "application/x-httpd-php" },
      { .ext = ".ppt", .mimetype = "application/vnd.ms-powerpoint" },
      { .ext = ".pptx", .mimetype = "application/vnd.openxmlformats-officedocument.presentationml.presentation" },
      { .ext = ".rar", .mimetype = "application/vnd.rar" },
      { .ext = ".rtf", .mimetype = "application/rtf" },
      { .ext = ".sh", .mimetype = "application/x-sh" },
      { .ext = ".svg", .mimetype = "image/svg+xml" },
      { .ext = ".tar", .mimetype = "application/x-tar" },
      { .ext = ".tiff", .mimetype = "image/tiff" },
      { .ext = ".ts", .mimetype = "video/mp2t" },
      { .ext = ".ttf", .mimetype = "font/ttf" },
      { .ext = ".txt", .mimetype = "text/plain" },
      { .ext = ".vsd", .mimetype = "application/vnd.visio" },
      { .ext = ".wav", .mimetype = "audio/wav" },
      { .ext = ".weba", .mimetype = "audio/webm" },
      { .ext = ".webm", .mimetype = "video/webm" },
      { .ext = ".webp", .mimetype = "image/webp" },
      { .ext = ".woff", .mimetype = "font/woff" },
      { .ext = ".woff2", .mimetype = "font/woff2" },
      { .ext = ".xhtml", .mimetype = "application/xhtml+xml" },
      { .ext = ".xls", .mimetype = "application/vnd.ms-excel" },
      { .ext = ".xlsx", .mimetype = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet" },
      { .ext = ".xml", .mimetype = "application/xml" },
      { .ext = ".xul", .mimetype = "application/vnd.mozilla.xul+xml" },
      { .ext = ".zip", .mimetype = "application/x-zip-compressed" },
      { .ext = ".3gp", .mimetype = "audio/3gpp" },
      { .ext = ".3g2", .mimetype = "audio/3gpp2" },
      { .ext = ".7z", .mimetype = "application/x-7z-compressed" },
    };

    for (unsigned i = 0; i < px_n_elements(map); ++i) {
      if (strcmp(dot, map[i].ext) == 0)
        return map[i].mimetype;
    }
  }
  return "application/octet-stream";
}

char const* px_get_mimetype(char const* filename) {
  char const* mimetype = NULL;
  mimetype = px_get_mimetype_by_extension(filename);
  if (!mimetype)
    mimetype = "application/octet-stream";
  return mimetype;
}

int escape_nongraph(unsigned char c) {
  return !isgraph(c);
}

_Bool polluxd_deny_cb(struct polluxd_client* client) {
  //char* path_str = px_path_to_str(&client->routed_path);
  char* encoded_path = client->routed_path_str
                       ? px_url_encode_str(client->routed_path_str, escape_nongraph)
                       : NULL;
  char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);

  px_log_info("%s: denying access to %s -> %s",
              client->gemctx.conn.peer_addr_str,
              encoded_url_path ? encoded_url_path : "(no url)",
              encoded_path ? encoded_path : "(no path)");
  free(encoded_url_path);
  free(encoded_path);

  struct px_conn_buffer *cb = px_conn_buffer_get();
  if (!cb)
    return false;

  static char const header[] = "50 Permission denied\r\n";
  const size_t n_cp = sizeof(header) - 1;
  memcpy(cb->data, header, n_cp);
  cb->data_sz = n_cp;

  (void)px_gemini_context_queue_data(&client->gemctx, cb);

  return false;
}

struct px_fileinfo open_file(char const* pathname,
                             char const* index_file,
                             _Bool autoindex,
                             _Bool use_sockets)

{
  int fd = open(pathname, O_RDONLY | O_CLOEXEC | O_NONBLOCK);
  if (fd < 0) {
    if (!use_sockets)
      return (struct px_fileinfo) { 0 };

    // try to open up a socket
    struct sockaddr_un un = { 0 };
    size_t pnlen = pathname ? strlen(pathname) : 0;
    if (pnlen > sizeof(un.sun_path))
      return (struct px_fileinfo) { 0 };
    un.sun_family = AF_UNIX;
    memcpy(un.sun_path, pathname, pnlen);

    fd = socket(AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC | SOCK_NONBLOCK, 0);
    if (fd < 0)
      return (struct px_fileinfo) { 0 };

    // block waiting for an accept
    // this will succeed whether or not a server accepts the connection
    if (connect(fd, (struct sockaddr const*)&un, sizeof(un)) < 0) {
      close(fd);
      return (struct px_fileinfo) { 0 };
    }
  }

  struct stat s;
  if (fstat(fd, &s) < 0) {
    close(fd);
    return (struct px_fileinfo) { 0 };
  }

  switch (s.st_mode & S_IFMT) {
  case S_IFIFO:
    // fallthrough
  case S_IFSOCK :
    // fallthrough
  case S_IFREG:
  {
    FILE* f = fdopen(fd, "r");
    if (!f) {
      close(fd);
      return (struct px_fileinfo) { 0 };
    }

    char* path = strdup(pathname);
    if (!path) {
      fclose(f);
      return (struct px_fileinfo) { 0 };
    }

    _Bool reg_file = (s.st_mode & S_IFMT) == S_IFREG;
    return (struct px_fileinfo) { .file = f, .pathname = path, .regular_file = reg_file };
  }
  case S_IFDIR:
    if (!index_file && !autoindex) {
      close(fd);
      return (struct px_fileinfo) { 0 };
    }
    break;
  default :
    close(fd);
    return (struct px_fileinfo) { 0 };
  }

  // if we have an index file then use our current fd to open it
  if (index_file) {
    int ffd = openat(fd, index_file, O_RDONLY | O_CLOEXEC | O_NONBLOCK);

    if (ffd >= 0) {
      close(fd);

      FILE* filp = fdopen(ffd, "r");
      if (!filp) {
        close(ffd);
        return (struct px_fileinfo) { 0 };
      }

      char* index_path = NULL;
      int r = asprintf(&index_path, "%s/%s", pathname, index_file);
      if (r < 0 || !index_path) {
        free(index_path);
        fclose(filp);
        return (struct px_fileinfo) { 0 };
      }

      struct stat s;
      _Bool reg_file = ((fstat(ffd, &s) == 0) && (s.st_mode & S_IFMT) == S_IFREG);
      return (struct px_fileinfo) { .file = filp, .pathname = index_path, .regular_file = reg_file };
    }
  }

  if (!autoindex)
    return (struct px_fileinfo) { 0 };

  DIR* d = fdopendir(fd);
  if (!d) {
    close(fd);
    return (struct px_fileinfo) { 0 };
  }

  char* dpath = strdup(pathname);
  if (!dpath) {
    closedir(d);
    return (struct px_fileinfo) { 0 };
  }

  return (struct px_fileinfo) { .dir = d, .pathname = dpath };
}

static _Bool send_file_header(struct polluxd_client* client, struct px_conn_buffer* cb) {
  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;

  // path_str is used to open the file/directory, must succeed or the connection dies
  char* path_str = px_path_to_str(&client->routed_path);
  {
    if (!path_str)
      return false;

    char* encoded_path = px_url_encode_str(path_str, escape_nongraph);
    char* encoded_url_path = px_url_encode_str(client->gemctx.request.url.path, escape_nongraph);
    px_log_info("%s: requesting path %s -> %s",
        client->gemctx.conn.peer_addr_str,
        encoded_url_path ? encoded_url_path : "",
        encoded_path ? encoded_path : "");
    free(encoded_path);
    free(encoded_url_path);

    char const* index_file = (client->route && client->route->index_file) ? client->route->index_file : NULL;
    _Bool autoindex = client->route ? client->route->autoindex : false;
    client->fileinfo = open_file(path_str, index_file, autoindex, true);
  }

  free(path_str);

  // construct and send the gemini header
  if (client->fileinfo.file) {
    px_log_info("%s: serving file %s",
        client->gemctx.conn.peer_addr_str,
        client->fileinfo.pathname ? client->fileinfo.pathname : "(null)");
    char const* mimetype = px_get_mimetype(client->fileinfo.regular_file ? client->fileinfo.pathname : NULL);
    int r = snprintf((char*)cb->data, cb_max_sz, "20%s%s\r\n", mimetype ? " " : "", mimetype);
    if (r > 0)
      cb->data_sz = (size_t)r < cb_max_sz ? (size_t)r : cb_max_sz;
    return (r >= 0);
  } else if (client->fileinfo.dir) {
    px_log_info("%s: autoindexing directory %s",
        client->gemctx.conn.peer_addr_str,
        client->fileinfo.pathname ? client->fileinfo.pathname : "(null)");
    // if we have a dir opened then we are autoindexing
    int r = snprintf((char*)cb->data, cb_max_sz, "20 text/gemini\r\n# Directory index\n");
    if (r > 0)
      cb->data_sz = (size_t)r < cb_max_sz ? (size_t)r : cb_max_sz;
    return (r >= 0);
  }

  px_log_info("%s: not found", client->gemctx.conn.peer_addr_str);
  int r = snprintf((char*)cb->data, cb_max_sz, "51 Not found\r\n");
  if (r > 0)
    cb->data_sz = (size_t)r < cb_max_sz ? (size_t)r : cb_max_sz;
  return false;
}

static _Bool list_dir(struct polluxd_client* client, struct px_conn_buffer* cb) {
  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;
  size_t added_sz = 0;
  while (true) {
    // save the current position in case we run out of space
    long dpos = telldir(client->fileinfo.dir);
    if (dpos < 0) // error? assume we're done
      return false;

    errno = 0;
    struct dirent* de = readdir(client->fileinfo.dir);
    if (!de) {
      int e = errno;
      if (e == 0) // end of directory, done
        return false;

      if (e == EAGAIN || e == EWOULDBLOCK) { // blocking on dir info.  probably won't happen?
        client->file_ev.fd = dirfd(client->fileinfo.dir);
        client->file_ev.events = PX_EVENT_READ;
        client->file_ev.has_timeout = false;
        client->file_ev.priv = client;
        px_queue_extract(&client->file_ev.eventq);
        px_queue_append(&client->pxd->workq.events_head, &client->file_ev.eventq);
        return true;
      }
      return false;
    }

    char* d_name_encoded = px_url_encode_str(de->d_name, escape_nongraph);
    int r = snprintf((char*)cb->data + added_sz, cb_max_sz - added_sz,
        "=> ./%s%s %s\n", d_name_encoded ? d_name_encoded : "?", de->d_type == DT_DIR ? "/" : "", de->d_name);
    free(d_name_encoded);

    if (r < 0) {
      return false;
    } else if ((size_t)r > (cb_max_sz - added_sz)) {
      // if we didn't even add one directory then we got a buffer that's too
      // small.  assume that's always going to happen and break out of adding
      // directories
      if (added_sz == 0)
        return false;

      // reset the dir position to before the current entry so it can be read
      // on the next iteration
      seekdir(client->fileinfo.dir, dpos);
      return true;
    }

    added_sz += r;
    cb->data_sz = added_sz;
  }
  return false;
}

static _Bool send_file(struct polluxd_client* client, struct px_conn_buffer* cb) {
  const size_t cb_max_sz = PX_TLS_MAX_PLAINTEXT_SZ;

  errno = 0;
  size_t r = fread(cb->data, 1, cb_max_sz, client->fileinfo.file);
  if (r == 0) {
    int e = errno;

    // end of file?  we're done
    if (feof(client->fileinfo.file)) {
      px_log_info("%s: done transmitting file", client->gemctx.conn.peer_addr_str);
      return false;
    }

    // error?  if we just don't have data available then it's ok
    if (e == EAGAIN || e == EWOULDBLOCK) {
      clearerr(client->fileinfo.file);
      client->file_ev.fd = fileno(client->fileinfo.file);
      client->file_ev.events = PX_EVENT_READ;
      client->file_ev.has_timeout = false;
      client->file_ev.priv = client;
      px_queue_extract(&client->file_ev.eventq);
      px_queue_append(&client->pxd->workq.events_head, &client->file_ev.eventq);
      return true;
    }
    return false;
  }

  // successful read
  cb->data_sz = (size_t)r < cb_max_sz ? r : cb_max_sz;
  return true;
}

_Bool polluxd_file_cb(struct polluxd_client* client) {

  struct px_conn_buffer* cb = px_conn_buffer_get();
  if (!cb) // no memory
    return false;

  _Bool ret = false;
  if (!client->fileinfo.file && !client->fileinfo.dir) { // need to open a file/dir
    // if no file or dir has been added to the fileinfo then we need to open
    // one and send a header
    ret = send_file_header(client, cb);
  } else if (client->fileinfo.file) {
    ret = send_file(client, cb);
  } else if (client->fileinfo.dir) {
    ret = list_dir(client, cb);
  }

  if (cb->data_sz > 0) {
    if (!px_gemini_context_queue_data(&client->gemctx, cb))
      ret = false;
  } else {
    px_conn_buffer_free(cb);
  }

  return ret;
}

#define CGI_BUFFER_SZ 4096
// @brief pack the CGI environment into a buffer and send it to the cgi process
static _Bool send_cgi_environment(struct polluxd_client* client, int cgi_sock) {

  // simple protocol:
  // make a 4k buffer, write key=value\0 strings into it (which will be used as
  // environment vars), and send it over cgi_sock as one big block
  char* buf = (char*)calloc(1, CGI_BUFFER_SZ);

  if (!buf)
    return false;

  char* out = buf;
  size_t out_sz = CGI_BUFFER_SZ;
  size_t buf_sz = 0;

  struct kv { char const* key; char const* val; };

  char* url_str = px_url_to_str(&client->gemctx.request.url);
  const SSL_CIPHER *cipher = SSL_get_current_cipher(client->gemctx.conn.ssl);
  struct kv env[] = {
    { "SERVER_PROTOCOL", "gemini" },
    { "SERVER_SOFTWARE", "libpxd/polluxd" },
    { "GEMINI_URL", url_str },
    { "PATH_TRANSLATED", client->routed_path_str },
    { "SCRIPT_NAME", client->gemctx.request.url.path },
    { "QUERY_STRING", client->gemctx.request.url.query },
    { "HOSTNAME", client->gemctx.request.url.host },
    { "SERVER_NAME", client->gemctx.sni_hostname },
    { "SERVER_ADDR", client->gemctx.conn.host_addr_str },
    { "REMOTE_ADDR", client->gemctx.conn.peer_addr_str },
    { "TLS_CIPHER", SSL_CIPHER_get_name(cipher) },
    { "TLS_VERSION", SSL_CIPHER_get_version(cipher) },
    { "AUTH_TYPE", client->gemctx.client_cert.cert ? "CERTIFICATE" : "NONE" },
    { "TLS_CLIENT_HASH", client->gemctx.client_cert.digests.cert_digest },
    { "TLS_CLIENT_PUBKEY_HASH", client->gemctx.client_cert.digests.pubkey_digest }
  };

  for (unsigned i = 0, n = px_n_elements(env); i < n; ++i) {
    if (out_sz == 0)
      goto ERR;

    if (!env[i].val)
      continue;

    int r = snprintf(out, out_sz, "%s=%s", env[i].key, env[i].val ? env[i].val : "");
    if (r < 0 || (size_t)r >= out_sz)
      goto ERR;
    out += r;
    out_sz -= r;
    buf_sz += r + 1;
    if (out_sz > 0) {
      --out_sz;
      ++out;
    }
  }

  ssize_t tx = send(cgi_sock, buf, buf_sz, MSG_NOSIGNAL);
  if (tx < 0 || (size_t)tx != buf_sz) {
    int e = errno;
    px_log_error("%s: could not send cgi environment: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    goto ERR;
  }

  free(url_str);
  free(buf);
  return true;
ERR:
  free(url_str);
  free(buf);
  return false;
}

_Bool polluxd_cgi_cb(struct polluxd_client* client) {

  int cgi_sock = -1;
  if (!polluxd_recv_cgi_fd(client, client->route->cgi_helper_sock, &cgi_sock) || cgi_sock < 0) {
    px_log_warn("%s: could not get cgi communication pipe", client->gemctx.conn.peer_addr_str);
    return false;
  }

  px_log_info("%s: opened cgi on fd %d, requesting %s",
              client->gemctx.conn.peer_addr_str,
              cgi_sock,client->routed_path_str
              ? client->routed_path_str
              : "(none)");

  if (!send_cgi_environment(client, cgi_sock)) {
    close(cgi_sock);
    return false;
  }

  FILE* cgif = fdopen(cgi_sock, "r");
  if (!cgif) {
    int e = errno;
    px_log_warn("%s: could not fdopen: %s", client->gemctx.conn.peer_addr_str, strerror(e));
    close(cgi_sock);
    return false;
  }
  client->fileinfo.file = cgif;
  client->data_cb = polluxd_file_cb;
  px_log_info("%s: launching cgi script, reading from fd %d", client->gemctx.conn.peer_addr_str, fileno(cgif));
  return true;
}
