#ifndef PX_URI_H__
#define PX_URI_H__

#include <stdbool.h>
#include <stdint.h>

struct px_url {
  char* scheme;
  char* userinfo;
  char* host;
  char* port;
  char* path;
  char* query;
  char* fragment;
};

// @brief initialie a url structure
void px_url_init(struct px_url* url);

// @brief free a url's resource and reinitialize it
void px_url_reset(struct px_url* url);

// @brief creates a px_url with decoded components from a buffer (i.e. parses the url)
// @param url the url to output to
_Bool px_url_from_bytes(struct px_url* url, uint8_t const* buf, unsigned buflen);

// @brief encode a url structure with proper semantics for each of the url fields
// @param out the url to output encoded data to
// @param url the url to encode
// @return true if conversion completed successfully, otherwise false
_Bool px_url_encode(struct px_url* out, struct px_url const* url);

// @brief percent-encode a string for use in a url
// @param str the string to percent-encode
// @param encode_char function run on each character of str.  should return:
//        < 0 character should be ignored (not copied or encoded)
//          0 character does not need to be encoded, copy it
//        > 0 character needs to be percent-encoded
// @return a pointer to a malloc-allocated string or NULL if encoding failed
char* px_url_encode_str(char const* str, int (* const encode_char)(unsigned char));

// @brief decode a percent-encoded string into raw form
// @param str the string to decode
// @return a malloc-allocated string
char* px_url_decode_str(char const* str);

// @brief decode a percent-encoded url to its raw form
// @param decoded the url to output values to.  must be initialized
// @param url the url to decode
_Bool px_url_decode(struct px_url* decoded, struct px_url const* url);

// @brief converts a url to a string.  necessary characters are percent-encoded
// @param url the url to convert
// @return a malloc-allocated string
char* px_url_to_str(struct px_url const* url);

// @brief like px_url_to_string but percent-encoding is not done.  use px_url_to_str instead
// @param url the url to convert
// @return a malloc-allocated string
char* px_url_to_str_unencoded(struct px_url const* url);

#endif // PX_URI_H__
